// chatWithClawdbot.js (OpenClaw AI integration - filename kept for backwards compatibility)
import { execFile } from 'child_process';
import { readFileSync } from 'fs';
import { dirname, join } from 'path';
import { fileURLToPath } from 'url';
import dotenv from 'dotenv';
dotenv.config();

const __dirname = dirname(fileURLToPath(import.meta.url));
const OPENCLAW_BIN = process.env.OPENCLAW_BIN || 'openclaw';
const OPENCLAW_TIMEOUT = parseInt(process.env.OPENCLAW_TIMEOUT || '120') * 1000;

const NOTIFY_CHANNEL = process.env.OPENCLAW_NOTIFY_CHANNEL || 'whatsapp';
const NOTIFY_TARGET = process.env.OPENCLAW_NOTIFY_TARGET || '';

/**
 * Send a message to the OpenClaw owner (you).
 * @param {string} message - Message to send
 * @returns {Promise<boolean>}
 */
export function notifyUser(message) {
  if (!NOTIFY_TARGET) {
    console.warn('⚠️ OPENCLAW_NOTIFY_TARGET not set in .env. Skipping notification.');
    return Promise.resolve(false);
  }
  return new Promise((resolve) => {
    const args = ['message', 'send', '--channel', NOTIFY_CHANNEL, '--target', NOTIFY_TARGET, '--message', message];
    execFile(OPENCLAW_BIN, args, { timeout: 30000 }, (err) => {
      if (err) {
        console.error('❌ Failed to notify:', err.message);
        resolve(false);
      } else {
        console.log('✅ Notification sent');
        resolve(true);
      }
    });
  });
}

/**
 * Run an openclaw agent task and return the response text.
 * @param {string} message - The prompt/message
 * @param {number} timeout - Timeout in ms
 * @returns {Promise<string>}
 */
function runOpenclaw(message, timeout = OPENCLAW_TIMEOUT) {
  return new Promise((resolve, reject) => {
    const args = ['agent', '--local', '--json', '--session-id', 'rtms-meeting-assistant', '--message', message];
    execFile(OPENCLAW_BIN, args, { timeout, maxBuffer: 10 * 1024 * 1024 }, (err, stdout, stderr) => {
      if (err) {
        console.error('❌ OpenClaw error:', err.message);
        if (stderr) console.error('stderr:', stderr);
        return reject(err);
      }
      try {
        const result = JSON.parse(stdout);
        // openclaw agent --json returns { payloads: [{ text }], meta: { ... } }
        if (result.payloads && Array.isArray(result.payloads)) {
          const text = result.payloads.map(p => p.text).filter(Boolean).join('\n');
          resolve(text || stdout.trim());
        } else {
          resolve(result.reply || result.message || result.content || stdout.trim());
        }
      } catch {
        // Not JSON, return raw output
        resolve(stdout.trim());
      }
    });
  });
}

/**
 * Sends a message to OpenClaw agent for processing
 * @param {string} message - The user message
 * @param {string} _model - Ignored (uses OpenClaw's configured model)
 * @param {string[]} images - Base64 images (passed as context in prompt)
 * @param {boolean} isRetry - Whether this is a retry
 * @returns {Promise<string>}
 */
export async function chatWithClawdbot(message, _model = '', images = [], isRetry = false) {
  try {
    let fullMessage = message;

    // If images are provided, mention them in the prompt
    // (OpenClaw agent can process text but images need to be described)
    if (images.length > 0) {
      fullMessage += `\n\n[Note: ${images.length} image(s) were captured from screen share but cannot be passed directly. Please analyze based on the text/transcript content provided.]`;
    }

    console.log('🤖 Sending to OpenClaw agent...');
    const response = await runOpenclaw(fullMessage);
    return response;
  } catch (err) {
    console.error('❌ Error with OpenClaw:', err.message);

    if (!isRetry) {
      console.log('🔄 Retrying OpenClaw...');
      return await chatWithClawdbot(message, _model, images, true);
    }

    throw err;
  }
}

/**
 * Fast/lightweight chat (same as regular for OpenClaw — no separate fast model)
 * @param {string} message
 * @returns {Promise<string>}
 */
export async function chatWithClawdbotFast(message) {
  try {
    console.log('🤖 Sending to OpenClaw agent (fast)...');
    return await runOpenclaw(message);
  } catch (err) {
    console.error('❌ Error with OpenClaw:', err.message);
    throw err;
  }
}

/**
 * Generate strategic dialog suggestions for meeting facilitation
 * @param {string} transcript - Full meeting transcript text
 * @returns {Promise<string[]>} Array of 4 RPG-style dialog suggestions
 */
export async function generateDialogSuggestions(transcript) {
  try {
     const dialogPromptTemplate = readFileSync(join(__dirname, 'query_prompt_dialog_suggestions.md'), 'utf-8');
    const filledPrompt = dialogPromptTemplate.replace(/\{\{meeting_transcript\}\}/g, transcript);

    console.log('🗣️ Generating dialog suggestions via OpenClaw...');
    const response = await runOpenclaw(filledPrompt);

    const suggestions = response
      .split('\n')
      .map(line => line.trim())
      .filter(line => line.length > 0 && !line.startsWith('Response:') && !line.startsWith('Only return'));

    console.log(`✅ Generated ${suggestions.length} dialog suggestions`);
    return suggestions.slice(0, 4);
  } catch (err) {
    console.error('❌ Error generating dialog suggestions:', err.message);
    return [
      "Continue exploring the key points raised so far",
      "Invite participants to share their perspectives",
      "Summarize the discussion and identify next priorities",
      "Seek consensus on the primary objectives"
    ];
  }
}

/**
 * Analyze sentiment from full meeting transcript for multiple users
 * @param {string} transcript - Full meeting transcript text
 * @returns {Promise<Object>} Object with user keys and {positive, neutral, negative} values
 */
export async function analyzeSentiment(transcript) {
  try {
     const sentimentPromptTemplate = readFileSync(join(__dirname, 'query_prompt_sentiment_analysis.md'), 'utf-8');
    const filledPrompt = sentimentPromptTemplate.replace(/\{\{meeting_transcript\}\}/g, transcript);

    console.log('😊 Analyzing sentiment via OpenClaw...');
    const response = await runOpenclaw(filledPrompt);

    let jsonContent = response.trim();
    if (jsonContent.startsWith('```json')) {
      jsonContent = jsonContent.replace(/```json\s*/, '').replace(/\s*```$/, '');
    } else if (jsonContent.startsWith('```')) {
      jsonContent = jsonContent.replace(/```\s*/, '').replace(/\s*```$/, '');
    }

    try {
      const sentimentData = JSON.parse(jsonContent);
      console.log('✅ Sentiment analysis completed:', Object.keys(sentimentData).length, 'users analyzed');
      return sentimentData;
    } catch (parseError) {
      console.error('❌ Error parsing sentiment JSON:', parseError.message);
      return {};
    }
  } catch (err) {
    console.error('❌ Error analyzing sentiment:', err.message);
    return {};
  }
}

/**
 * Generate a real-time meeting summary from transcript and images
 * @param {string} transcript - Full current meeting transcript in VTT format
 * @param {string} meetingEvents - Meeting events log
 * @param {string[]} imageBase64Array - Array of base64 encoded screen share images
 * @param {string} streamId - Stream ID
 * @param {string} meetingUuid - Meeting UUID
 * @returns {Promise<string>} Real-time meeting summary
 */
export async function generateRealTimeSummary(transcript, meetingEvents = '', imageBase64Array = [], streamId = '', meetingUuid = '') {
  try {
     const summaryPromptTemplate = readFileSync(join(__dirname, 'summary_prompt.md'), 'utf-8');
    const todayDate = new Date().toISOString();

    const filledPrompt = summaryPromptTemplate
      .replace(/\{\{raw_transcript\}\}/g, transcript)
      .replace(/\{\{meeting_events\}\}/g, meetingEvents)
      .replace(/\{\{meeting_uuid\}\}/g, meetingUuid)
      .replace(/\{\{stream_id\}\}/g, streamId)
      .replace(/\{\{TODAYDATE\}\}/g, todayDate);

    console.log('📝 Generating real-time summary via OpenClaw...');
    const response = await chatWithClawdbot(filledPrompt, '', imageBase64Array);

    console.log('✅ Real-time summary generated');
    return response;
  } catch (err) {
    console.error('❌ Error generating real-time summary:', err.message);
    return 'Unable to generate summary at this time. Meeting in progress...';
  }
}

/**
 * Query the current meeting transcript for specific questions
 * @param {string} transcript - Full current meeting transcript
 * @param {string} userQuery - User's question about the meeting
 * @returns {Promise<string>} Contextual answer based on transcript
 */
export async function queryCurrentMeeting(transcript, userQuery) {
  try {
     const queryPromptTemplate = readFileSync(join(__dirname, 'query_prompt_current_meeting.md'), 'utf-8');
    const filledPrompt = queryPromptTemplate
      .replace(/\{\{meeting_transcript\}\}/g, transcript)
      .replace(/\{\{user_query\}\}/g, userQuery);

    console.log('🔍 Querying current meeting via OpenClaw...');
    const response = await runOpenclaw(filledPrompt);

    console.log('✅ Meeting query answered');
    return response;
  } catch (err) {
    console.error('❌ Error querying current meeting:', err.message);
    return 'I apologize, but I was unable to analyze the current meeting transcript. Please try again later.';
  }
}
