#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Zettelkasten - 卡片盒笔记法实现
支持灵感捕获、AI扩展、关联检测和每日回顾功能
"""

import os
import time
import json
import random
import re
from datetime import datetime
from collections import defaultdict

class Zettelkasten:
    def __init__(self, user_id=None):
        self.user_id = user_id or "default"
        self.db_file = f"zettelkasten_{self.user_id}_db.json"
        self.cards = self._load_db()
        self.tag_synonyms = {
            '冥想': ['正念', '冥想练习', '专注力'],
            '效率': [' productivity', '工作效率', '时间管理'],
            '睡眠': ['休息', '睡眠质量', '生物钟'],
            '心理学': ['心理', '认知科学', '行为科学'],
            '习惯': ['习惯养成', '行为模式', '自律'],
            '机器人': ['AI', '机器学习', '自动化'],
            '宠物': ['动物', '猫咪', '狗狗'],
        }
    
    def _load_db(self):
        """加载卡片数据库"""
        if os.path.exists(self.db_file):
            with open(self.db_file, 'r', encoding='utf-8') as f:
                return json.load(f)
        return []
    
    def _save_db(self):
        """保存卡片数据库"""
        with open(self.db_file, 'w', encoding='utf-8') as f:
            json.dump(self.cards, f, ensure_ascii=False, indent=2)
    
    def analyze_content(self, content):
        """分析灵感内容，拆解观点"""
        # 拆分多个观点
        sentences = re.split(r'[。！？；;]+', content)
        sentences = [s.strip() for s in sentences if s.strip()]
        
        # 判断是否有多个独立观点
        if len(sentences) >= 3 and len(content) > 100:
            return sentences
        return [content]
    
    def generate_tags(self, content):
        """生成3-5个相关标签"""
        # 关键词提取
        words = re.findall(r'\b[a-zA-Z\u4e00-\u9fa5]{2,}\b', content.lower())
        
        # 标签生成
        tags = set()
        
        # 匹配同义词
        for word in words:
            for key, synonyms in self.tag_synonyms.items():
                if word in synonyms or word == key:
                    tags.add(key)
                    break
        
        # 添加通用标签
        if not tags:
            tags = {'思考', '想法', '创意'}
        
        # 确保3-5个标签
        while len(tags) < 3:
            tags.add(random.choice(['观点', '见解', '发现', '研究', '探索']))
        
        return list(tags)[:5]
    
    def generate_title(self, content):
        """生成标题"""
        # 提取核心关键词
        sentences = content.split('.')[:2]
        title = sentences[0].strip()
        
        if len(title) > 50:
            title = title[:47] + '...'
        
        return title
    
    def determine_type(self, content):
        """确定笔记类型"""
        if any(key in content for key in ['文献', '论文', '研究', '引用']):
            return '文献'
        elif len(content) > 300:
            return '永久'
        else:
            return '闪念'
    
    def generate_ai_insight(self, content):
        """生成AI洞察"""
        insights = [
            "这个观点涉及心理学领域，可能与习惯养成相关",
            "此观点具有实践指导意义，可应用于日常生活",
            "这是一个有价值的发现，值得深入研究和验证",
            "这个想法与时间管理和效率提升相关",
            "此观点涉及健康管理，可能对生活质量有积极影响",
            "这是一个创新的技术想法，具有广阔的应用前景",
            "这个观点可以进一步扩展到商业应用和产品化",
        ]
        
        # 根据内容选择相关洞察
        if any(key in content for key in ['机器人', 'AI', '机器学习']):
            return "这是一个创新的技术想法，具有广阔的应用前景"
        elif any(key in content for key in ['冥想', '睡眠', '健康']):
            return "此观点涉及健康管理，可能对生活质量有积极影响"
        else:
            return random.choice(insights)
    
    def generate_extended_suggestions(self, content):
        """生成扩展建议"""
        suggestions = []
        
        # 根据内容类型生成不同建议
        if any(key in content for key in ['机器人', 'AI', '机器学习']):
            suggestions.append("技术实现方向：使用CNN/RNN模型进行音频特征提取和分类")
            suggestions.append("数据收集：用户上传音频+标签的众包模式")
            suggestions.append("应用场景：宠物健康监测、情绪识别、行为预测")
            suggestions.append("商业模式：可考虑付费增值服务和API开放")
        elif any(key in content for key in ['冥想', '睡眠', '效率']):
            suggestions.append("科学验证：可通过对照实验验证效果")
            suggestions.append("习惯养成：结合微习惯理论设计实践方案")
            suggestions.append("测量指标：使用心率变异性、专注度测试量化效果")
            suggestions.append("产品化：可开发相关APP或硬件产品")
        elif any(key in content for key in ['宠物', '猫', '狗', '咪咪']):
            suggestions.append("宠物行为学：结合动物行为学理论解读信号")
            suggestions.append("健康监测：通过叫声变化发现潜在健康问题")
            suggestions.append("情感连接：增进主人与宠物之间的情感纽带")
            suggestions.append("市场机会：宠物用品市场的新兴增长点")
        else:
            suggestions.append("可以考虑与现有知识体系建立联系")
            suggestions.append("需要进一步验证观点的普适性")
            suggestions.append("探索可能的应用场景和商业价值")
            suggestions.append("考虑与其他领域的交叉融合")
        
        return suggestions[:3]  # 只返回3条建议
    
    def save_ai_suggestions(self, content, original_card_id=None):
        """保存AI建议到独立卡片库"""
        suggestions = self.generate_extended_suggestions(content)
        
        ai_cards = []
        now = datetime.now()
        timestamp = int(time.time())
        
        for i, suggestion in enumerate(suggestions):
            card = {
                'ID': f'AI_{timestamp + i}',
                'Title': f'扩展建议：{suggestion[:30]}...',
                'Content': suggestion,
                'Tags': ['AI建议', '扩展想法', '研究方向'],
                'Type': '闪念',
                'Date': now.strftime('%Y-%m-%d'),
                'Timestamp': timestamp + i,
                'AI_Insight': '这是AI生成的扩展建议，可作为进一步研究的方向',
                'Related_Card': original_card_id
            }
            
            ai_cards.append(card)
            self.cards.append(card)
        
        self._save_db()
        
        return ai_cards
    
    def create_card(self, content, card_type=None, save_ai_suggestions=True):
        """创建新卡片"""
        now = datetime.now()
        timestamp = int(time.time())
        
        # 分析内容
        insights = self.analyze_content(content)
        
        cards = []
        
        for i, insight in enumerate(insights):
            # 生成元数据
            title = self.generate_title(insight)
            if i > 0:
                title += f" ({i+1})"
            
            tags = self.generate_tags(insight)
            card_type = card_type or self.determine_type(insight)
            ai_insight = self.generate_ai_insight(insight)
            
            # 创建卡片
            card = {
                'ID': timestamp + i,
                'Title': title,
                'Content': insight,
                'Tags': tags,
                'Type': card_type,
                'Date': now.strftime('%Y-%m-%d'),
                'Timestamp': timestamp + i,
                'AI_Insight': ai_insight
            }
            
            cards.append(card)
            self.cards.append(card)
        
        # 保存数据库
        self._save_db()
        
        # 自动保存AI建议
        ai_cards = []
        if save_ai_suggestions and cards:
            ai_cards = self.save_ai_suggestions(content, cards[0]['ID'])
        
        # 生成Markdown输出
        markdown_output = ""
        for card in cards:
            markdown_output += f"""```markdown
---
ID: {card['ID']}
Tags: {' '.join(f'#{tag}' for tag in card['Tags'])}
Type: {card['Type']}
Date: {card['Date']}
---

## {card['Title']}
{card['Content']}

> AI 洞察：{card['AI_Insight']}
```\n"""
        
        # 添加AI建议
        if ai_cards:
            markdown_output += "\n## 🚀 AI扩展建议\n"
            for card in ai_cards:
                markdown_output += f"""
```markdown
---
ID: {card['ID']}
Tags: {' '.join(f'#{tag}' for tag in card['Tags'])}
Type: {card['Type']}
Date: {card['Date']}
Related: {card['Related_Card']}
---

## {card['Title']}
{card['Content']}

> AI 洞察：{card['AI_Insight']}
```
"""
        
        return cards, ai_cards, markdown_output
    
    def find_connections(self, content):
        """寻找潜在联系 - 包含AI建议卡片"""
        if not self.cards or len(self.cards) < 2:
            return None
        
        # 模拟检索关联概念
        keywords = set(re.findall(r'\b[a-zA-Z\u4e00-\u9fa5]{2,}\b', content.lower()))
        
        potential_connections = []
        
        # 搜索所有卡片（包括AI建议）
        for card in self.cards:
            # 跳过自己
            if 'Content' in card and card['Content'] == content:
                continue
                
            card_keywords = set()
            
            # 提取卡片关键词
            for tag in card['Tags']:
                card_keywords.add(tag.lower())
                if tag.lower() in self.tag_synonyms:
                    card_keywords.update([s.lower() for s in self.tag_synonyms[tag.lower()]])
            
            if 'Content' in card:
                card_keywords.update(set(re.findall(r'\b[a-zA-Z\u4e00-\u9fa5]{2,}\b', card['Content'].lower())))
            
            # 计算相似度
            common = keywords.intersection(card_keywords)
            if common:
                potential_connections.append({
                    'card': card,
                    'common_keywords': list(common)
                })
        
        # 随机选择2个
        if len(potential_connections) > 2:
            return random.sample(potential_connections, 2)
        
        return potential_connections
    
    def suggest_connection(self, connections):
        """生成关联建议"""
        if not connections:
            return """🔍 系统提示：
暂时没有找到相关的旧笔记，继续积累卡片吧！
"""
        
        response = """🔗 潜在联系：\n"""
        
        for conn in connections:
            card = conn['card']
            relation = random.choice(['支持', '延伸', '补充', '对比'])
            response += f"你之前在 {card['Date']} 提到过 \"{card['Title']}\", 这两个观点似乎存在 {relation} 关系？\n"
        
        return response
    
    def get_random_card(self):
        """获取随机卡片"""
        if not self.cards:
            return None
        return random.choice(self.cards)
    
    def generate_daily_review(self):
        """生成每日回顾"""
        card = self.get_random_card()
        if not card:
            return "📅 每日回顾：\n还没有任何卡片，开始记录你的第一个灵感吧！"
        
        return f"""📅 每日回顾：
```markdown
---
ID: {card['ID']}
Tags: {' '.join(f'#{tag}' for tag in card['Tags'])}
Type: {card['Type']}
Date: {card['Date']}
---

## {card['Title']}
{card['Content']}

> AI 洞察：{card['AI_Insight']}
```\n
💡 思考：这个观点现在对你还有价值吗？是否需要更新或关联新想法？
"""

def handle_message(user_input, user_id=None):
    """消息处理函数"""
    zk = Zettelkasten(user_id)
    
    if user_input.startswith("记录灵感："):
        content = user_input[6:].strip()
        
        if not content:
            return "❌ 请输入灵感内容"
        
        cards, ai_cards, markdown = zk.create_card(content)
        
        # 寻找关联
        connections = zk.find_connections(content)
        suggestion = zk.suggest_connection(connections)
        
        if suggestion:
            markdown += f"\n{suggestion}\n"
            markdown += "\n💡 输入你的反馈或 '停止' 结束对话："
        
        return markdown
    
    elif user_input == "每日回顾":
        return zk.generate_daily_review()
    
    elif user_input == "统计":
        return f"📊 当前共有 {len(zk.cards)} 张卡片"
    
    else:
        return "❌ 未知命令，请使用 '记录灵感：[内容]' 或 '每日回顾'"

if __name__ == '__main__':
    print("📝 Zettelkasten 卡片盒笔记法系统已启动")
    print("输入 '记录灵感：[内容]' 开始记录，输入 'exit' 退出")
    
    while True:
        user_input = input("\n> ").strip()
        
        if user_input.lower() in ['exit', 'quit', '停止']:
            print("👋 再见！")
            break
        
        response = handle_message(user_input)
        print(response)