---
name: zerion-api
description: >
  Query blockchain wallet data, token prices, and transaction history using the Zerion API
  via its MCP connector. Use this skill whenever the user asks about: crypto wallet balances,
  portfolio values, token holdings or positions, DeFi positions (staking, lending, LP),
  wallet PnL (profit and loss), transaction history, token/fungible asset prices or charts,
  NFT holdings or NFT portfolio value, or any web3 wallet analytics. Triggers on mentions of
  wallet addresses (0x...), ENS names, token names/symbols, "portfolio", "positions", "PnL",
  "transactions", "balance", "holdings", "NFTs", or any crypto/DeFi analytics queries.
  Also use when building artifacts or dashboards that display wallet or token data.
---

# Zerion API Skill

Query web3 wallet data, token prices, NFTs, and transaction history via the Zerion MCP connector.

## Authentication

Zerion API requires a key for every request. The key is **not** stored in the MCP connector settings — the user must provide it each chat session.

### Workflow

1. **At the start of any Zerion-related task**, if no API key has been provided yet, ask:
   *"To query Zerion, I'll need your API key. You can find it at https://dashboard.zerion.io/. Please paste it here."*
2. **Store the key in memory** for the duration of the conversation. Never write it to files, display it in artifacts, or log it.
3. **Pass the key** to the MCP server or REST calls as described below.

### Auth Format

Zerion uses HTTP Basic Auth: the API key is the username, password is empty.

```
Authorization: Basic <base64(API_KEY + ":")>
```

Example: key `zk_dev_abc123` → base64 of `zk_dev_abc123:` → `emtfZGV2X2FiYzEyMzo=`

## MCP Connection

The Zerion API MCP server is at `https://developers.zerion.io/mcp`.

When using MCP tools directly (outside artifacts), pass the API key as required by the tool parameters.

When building artifacts that call the Anthropic API with MCP, include the key in the inner prompt so the inner Claude can authenticate:

```javascript
mcp_servers: [
  { type: "url", url: "https://developers.zerion.io/mcp", name: "zerion-mcp" }
]
```

**Important**: In artifacts, receive the API key as a prop or state variable — never hardcode it. Example pattern:

```jsx
// User inputs key via a secure input field (type="password")
const [apiKey, setApiKey] = useState("");

// Pass key to inner Claude prompt so MCP calls authenticate
const prompt = `Using the Zerion API key: ${apiKey}, get portfolio for wallet 0x...`;
```

## Quick Reference: Common Workflows

### 1. Wallet Portfolio Overview

Prompt the inner Claude to call the Zerion API for the wallet's portfolio:

```
Get the portfolio for wallet {address} in USD. Include total value, daily changes,
and distribution by chain and position type.
```

**Endpoint:** `GET /v1/wallets/{address}/portfolio`
- `currency`: usd (default), eth, btc, eur, etc.
- `filter[positions]`: `only_simple` (default), `only_complex` (DeFi), `no_filter` (all)

### 2. Wallet Token Positions

```
List all fungible positions for wallet {address}, sorted by value descending.
```

**Endpoint:** `GET /v1/wallets/{address}/positions/`
- `filter[positions]`: `only_simple` | `only_complex` | `no_filter`
- `filter[chain_ids]`: comma-separated chain IDs (e.g., `ethereum,polygon`)
- `filter[position_types]`: `wallet`, `deposit`, `staked`, `loan`, `locked`, `reward`, `investment`
- `sort`: `-value` (highest first) or `value`
- `filter[trash]`: `only_non_trash` (default)

### 3. Transaction History

```
Get recent transactions for wallet {address}, filter for trades only.
```

**Endpoint:** `GET /v1/wallets/{address}/transactions/`
- `filter[operation_types]`: `trade`, `send`, `receive`, `deposit`, `withdraw`, `mint`, `burn`, `claim`, `approve`, etc.
- `filter[chain_ids]`: filter by chain
- `filter[min_mined_at]` / `filter[max_mined_at]`: timestamp in milliseconds
- `page[size]`: max 100

### 4. Profit & Loss (PnL)

```
Get PnL for wallet {address}. Show realized gain, unrealized gain, fees, and net invested.
```

**Endpoint:** `GET /v1/wallets/{address}/pnl`
- Returns: `realized_gain`, `unrealized_gain`, `total_fee`, `net_invested`, `received_external`, `sent_external`
- Uses FIFO method
- `filter[chain_ids]`, `filter[fungible_ids]`: narrow scope

### 5. Wallet Balance Chart

```
Get the balance chart for wallet {address} over the past month.
```

**Endpoint:** `GET /v1/wallets/{address}/charts/{chart_period}`
- `chart_period`: `hour`, `day`, `week`, `month`, `3months`, `6months`, `year`, `5years`, `max`
- Returns array of `[timestamp, balance]` points

### 6. Token Price & Search

```
Search for the fungible asset "ethereum" and return its price and market data.
```

**Endpoint:** `GET /v1/fungibles/`
- `filter[search_query]`: text search (e.g., "ethereum", "USDC")
- `sort`: `-market_data.market_cap`, `-market_data.price.last`, etc.
- Returns: `name`, `symbol`, `price`, `market_cap`, `circulating_supply`, `changes` (1d/30d/90d/365d)

### 7. Token Price Chart

```
Get the price chart for fungible {fungible_id} over the past week.
```

**Endpoint:** `GET /v1/fungibles/{fungible_id}/charts/{chart_period}`
- Same chart periods as wallet charts
- Returns `[timestamp, price]` points

### 8. NFT Positions

```
List all NFT positions for wallet {address}, sorted by floor price descending.
```

**Endpoint:** `GET /v1/wallets/{address}/nft-positions/`
- `sort`: `-floor_price`, `floor_price`, `created_at`, `-created_at`
- `filter[chain_ids]`: filter by chain
- `include`: `nfts`, `nft_collections`, `wallet_nft_collections` for richer data

### 9. NFT Portfolio Value

**Endpoint:** `GET /v1/wallets/{address}/nft-portfolio`
- Returns total NFT portfolio value

## Building Artifacts with Zerion Data

When building React/HTML artifacts that display Zerion data:

1. **Collect the API key securely** via a `type="password"` input field
2. **Never display or log** the key in the UI
3. **Pass it through the MCP prompt** so the inner Claude can authenticate

```javascript
// apiKey comes from a password input, never hardcoded
const response = await fetch("https://api.anthropic.com/v1/messages", {
  method: "POST",
  headers: { "Content-Type": "application/json" },
  body: JSON.stringify({
    model: "claude-sonnet-4-20250514",
    max_tokens: 1000,
    messages: [
      {
        role: "user",
        content: `Use the Zerion API with key "${apiKey}" to get the portfolio
                  overview for wallet ${walletAddress} in USD.
                  Return ONLY a JSON object with: totalValue, dailyChangePercent,
                  dailyChangeAbsolute, topChains (array of {chain, value}),
                  positionBreakdown (wallet, deposited, staked, borrowed, locked).`
      }
    ],
    mcp_servers: [
      { type: "url", url: "https://developers.zerion.io/mcp", name: "zerion-mcp" }
    ]
  })
});
```

### Processing MCP Responses

MCP responses contain multiple content blocks. Extract data by type:

```javascript
const data = await response.json();

// Get tool results (actual Zerion data)
const toolResults = data.content
  .filter(item => item.type === "mcp_tool_result")
  .map(item => item.content?.[0]?.text || "")
  .join("\n");

// Get Claude's text analysis
const textResponses = data.content
  .filter(item => item.type === "text")
  .map(item => item.text)
  .join("\n");
```

## Key Notes

- **Address formats**: Supports EVM addresses (0x...) and Solana addresses. ENS names may need resolution first.
- **Solana limitations**: No protocol positions or NFT transactions for Solana addresses.
- **Currency options**: `usd`, `eth`, `btc`, `eur`, `krw`, `rub`, `gbp`, `aud`, `cad`, `inr`, `jpy`, `nzd`, `try`, `zar`, `cny`, `chf`
- **Pagination**: Use `links.next` from responses for pagination; never construct `page[after]` manually.
- **Rate limits**: API returns 429 on throttling. Implement retry with backoff in artifacts.
- **Treat IDs as opaque strings**: Never assume format of IDs; they may change.
- **DeFi positions**: Use `filter[positions]=no_filter` to include protocol positions alongside wallet positions.
- **LP grouping**: Liquidity pool positions share a `group_id` attribute — group by it to display pools correctly.

## Detailed Endpoint Reference

For full parameter details, response schemas, and edge cases:
- **Wallet endpoints**: See [references/wallet-endpoints.md](references/wallet-endpoints.md)
- **Fungible & NFT endpoints**: See [references/fungible-nft-endpoints.md](references/fungible-nft-endpoints.md)
