#!/bin/bash
# Skill Auditor - Markdown Report Generator
# Generates a comprehensive audit report for all installed skills
# Usage: bash report.sh [skills-dir] [--output /path/to/report.md]

SCRIPT_DIR="$(cd "$(dirname "$0")" && pwd)"
SKILLS_DIR="${1:-/opt/homebrew/lib/node_modules/openclaw/skills}"
OUTPUT=""

shift 2>/dev/null
while [[ $# -gt 0 ]]; do
    case $1 in
        --output) OUTPUT="$2"; shift 2 ;;
        *) shift ;;
    esac
done

if [ -z "$OUTPUT" ]; then
    OUTPUT="/tmp/skill-audit-report-$(date +%Y%m%d).md"
fi

echo "Generating audit report..."
echo "Skills directory: $SKILLS_DIR"
echo "Output: $OUTPUT"

# Collect all results
BATCH_JSON=$(bash "$SCRIPT_DIR/audit-all.sh" "$SKILLS_DIR" --json 2>/dev/null)

python3 << PYREPORT > "$OUTPUT"
import json, sys
from datetime import datetime

batch = json.loads('''$BATCH_JSON''')
total = batch["total"]
pass_count = batch["pass"]
review_count = batch["review"]
fail_count = batch["fail"]
results = batch["results"]

now = datetime.now().strftime("%Y-%m-%d %H:%M")

print(f"# Skill Audit Report")
print(f"_Generated: {now}_")
print(f"_Skills directory: $SKILLS_DIR_")
print()
print(f"## Summary")
print()
print(f"| Metric | Count |")
print(f"|--------|-------|")
print(f"| Total Skills | {total} |")
print(f"| ✅ Pass | {pass_count} |")
print(f"| ⚠️ Review | {review_count} |")
print(f"| ❌ Fail | {fail_count} |")
print()

# Pass rate
pass_rate = (pass_count / total * 100) if total > 0 else 0
print(f"**Pass Rate: {pass_rate:.0f}%**")
print()

# Failures first
fails = [r for r in results if r.get("verdict") == "fail"]
if fails:
    print(f"## ❌ Critical Failures ({len(fails)})")
    print()
    for r in fails:
        name = r.get("skill", "unknown")
        items = r.get("items", [])
        crits = [i for i in items if i["severity"] == "critical"]
        print(f"### {name}")
        print(f"- Criticals: {r.get('criticals', 0)}")
        for c in crits:
            print(f"  - **{c['check']}**: {c['message']}")
            if c.get("file"):
                print(f"    - File: {c['file']}")
        print()

# Reviews
reviews = [r for r in results if r.get("verdict") == "review"]
if reviews:
    print(f"## ⚠️ Needs Review ({len(reviews)})")
    print()
    for r in reviews:
        name = r.get("skill", "unknown")
        items = r.get("items", [])
        warns = [i for i in items if i["severity"] == "warning"]
        print(f"### {name}")
        for w in warns:
            print(f"- **{w['check']}**: {w['message']}")
        print()

# Pass list (compact)
passes = [r for r in results if r.get("verdict") == "pass"]
if passes:
    print(f"## ✅ Passing Skills ({len(passes)})")
    print()
    print("| Skill | Infos | Allowlisted |")
    print("|-------|-------|-------------|")
    for r in passes:
        name = r.get("skill", "unknown")
        infos = r.get("infos", 0)
        al = "✓" if r.get("allowlisted") else ""
        print(f"| {name} | {infos} | {al} |")
    print()

# Allowlist summary
allowlisted = [r for r in results if r.get("allowlisted")]
if allowlisted:
    print(f"## Allowlisted Skills ({len(allowlisted)})")
    print()
    print("These skills legitimately handle credentials and have criticals downgraded to info:")
    print()
    for r in allowlisted:
        print(f"- {r.get('skill', 'unknown')}")
    print()

print("---")
print(f"_Report generated by Skill Auditor v3.0.0_")
PYREPORT

echo ""
echo "Report written to: $OUTPUT"
echo ""
head -20 "$OUTPUT"
echo "..."
