---
name: yahooquery
description: Access Yahoo Finance data including real-time pricing, fundamentals, analyst estimates, options, news, and historical data via the yahooquery Python library.
---

# yahooquery Skill

Comprehensive access to Yahoo Finance data via the `yahooquery` Python library. This library provides programmatic access to nearly all Yahoo Finance endpoints, including real-time pricing, fundamentals, analyst estimates, options, news, and premium research.

## Core Classes

### 1. **Ticker** (Company-Specific Data)
The primary interface for retrieving data about one or more securities.

```python
from yahooquery import Ticker

# Single or multiple symbols
aapl = Ticker('AAPL')
tickers = Ticker('AAPL MSFT NVDA', asynchronous=True)
```

### 2. **Screener** (Predefined Stock Lists)
Access to pre-built screeners for discovering stocks by criteria.

```python
from yahooquery import Screener

s = Screener()
screeners = s.available_screeners  # List all available screeners
data = s.get_screeners(['day_gainers', 'most_actives'], count=10)
```

### 3. **Research** (Premium Subscription Required)
Access proprietary research reports and trade ideas.

```python
from yahooquery import Research

r = Research(username='you@email.com', password='password')
reports = r.reports(report_type='Analyst Report', report_date='Last Week')
trades = r.trades(trend='Bullish', term='Short term')
```

---

## Ticker Class: Data Modules

The `Ticker` class exposes dozens of data endpoints via properties and methods.

### 📊 **Financial Statements**
- `.income_statement(frequency='a', trailing=True)` - Income statement (annual/quarterly)
- `.balance_sheet(frequency='a', trailing=True)` - Balance sheet
- `.cash_flow(frequency='a', trailing=True)` - Cash flow statement
- `.all_financial_data(frequency='a')` - Combined financials + valuation measures
- `.valuation_measures` - EV/EBITDA, P/E, P/B, P/S across periods

### 📈 **Pricing & Market Data**
- `.price` - Current pricing, market cap, 52-week range
- `.history(period='1y', interval='1d', start=None, end=None)` - Historical OHLC
  - **period**: `1d`, `5d`, `1mo`, `3mo`, `6mo`, `1y`, `2y`, `5y`, `10y`, `ytd`, `max`
  - **interval**: `1m`, `2m`, `5m`, `15m`, `30m`, `60m`, `90m`, `1h`, `1d`, `5d`, `1wk`, `1mo`, `3mo`
- `.option_chain` - Full options chain (all expirations)

### 🔍 **Analysis & Estimates**
- `.calendar_events` - Next earnings date, EPS/revenue estimates
- `.earning_history` - Actual vs. estimated EPS (last 4 quarters)
- `.earnings` - Historical quarterly/annual earnings and revenue
- `.earnings_trend` - Analyst estimates for upcoming periods
- `.recommendation_trend` - Buy/Sell/Hold rating changes over time
- `.gradings` - Recent analyst upgrades/downgrades

### 🏢 **Company Fundamentals**
- `.asset_profile` - Address, industry, sector, business summary, officers
- `.company_officers` - Executives with compensation details
- `.summary_profile` - Condensed company information
- `.key_stats` - Forward P/E, profit margin, beta, shares outstanding
- `.financial_data` - Financial KPIs (ROE, ROA, debt-to-equity, margins)

### 👥 **Ownership & Governance**
- `.insider_holders` - List of insider holders and positions
- `.insider_transactions` - Recent buy/sell transactions by insiders
- `.institution_ownership` - Top institutional holders
- `.fund_ownership` - Top mutual fund holders
- `.major_holders` - Ownership summary (institutional %, insider %, float)

### 🌍 **ESG & Ratings**
- `.esg_scores` - Environmental, Social, Governance scores and controversies
- `.recommendation_rating` - Analyst consensus (Strong Buy → Strong Sell)

### 📰 **News & Insights**
- `.news()` - Recent news articles
- `.technical_insights` - Bullish/bearish technical patterns

### 💰 **Funds & ETFs Only**
- `.fund_holding_info` - Top holdings, bond/equity breakdown
- `.fund_performance` - Historical performance and returns
- `.fund_bond_holdings` / `.fund_bond_ratings` - Bond maturity and credit ratings
- `.fund_equity_holdings` - P/E, P/B, P/S for equity holdings

### 📊 **Other Modules**
- `.summary_detail` - Trading stats (day high/low, volume, avg volume)
- `.default_key_statistics` - Enterprise value, trailing P/E, forward P/E
- `.index_trend` - Performance relative to a benchmark index
- `.quote_type` - Security type, exchange, market

---

## Global Functions

```python
import yahooquery as yq

# Search
results = yq.search('NVIDIA')

# Market Data
market = yq.get_market_summary(country='US')  # Major indices snapshot
trending = yq.get_trending(country='US')  # Trending tickers

# Utilities
currencies = yq.get_currencies()  # List of supported currencies
exchanges = yq.get_exchanges()  # List of exchanges
rate = yq.currency_converter('USD', 'EUR')  # Exchange rate
```

---

## Configuration & Keyword Arguments

The `Ticker`, `Screener`, and `Research` classes accept these optional parameters:

### Performance & Reliability
- `asynchronous=True` - Make requests asynchronously (for multiple symbols)
- `max_workers=8` - Number of concurrent workers (when async)
- `retry=5` - Number of retry attempts
- `backoff_factor=0.3` - Exponential backoff between retries
- `status_forcelist=[429, 500, 502, 503, 504]` - HTTP codes to retry
- `timeout=5` - Request timeout in seconds

### Data Format & Validation
- `formatted=False` - If `True`, returns data with `{raw, fmt, longFmt}` structure
- `validate=True` - Validate symbols on instantiation (invalid → `.invalid_symbols`)
- `country='United States'` - Regional data/news (france, germany, canada, etc.)

### Network & Auth
- `proxies={'http': 'http://proxy:port'}` - HTTP/HTTPS proxy
- `user_agent='...'` - Custom user agent string
- `verify=True` - SSL certificate verification
- `username='you@email.com'` / `password='...'` - Yahoo Finance Premium login

### Advanced (Shared Sessions)
- `session=...` / `crumb=...` - Share auth between `Research` and `Ticker` instances

---

## Best Practices

### 1. **Async for Multiple Symbols**
```python
tickers = Ticker('AAPL MSFT NVDA TSLA', asynchronous=True)
prices = tickers.price  # Returns dict keyed by symbol
```

### 2. **Handling DataFrames**
Most financial methods return `pandas.DataFrame`. Convert for JSON output:
```python
df = aapl.income_statement()
print(df.to_json(orient='records', date_format='iso'))
```

### 3. **Historical Data - 1-Minute Intervals**
Yahoo limits 1-minute data to 7 days per request. For 30 days:
```python
tickers = Ticker('AAPL', asynchronous=True)
df = tickers.history(period='1mo', interval='1m')  # Makes 4 requests automatically
```

### 4. **Premium Users: Combining Research + Ticker**
```python
r = Research(username='...', password='...')
reports = r.reports(sector='Technology', investment_rating='Bullish')

# Reuse session for Ticker
tickers = Ticker('AAPL', session=r.session, crumb=r.crumb)
data = tickers.asset_profile
```

---

## Common Use Cases

### Portfolio Analysis
```python
portfolio = Ticker('AAPL MSFT NVDA', asynchronous=True)
summary = portfolio.summary_detail
earnings = portfolio.earnings
history = portfolio.history(period='1y')
```

### Screening & Discovery
```python
s = Screener()
gainers = s.get_screeners(['day_gainers'], count=20)
# Returns DataFrame with price, volume, % change, etc.
```

### Options Analysis
```python
nvda = Ticker('NVDA')
options = nvda.option_chain
# Filter for calls/puts, strikes, expirations
```

### Earnings Calendar
```python
tickers = Ticker('AAPL MSFT NVDA')
calendar = tickers.calendar_events
# Shows next earnings date + analyst estimates
```

---

## Reference Documentation

Full API docs at: `/Users/henryzha/.openclaw/workspace-research/skills/yahooquery/references/`

- `index.md` - Overview of classes and functions
- `ticker/` - Detailed breakdown of all Ticker methods
- `screener.md` - Screener class guide
- `research.md` - Research class (Premium)
- `keyword_arguments.md` - Complete list of configuration options
- `misc.md` - Global utility functions
- `advanced.md` - Sharing sessions between Research and Ticker

---

## Environment

- **Installation**: `python3 -m pip install yahooquery`
- **Dependencies**: pandas, requests-futures, tqdm, beautifulsoup4, lxml
- **Python Version**: 3.7+

---

## Notes

- Yahoo Finance may rate-limit or block requests. Use `retry`, `backoff_factor`, and `status_forcelist` for robustness.
- Premium features (Research class) require a paid Yahoo Finance Premium subscription.
- Data accuracy and availability depend on Yahoo Finance's upstream data providers.
