#!/usr/bin/env python3
"""
Yahoo Finance FOREX News Fetcher
Fetches real-time FOREX news and market data for major currency pairs
"""

import argparse
import json
import sys
from datetime import datetime
from typing import Dict, List, Optional

try:
    import yfinance as yf
except ImportError:
    print("Error: yfinance package not found. Install with: pip install yfinance>=0.2.40", file=sys.stderr)
    sys.exit(1)

# Mapping of user-friendly pair names to Yahoo Finance symbols
PAIR_SYMBOLS = {
    "EURUSD": "EURUSD=X",
    "GBPUSD": "GBPUSD=X",
    "USDJPY": "USDJPY=X",
    "USDCHF": "USDCHF=X",
    "AUDUSD": "AUDUSD=X",
    "USDCAD": "USDCAD=X",
    "NZDUSD": "NZDUSD=X",
}

# Sentiment keywords for analysis
BULLISH_KEYWORDS = [
    "strengthens", "rallies", "rally", "gains", "rises", "climbs",
    "hawkish", "rate hike", "growth", "strong", "surge", "up",
    "bullish", "optimistic", "positive", "recovery", "rebounds"
]

BEARISH_KEYWORDS = [
    "weakens", "falls", "declines", "drops", "slides", "plunges",
    "dovish", "rate cut", "recession", "weak", "down", "slumps",
    "bearish", "pessimistic", "negative", "slowdown", "tumbles"
]


def calculate_sentiment(news_items: List[Dict]) -> Dict:
    """
    Calculate sentiment score based on news headlines
    
    Returns:
        dict: Sentiment analysis with score and recommendation
    """
    sentiment_score = 0
    
    for item in news_items:
        title = item.get("title", "").lower()
        
        # Count bullish keywords
        for keyword in BULLISH_KEYWORDS:
            if keyword in title:
                sentiment_score += 1
        
        # Count bearish keywords
        for keyword in BEARISH_KEYWORDS:
            if keyword in title:
                sentiment_score -= 1
    
    # Determine recommendation based on score
    if sentiment_score > 2:
        recommendation = "BUY"
    elif sentiment_score < -2:
        recommendation = "SELL"
    else:
        recommendation = "HOLD"
    
    return {
        "pair_sentiment": sentiment_score,
        "recommendation": recommendation,
        "analysis": f"Sentiment is {'bullish' if sentiment_score > 0 else 'bearish' if sentiment_score < 0 else 'neutral'} based on keyword analysis"
    }


def fetch_forex_data(pair: str, limit: int = 10) -> Dict:
    """
    Fetch FOREX news and market data from Yahoo Finance
    
    Args:
        pair: Currency pair code (e.g., EURUSD)
        limit: Maximum number of news articles to fetch
        
    Returns:
        dict: FOREX data including rate, news, and sentiment
    """
    if pair not in PAIR_SYMBOLS:
        raise ValueError(f"Unsupported pair: {pair}. Supported pairs: {', '.join(PAIR_SYMBOLS.keys())}")
    
    symbol = PAIR_SYMBOLS[pair]
    
    # Fetch ticker data
    ticker = yf.Ticker(symbol)
    
    # Get current rate and info
    try:
        info = ticker.info
        current_rate = info.get("regularMarketPrice") or info.get("previousClose")
        change_pct = info.get("regularMarketChangePercent", 0)
    except Exception as e:
        # Fallback if info is unavailable
        try:
            hist = ticker.history(period="1d")
            if not hist.empty:
                current_rate = hist['Close'].iloc[-1]
                change_pct = ((hist['Close'].iloc[-1] - hist['Open'].iloc[0]) / hist['Open'].iloc[0]) * 100
            else:
                current_rate = None
                change_pct = 0
        except:
            current_rate = None
            change_pct = 0
    
    # Fetch news
    news_items = []
    try:
        news = ticker.news
        for article in news[:limit]:
            news_item = {
                "title": article.get("title", ""),
                "publisher": article.get("publisher", "Unknown"),
                "published": datetime.fromtimestamp(article.get("providerPublishTime", 0)).strftime("%Y-%m-%d %H:%M:%S"),
            }
            
            # Add link if available
            if "link" in article:
                news_item["link"] = article["link"]
            
            news_items.append(news_item)
    except Exception as e:
        # If news fetch fails, continue with empty news
        pass
    
    # Calculate sentiment
    sentiment = calculate_sentiment(news_items)
    
    # Build response
    response = {
        "pair": pair,
        "symbol": symbol,
        "current_rate": round(current_rate, 5) if current_rate else None,
        "change_pct": round(change_pct, 3) if change_pct else 0,
        "news_count": len(news_items),
        "news": news_items,
        "sentiment": sentiment,
        "timestamp": datetime.now().strftime("%Y-%m-%d %H:%M:%S")
    }
    
    return response


def main():
    """Main entry point"""
    parser = argparse.ArgumentParser(
        description="Fetch FOREX news and market data from Yahoo Finance"
    )
    parser.add_argument(
        "pair",
        type=str,
        help=f"Currency pair ({', '.join(PAIR_SYMBOLS.keys())})"
    )
    parser.add_argument(
        "--limit",
        type=int,
        default=10,
        help="Maximum number of news articles to fetch (default: 10, max: 50)"
    )
    parser.add_argument(
        "--pretty",
        action="store_true",
        help="Pretty print JSON output"
    )
    
    args = parser.parse_args()
    
    # Validate limit
    if args.limit < 1 or args.limit > 50:
        print("Error: limit must be between 1 and 50", file=sys.stderr)
        sys.exit(1)
    
    # Normalize pair name
    pair = args.pair.upper().replace("/", "").replace("-", "")
    
    try:
        # Fetch data
        data = fetch_forex_data(pair, args.limit)
        
        # Output JSON
        if args.pretty:
            print(json.dumps(data, indent=2))
        else:
            print(json.dumps(data))
        
    except ValueError as e:
        print(f"Error: {e}", file=sys.stderr)
        sys.exit(1)
    except Exception as e:
        print(f"Error fetching FOREX data: {e}", file=sys.stderr)
        sys.exit(1)


if __name__ == "__main__":
    main()
