# API Examples

This document provides detailed examples of using the Yahoo Finance FOREX skill.

## Basic Usage

### Fetch EUR/USD News

```bash
python3 scripts/fetch_forex_news.py EURUSD --limit 5
```

**Sample Output:**
```json
{
  "pair": "EURUSD",
  "symbol": "EURUSD=X",
  "current_rate": 1.10250,
  "change_pct": 0.136,
  "news_count": 5,
  "news": [
    {
      "title": "ECB maintains hawkish stance on rates",
      "publisher": "Reuters",
      "published": "2026-02-02 14:30:00",
      "link": "https://finance.yahoo.com/news/ecb-rates-123456.html"
    },
    {
      "title": "Euro strengthens against dollar on strong PMI data",
      "publisher": "Bloomberg",
      "published": "2026-02-02 13:15:00",
      "link": "https://finance.yahoo.com/news/euro-pmi-234567.html"
    }
  ],
  "sentiment": {
    "pair_sentiment": 3,
    "recommendation": "BUY",
    "analysis": "Sentiment is bullish based on keyword analysis"
  },
  "timestamp": "2026-02-02 15:45:23"
}
```

## Advanced Examples

### Get GBP/USD with Pretty Print

```bash
python3 scripts/fetch_forex_news.py GBPUSD --limit 8 --pretty
```

This will format the JSON output with indentation for better readability.

### Check USD/JPY Sentiment

```bash
python3 scripts/fetch_forex_news.py USDJPY --limit 10
```

### Get Recent AUD/USD News

```bash
python3 scripts/fetch_forex_news.py AUDUSD --limit 15
```

## Pair Name Variations

The script accepts multiple formats for pair names:

```bash
# All of these work for EUR/USD:
python3 scripts/fetch_forex_news.py EURUSD
python3 scripts/fetch_forex_news.py EUR/USD
python3 scripts/fetch_forex_news.py EUR-USD
python3 scripts/fetch_forex_news.py eurusd
```

## Integration with OpenClaw

When OpenClaw receives a FOREX-related query, it should:

1. **Identify the currency pair** from the user's question
2. **Run the script** with appropriate parameters
3. **Parse the JSON output**
4. **Analyze the data** and provide insights

### Example OpenClaw Workflow

**User Query:** "What's happening with the euro today?"

**OpenClaw Actions:**
```bash
# Step 1: Fetch EUR/USD data
python3 scripts/fetch_forex_news.py EURUSD --limit 8

# Step 2: Parse JSON output
# Step 3: Generate response based on sentiment and news
```

**OpenClaw Response:**
```
The EUR/USD is currently trading at 1.10250, up 0.136% today.

Recent News:
• ECB maintains hawkish stance on rates (Reuters, 2h ago)
• Euro strengthens against dollar on strong PMI data (Bloomberg, 3h ago)

Sentiment Analysis: BULLISH (Score: +3)
Recommendation: BUY

The euro is showing strength today driven by the ECB's hawkish comments
and positive economic data. The pair has bullish momentum.
```

## Error Handling

### Invalid Pair

```bash
python3 scripts/fetch_forex_news.py INVALID
```

**Output:**
```
Error: Unsupported pair: INVALID. Supported pairs: EURUSD, GBPUSD, USDJPY, USDCHF, AUDUSD, USDCAD, NZDUSD
```

### Invalid Limit

```bash
python3 scripts/fetch_forex_news.py EURUSD --limit 100
```

**Output:**
```
Error: limit must be between 1 and 50
```

## Output Fields Reference

| Field | Type | Description |
|-------|------|-------------|
| `pair` | string | Currency pair code (e.g., EURUSD) |
| `symbol` | string | Yahoo Finance symbol (e.g., EURUSD=X) |
| `current_rate` | float | Current exchange rate (5 decimals) |
| `change_pct` | float | 24-hour percentage change |
| `news_count` | integer | Number of news articles returned |
| `news[]` | array | Array of news articles |
| `news[].title` | string | Article headline |
| `news[].publisher` | string | News source |
| `news[].published` | string | Publication timestamp (YYYY-MM-DD HH:MM:SS) |
| `news[].link` | string | Article URL (optional) |
| `sentiment` | object | Sentiment analysis results |
| `sentiment.pair_sentiment` | integer | Sentiment score (-10 to +10) |
| `sentiment.recommendation` | string | Trading recommendation (BUY/SELL/HOLD) |
| `sentiment.analysis` | string | Human-readable sentiment description |
| `timestamp` | string | Query timestamp (YYYY-MM-DD HH:MM:SS) |

## Tips for Best Results

1. **Use appropriate limits**: 5-10 articles for quick analysis, 15-20 for comprehensive review
2. **Check the timestamp**: Ensure data is recent
3. **Read multiple articles**: Don't rely on a single headline
4. **Consider the source**: Some publishers are more reliable than others
5. **Cross-reference sentiment**: Compare keyword sentiment with actual news content
6. **Watch for breaking news**: Recent articles (< 1 hour old) may have higher impact
7. **Monitor multiple pairs**: USD crosses often move together based on dollar strength

## Python Integration

You can also import and use the script as a Python module:

```python
import sys
sys.path.append('scripts')
from fetch_forex_news import fetch_forex_data

# Fetch data
data = fetch_forex_data('EURUSD', limit=10)

# Access fields
print(f"EUR/USD: {data['current_rate']}")
print(f"Sentiment: {data['sentiment']['recommendation']}")

for article in data['news']:
    print(f"- {article['title']}")
```

## Automation Examples

### Cron Job for Regular Updates

```bash
# Run every hour during market hours
0 8-17 * * 1-5 python3 /path/to/scripts/fetch_forex_news.py EURUSD --limit 5 > /tmp/eurusd_update.json
```

### Shell Script for Multiple Pairs

```bash
#!/bin/bash
# fetch_all_pairs.sh

PAIRS=("EURUSD" "GBPUSD" "USDJPY" "AUDUSD")

for pair in "${PAIRS[@]}"; do
    echo "Fetching $pair..."
    python3 scripts/fetch_forex_news.py $pair --limit 5 > "/tmp/${pair}_news.json"
done
```

## Troubleshooting

### No News Returned

If `news_count` is 0:
- The pair may not have recent news
- Yahoo Finance API may be experiencing issues
- Try again in a few minutes
- Check with a different pair to verify connectivity

### Rate Data is Null

If `current_rate` is null:
- Yahoo Finance may not have current data for this symbol
- Market may be closed (weekends, holidays)
- API may be temporarily unavailable

### Script Hangs

If the script doesn't return:
- Check internet connection
- Yahoo Finance API may be slow
- Kill the process and try again
- Reduce the limit parameter
