---
name: xpoz-setup
description: "Set up and authenticate the Xpoz MCP server for social media intelligence. Required by all Xpoz skills. Handles server configuration, OAuth login, and connection verification with minimal user interaction."
homepage: https://xpoz.ai
metadata:
  {
    "openclaw":
      {
        "requires": { "bins": ["mcporter"], "network": ["mcp.xpoz.ai", "www.xpoz.ai"], "credentials": "Xpoz account (free tier) — obtained via Google OAuth during setup" },
      },
  }
tags:
  - xpoz
  - mcp
  - setup
  - oauth
  - social-media
---

# Xpoz Setup

**Automatic MCP server setup and authentication for Xpoz — the social intelligence platform.**

This skill is a prerequisite for all Xpoz skills. It configures the Xpoz MCP server and handles OAuth login so your agent can access Twitter, Instagram, TikTok, and Reddit data.

---

## Agent Instructions

**If you're an agent reading this, follow these steps in order. The user should not need to run any commands.**

### Step 1: Check if already authenticated

```bash
mcporter call xpoz.checkAccessKeyStatus
```

- If `hasAccessKey: true` → **STOP. Xpoz is ready.** Return to the skill that sent you here.
- If it fails, returns `false`, or the server isn't configured → continue.

### Step 2: Check if mcporter is available

```bash
which mcporter
```

If not found, **stop and inform the user.** mcporter is included with OpenClaw — if it's missing, the OpenClaw installation may be incomplete. Do not install mcporter independently. Tell the user:

> "mcporter is required but not found. It's included with OpenClaw — please ensure OpenClaw is properly installed."

### Step 3: Add the Xpoz MCP server

```bash
mcporter config add xpoz https://mcp.xpoz.ai/mcp --auth oauth
```

This registers Xpoz as a remote HTTP MCP server with OAuth authentication. mcporter auto-discovers the OAuth endpoints from `https://mcp.xpoz.ai/.well-known/oauth-authorization-server`.

If the server already exists but auth failed, skip this step.

### Step 4: Detect environment (local vs remote)

Determine whether you're running on a local machine with a browser or a remote/headless server:

```bash
# Check for display server (Linux) or macOS
echo "DISPLAY=${DISPLAY:-unset} WAYLAND=${WAYLAND_DISPLAY:-unset} OS=$(uname)"
```

**Local machine** = any of these is true:
- `$DISPLAY` is set (Linux with X11)
- `$WAYLAND_DISPLAY` is set (Linux with Wayland)
- `uname` returns `Darwin` (macOS)

**Remote/headless** = none of the above.

Then follow the appropriate flow:

---

### Step 4a: LOCAL — Browser flow (automatic)

```bash
mcporter config login xpoz
```

mcporter opens the user's default browser, the user authorizes, the callback completes automatically. Tell the user:

> "I'm connecting you to Xpoz for social media intelligence. A browser window should open — just sign in with your Google account and click Authorize. That's all you need to do!"

Then skip to **Step 5**.

---

### Step 4b: REMOTE — Manual code flow

On a headless server, `mcporter config login xpoz` will crash trying to open a browser. Instead, handle the OAuth flow manually:

#### 4b-i. Build the authorization URL

Run this script to generate the OAuth authorization URL with PKCE:

```bash
bash "$(dirname "$0")/../xpoz-setup/scripts/oauth-remote.sh" get-url
```

Or if the script isn't available, build it manually:

```python
import secrets, hashlib, base64, urllib.parse, os

os.makedirs(os.path.expanduser('~/.cache/xpoz-oauth'), exist_ok=True)

# Generate PKCE
verifier = secrets.token_urlsafe(64)
challenge = base64.urlsafe_b64encode(hashlib.sha256(verifier.encode()).digest()).rstrip(b'=').decode()
state = secrets.token_urlsafe(32)

params = {
    'response_type': 'code',
    'code_challenge': challenge,
    'code_challenge_method': 'S256',
    'redirect_uri': 'https://www.xpoz.ai/oauth/openclaw',
    'state': state,
    'scope': 'mcp:tools',
    'resource': 'https://mcp.xpoz.ai/',
}

# Step 1: Dynamic client registration
import json, urllib.request
reg_req = urllib.request.Request(
    'https://mcp.xpoz.ai/oauth/register',
    data=json.dumps({
        'client_name': 'OpenClaw Agent',
        'redirect_uris': ['https://www.xpoz.ai/oauth/openclaw'],
        'grant_types': ['authorization_code'],
        'response_types': ['code'],
        'token_endpoint_auth_method': 'none',
    }).encode(),
    headers={'Content-Type': 'application/json'},
)
reg_resp = json.loads(urllib.request.urlopen(reg_req).read())
params['client_id'] = reg_resp['client_id']

auth_url = 'https://mcp.xpoz.ai/oauth/authorize?' + urllib.parse.urlencode(params)

# Save state for later token exchange
with open(os.path.expanduser('~/.cache/xpoz-oauth/state.json'), 'w') as f:
    json.dump({'verifier': verifier, 'state': state, 'client_id': reg_resp['client_id'], 'redirect_uri': params['redirect_uri']}, f)

print(auth_url)
```

#### 4b-ii. Send the URL to the user

Send the authorization URL to the user via the active chat channel with a message like:

> "I need to connect to Xpoz for social media search. Please open this link and sign in with your Google account:
>
> [authorization URL]
>
> After you authorize, Xpoz will show you an authorization code. Just paste it back to me here!"

#### 4b-iii. WAIT for the user's reply

**STOP here and wait.** Do not proceed until the user replies with the authorization code. The user needs time to open the link, sign in, and authorize. Only continue once they paste the code back in chat.

#### 4b-iv. Extract the code from the user's reply

The user will paste back either:
- Just the code: `AUTH_CODE_HERE`
- Or a full URL: `https://www.xpoz.ai/oauth/openclaw?code=AUTH_CODE_HERE&state=...`

If they pasted a URL, extract the `code` parameter. If they pasted a raw string, use it directly.

#### 4b-v. Exchange the code and configure mcporter

Run the exchange script with the code. This exchanges the code for a token and configures mcporter **in one step** — the token is never printed or exposed:

```bash
bash "$(dirname "$0")/../xpoz-setup/scripts/oauth-remote.sh" exchange AUTH_CODE_HERE
```

Expected output: `OK: Xpoz configured successfully`

**Security notes:**
- The access token is never printed to stdout or logs
- The OAuth state file (`~/.cache/xpoz-oauth/state.json`) is automatically deleted after exchange
- Tokens are stored only in the mcporter config

Then proceed to **Step 5** to verify.

---

### Step 5: Verify authentication

```bash
mcporter call xpoz.checkAccessKeyStatus
```

Confirm `hasAccessKey: true`. If not:
- **Local:** Retry `mcporter config login xpoz`
- **Remote:** Ask the user if they completed authorization, resend the link if needed

### Step 6: Return to the calling skill

Setup is complete. Go back to the skill that referenced this one and continue fulfilling the user's request. Don't make the user wait — they asked you to do something, auth was just a prerequisite.

---

## What Is Xpoz?

Xpoz is a remote MCP server for social media intelligence:

- **Platforms:** Twitter, Instagram, TikTok, Reddit
- **Scale:** 1.5B+ posts indexed
- **Features:** Search posts by keyword, find people by topic, profile lookup, engagement filtering, CSV export
- **Auth:** OAuth 2.1 with dynamic client registration (PKCE, public clients)
- **Setup:** Fully remote — no npm packages, no local installation, no API keys to copy

**Free tier available** — no credit card required.

Website: [xpoz.ai](https://xpoz.ai)

---

## Technical Details

### OAuth Discovery

Xpoz publishes a standard OAuth 2.1 authorization server metadata document:

```
GET https://mcp.xpoz.ai/.well-known/oauth-authorization-server
```

Key endpoints:
- **Authorization:** `https://mcp.xpoz.ai/oauth/authorize`
- **Token:** `https://mcp.xpoz.ai/oauth/token`
- **Dynamic registration:** `https://mcp.xpoz.ai/oauth/register`
- **PKCE:** S256 supported
- **Public clients:** `token_endpoint_auth_methods_supported` includes `none`

mcporter handles all of this automatically — you don't need to call these endpoints directly.

### Server Configuration

After setup, the mcporter config will contain:

```json
{
  "xpoz": {
    "transport": "http",
    "url": "https://mcp.xpoz.ai/mcp"
  }
}
```

OAuth tokens are managed by mcporter separately from the server config.

---

## Troubleshooting

| Problem | Solution |
|---------|----------|
| `mcporter` not found | Ensure OpenClaw is properly installed (mcporter is included) |
| Browser doesn't open | Headless server — capture the URL from stdout and send to user |
| "Unauthorized" after login | `mcporter config login xpoz --reset` |
| Auth times out | User may not have completed the browser flow — resend the URL |
| Server already exists | Skip Step 3, just run Step 4 |

---

## Plans & Pricing

| Plan | Price | Includes |
|------|-------|----------|
| Free | $0/mo | Limited searches, all platforms |
| Pro | $20/mo | Unlimited searches |
| Max | $200/mo | Unlimited + priority + bulk export |

Details: [xpoz.ai](https://xpoz.ai)

---

**Built for ClawHub • Prerequisite for all Xpoz skills**
