---
name: xlsx-pro
description: "Compétence pour manipuler les fichiers Excel (.xlsx, .xlsm, .csv, .tsv). Utiliser quand l'utilisateur veut : ouvrir, lire, éditer ou créer un fichier tableur ; ajouter des colonnes, calculer des formules, formater, créer des graphiques, nettoyer des données ; convertir entre formats tabulaires. Le livrable doit être un fichier tableur. NE PAS utiliser si le livrable est un document Word, HTML, script Python standalone, ou intégration Google Sheets."
version: "1.0.1"
author: "Eric Barotte"
---

# Compétence Excel pour OpenClawd

## TL;DR
- Génère/édite des fichiers Excel avec des **formules** (pas des valeurs hardcodées).
- Optionnel: **recalcul** via LibreOffice headless + détection d’erreurs Excel.
- Livrable attendu: un fichier tableur propre (XLSX/XLSM/CSV/TSV).


## Prérequis

### Dépendances Python
```bash
pip install openpyxl pandas xlrd xlwt
```

### LibreOffice (pour recalcul des formules)
```bash
# Ubuntu/Debian
sudo apt-get install libreoffice-calc libreoffice-common
```

## Règles de Qualité

### Police Professionnelle
- Utiliser une police cohérente (Arial, Times New Roman) sauf instruction contraire

### Zéro Erreur de Formule
- Tout fichier Excel DOIT être livré SANS erreurs (#REF!, #DIV/0!, #VALUE!, #N/A, #NAME?)

### Préservation des Templates
- Respecter EXACTEMENT le format et style existants lors de modifications
- Les conventions du template préexistant ont TOUJOURS priorité

## Standards pour Modèles Financiers

### Code Couleur (Standards Industrie)
- **Texte bleu (RGB: 0,0,255)** : Inputs hardcodés, valeurs modifiables
- **Texte noir (RGB: 0,0,0)** : TOUTES les formules et calculs
- **Texte vert (RGB: 0,128,0)** : Liens vers autres feuilles du même classeur
- **Texte rouge (RGB: 255,0,0)** : Liens externes vers autres fichiers
- **Fond jaune (RGB: 255,255,0)** : Hypothèses clés ou cellules à mettre à jour

### Formatage des Nombres
- **Années** : Format texte ("2024" pas "2,024")
- **Devises** : Format $#,##0 ; spécifier unités dans les en-têtes ("Revenue ($mm)")
- **Zéros** : Afficher comme "-" (format: "$#,##0;($#,##0);-")
- **Pourcentages** : Format 0.0% par défaut
- **Multiples** : Format 0.0x (EV/EBITDA, P/E)
- **Négatifs** : Parenthèses (123) pas moins -123

## CRITIQUE : Utiliser des Formules, PAS des Valeurs Hardcodées

**TOUJOURS utiliser des formules Excel au lieu de calculer en Python et hardcoder.**

### ❌ MAUVAIS - Hardcoding
```python
# Mauvais: Calcul Python puis hardcode
total = df['Sales'].sum()
sheet['B10'] = total  # Hardcode 5000

# Mauvais: Taux de croissance calculé en Python
growth = (df.iloc[-1]['Revenue'] - df.iloc[0]['Revenue']) / df.iloc[0]['Revenue']
sheet['C5'] = growth  # Hardcode 0.15
```

### ✅ CORRECT - Formules Excel
```python
# Bon: Laisser Excel calculer
sheet['B10'] = '=SUM(B2:B9)'

# Bon: Taux de croissance en formule Excel
sheet['C5'] = '=(C4-C2)/C2'

# Bon: Moyenne en fonction Excel
sheet['D20'] = '=AVERAGE(D2:D19)'
```

## Workflows

### Workflow Standard
1. **Choisir l'outil** : pandas pour données, openpyxl pour formules/formatage
2. **Créer/Charger** : Nouveau classeur ou fichier existant
3. **Modifier** : Données, formules, formatage
4. **Sauvegarder** : Écrire le fichier
5. **Recalculer (OBLIGATOIRE si formules)** : `python scripts/recalc.py output.xlsx`
6. **Vérifier et corriger** les erreurs détectées

### Lecture et Analyse avec pandas
```python
import pandas as pd

# Lire Excel
df = pd.read_excel('file.xlsx')  # Première feuille par défaut
all_sheets = pd.read_excel('file.xlsx', sheet_name=None)  # Dict de toutes les feuilles

# Analyser
df.head()      # Aperçu
df.info()      # Info colonnes
df.describe()  # Statistiques

# Écrire
df.to_excel('output.xlsx', index=False)
```

### Création de Fichiers Excel
```python
from openpyxl import Workbook
from openpyxl.styles import Font, PatternFill, Alignment

wb = Workbook()
sheet = wb.active

# Données
sheet['A1'] = 'Hello'
sheet['B1'] = 'World'
sheet.append(['Row', 'of', 'data'])

# Formule
sheet['B2'] = '=SUM(A1:A10)'

# Formatage
sheet['A1'].font = Font(bold=True, color='FF0000')
sheet['A1'].fill = PatternFill('solid', start_color='FFFF00')
sheet['A1'].alignment = Alignment(horizontal='center')

# Largeur colonne
sheet.column_dimensions['A'].width = 20

wb.save('output.xlsx')
```

### Édition de Fichiers Existants
```python
from openpyxl import load_workbook

# Charger fichier existant
wb = load_workbook('existing.xlsx')
sheet = wb.active  # ou wb['NomFeuille']

# Parcourir les feuilles
for sheet_name in wb.sheetnames:
    sheet = wb[sheet_name]
    print(f"Feuille: {sheet_name}")

# Modifier
sheet['A1'] = 'Nouvelle Valeur'
sheet.insert_rows(2)  # Insérer ligne
sheet.delete_cols(3)  # Supprimer colonne

# Ajouter feuille
new_sheet = wb.create_sheet('NouvelleFeuille')
new_sheet['A1'] = 'Data'

wb.save('modified.xlsx')
```

## Recalcul des Formules

Les fichiers créés par openpyxl contiennent les formules comme chaînes mais pas les valeurs calculées. Utiliser le script `recalc.py` :

```bash
python scripts/recalc.py <fichier_excel> [timeout_secondes]
```

Le script :
- Configure automatiquement la macro LibreOffice au premier lancement
- Recalcule toutes les formules
- Scanne TOUTES les cellules pour erreurs Excel
- Retourne JSON avec détails et emplacements des erreurs

### Interprétation de la Sortie
```json
{
  "status": "success",           // ou "errors_found"
  "total_errors": 0,             // Nombre total d'erreurs
  "total_formulas": 42,          // Nombre de formules
  "error_summary": {             // Présent si erreurs
    "#REF!": {
      "count": 2,
      "locations": ["Sheet1!B5", "Sheet1!C10"]
    }
  }
}
```

## Checklist de Vérification

### Vérifications Essentielles
- [ ] **Tester 2-3 références** : Vérifier qu'elles tirent les bonnes valeurs
- [ ] **Mapping colonnes** : Confirmer correspondance (colonne 64 = BL, pas BK)
- [ ] **Offset lignes** : Excel est 1-indexé (DataFrame row 5 = Excel row 6)

### Pièges Courants
- [ ] **Gestion NaN** : Vérifier valeurs nulles avec `pd.notna()`
- [ ] **Colonnes éloignées** : Données FY souvent en colonnes 50+
- [ ] **Correspondances multiples** : Chercher toutes les occurrences
- [ ] **Division par zéro** : Vérifier dénominateurs (#DIV/0!)
- [ ] **Références invalides** : Vérifier que toutes pointent vers cellules existantes (#REF!)
- [ ] **Références inter-feuilles** : Format correct (Sheet1!A1)

## Bonnes Pratiques

### Sélection de Bibliothèque
- **pandas** : Analyse de données, opérations en masse, export simple
- **openpyxl** : Formatage complexe, formules, fonctionnalités Excel spécifiques

### Avec openpyxl
- Indices de cellules en base 1 (row=1, column=1 = cellule A1)
- `data_only=True` pour lire valeurs calculées
- **Attention** : Sauvegarder après `data_only=True` remplace définitivement les formules par les valeurs
- Pour gros fichiers : `read_only=True` ou `write_only=True`

### Avec pandas
- Spécifier types de données : `pd.read_excel('file.xlsx', dtype={'id': str})`
- Pour gros fichiers, colonnes spécifiques : `usecols=['A', 'C', 'E']`
- Gestion des dates : `parse_dates=['date_column']`

## Style de Code

**IMPORTANT** : Code Python minimal et concis, sans commentaires superflus.

**Pour les fichiers Excel** :
- Commenter les cellules avec formules complexes
- Documenter les sources des données hardcodées
- Inclure notes pour calculs clés
