#!/usr/bin/env node
/**
 * analyze.mjs
 *
 * Analyze X content using xAI Grok:
 * - voice: Analyze account voice patterns
 * - trends: Research topic trends and angles
 * - post: Check post for AI signals and platform flags
 *
 * Examples:
 *   node {baseDir}/scripts/analyze.mjs voice @username
 *   node {baseDir}/scripts/analyze.mjs trends "AI agents"
 *   node {baseDir}/scripts/analyze.mjs post "Your draft post text"
 *   node {baseDir}/scripts/analyze.mjs post --url https://x.com/user/status/123
 */

import fs from "node:fs";
import os from "node:os";
import path from "node:path";

function usage(msg) {
  if (msg) console.error(msg);
  console.error(
    "Usage: analyze.mjs (voice|trends|post) <query> [--json] [--raw] [--model <id>] [--url <url>] [--days <n>]"
  );
  process.exit(2);
}

function readKeyFromClawdbotConfig() {
  try {
    const p = path.join(os.homedir(), ".clawdbot", "clawdbot.json");
    const raw = fs.readFileSync(p, "utf8");
    const j = JSON.parse(raw);

    return (
      process.env.XAI_API_KEY ||
      j?.env?.XAI_API_KEY ||
      j?.env?.vars?.XAI_API_KEY ||
      j?.skills?.entries?.["xai-plus"]?.apiKey ||
      j?.skills?.entries?.xai?.apiKey ||
      j?.skills?.entries?.["grok-search"]?.apiKey ||
      j?.skills?.entries?.["search-x"]?.apiKey ||
      null
    );
  } catch {
    return process.env.XAI_API_KEY || null;
  }
}

function readModelFromClawdbotConfig() {
  try {
    const p = path.join(os.homedir(), ".clawdbot", "clawdbot.json");
    const raw = fs.readFileSync(p, "utf8");
    const j = JSON.parse(raw);

    return (
      process.env.XAI_MODEL ||
      j?.env?.XAI_MODEL ||
      j?.env?.vars?.XAI_MODEL ||
      j?.skills?.entries?.["xai-plus"]?.model ||
      j?.skills?.entries?.xai?.model ||
      null
    );
  } catch {
    return process.env.XAI_MODEL || null;
  }
}

const args = process.argv.slice(2);
if (args.length < 2) usage();

const mode = args[0];
if (!["voice", "trends", "post"].includes(mode)) {
  usage(`Invalid mode: ${mode}`);
}

let queryParts = [];
let jsonOut = false;
let rawOut = false;
let model = readModelFromClawdbotConfig() || "grok-4-1-fast";
let days = 30;
let postUrl = null;

for (let i = 1; i < args.length; i++) {
  const a = args[i];
  if (a === "--json") jsonOut = true;
  else if (a === "--raw") rawOut = true;
  else if (a === "--model") {
    const v = args[++i];
    if (!v) usage("Missing value for --model");
    model = v;
  } else if (a === "--days") {
    const v = Number(args[++i]);
    if (!Number.isFinite(v) || v <= 0) usage("Bad value for --days");
    days = Math.floor(v);
  } else if (a === "--url") {
    const v = args[++i];
    if (!v) usage("Missing value for --url");
    postUrl = v;
  } else if (a.startsWith("-")) {
    usage(`Unknown flag: ${a}`);
  } else {
    queryParts.push(a);
  }
}

const query = queryParts.join(" ").trim();
if (!query && mode !== "post") usage("Missing query");
if (mode === "post" && !query && !postUrl) {
  usage("Post mode requires either text or --url");
}

const apiKey = readKeyFromClawdbotConfig();
if (!apiKey) {
  console.error(
    "Missing XAI_API_KEY. Set env var or add to ~/.clawdbot/clawdbot.json"
  );
  process.exit(1);
}

function isoDate(d) {
  const year = d.getFullYear();
  const month = String(d.getMonth() + 1).padStart(2, "0");
  const day = String(d.getDate()).padStart(2, "0");
  return `${year}-${month}-${day}`;
}

function computeDateRange(daysAgo) {
  const to = new Date();
  const from = new Date();
  from.setDate(from.getDate() - daysAgo);
  return { from_date: isoDate(from), to_date: isoDate(to) };
}

let prompt;
let tools = [];

if (mode === "voice") {
  const handle = query.replace(/^@/, "");
  const dateRange = computeDateRange(days);

  prompt = `Analyze the voice and writing patterns of @${handle} on X.

Use x_search to find their recent posts (last ${days} days).

Return ONLY valid JSON (no markdown) with this schema:
{
  "handle": "@${handle}",
  "analyzed_posts": number,
  "voice": {
    "tone": string,
    "personality": [string],
    "perspective": string,
    "energy_level": "high|medium|low"
  },
  "patterns": {
    "sentence_structure": [string],
    "vocabulary": [string],
    "formatting_quirks": [string],
    "recurring_phrases": [string]
  },
  "topics": [string],
  "best_posts": [
    {
      "url": string,
      "text": string,
      "why": string
    }
  ],
  "anti_patterns": [string]
}`;

  tools = [
    {
      type: "x_search",
      x_search: {
        from_date: dateRange.from_date,
        to_date: dateRange.to_date,
        allowed_x_handles: [handle],
      },
    },
  ];
} else if (mode === "trends") {
  const dateRange = computeDateRange(7);

  prompt = `Research trends and discussions about: ${query}

Use x_search to find recent conversations (last 7 days).

Return ONLY valid JSON (no markdown) with this schema:
{
  "topic": "${query}",
  "trends": [
    {
      "pattern": string,
      "description": string,
      "example_posts": [string]
    }
  ],
  "perspectives": [
    {
      "viewpoint": string,
      "supporters": [string]
    }
  ],
  "hashtags": [string],
  "key_accounts": [string],
  "posting_angles": [
    {
      "angle": string,
      "hook": string,
      "target_audience": string
    }
  ]
}`;

  tools = [
    {
      type: "x_search",
      x_search: {
        from_date: dateRange.from_date,
        to_date: dateRange.to_date,
      },
    },
  ];
} else if (mode === "post") {
  let postText = query;

  if (postUrl) {
    prompt = `Analyze this X post for quality and safety: ${postUrl}

Use x_search to fetch the post content, then check for:

AI DETECTION SIGNALS (score 0-10):
- Em-dashes (—) in casual posts (+2-3)
- Numbered lists in non-thread content (+2-3)
- Engagement bait closers: "What do you think?", "Would love your thoughts" (+2-3)
- Template phrases: "It's not X — it's Y", "Here's the thing:", "Let me be clear:" (+1-2)
- Perfect grammar throughout, no typos or fragments (+1-2)
- Formal vocabulary: "Certainly", "Indeed", "Furthermore" (+1 each)
- Hollow depth: sounds profound but says nothing specific (+1-2)
- Generic insights with no personal details or examples (+1-2)
- Missing human signals: no mess, no asides, no self-corrections (+1)

PLATFORM FLAG PATTERNS (score 0-10):
- Repetitive patterns across posts (high risk)
- Generic engagement bait questions (medium-high risk)
- Low-effort replies: "This!", "Great!", "So true!" (medium risk)
- Promotional spam or "DM me" (high risk)
- Template structures reused (high risk)

QUALITY SIGNALS (score 0-10):
- Specific details (names, dates, numbers, examples) (+)
- Authentic voice and genuine expression (+)
- Dense value worth re-reading (dwell time optimization) (+)
- Quotable standalone lines (+)

Return ONLY valid JSON (no markdown) with this schema:
{
  "post_url": "${postUrl}",
  "post_text": string,
  "ai_detection_score": number (0-10, 10=definitely AI),
  "ai_signals": [string],
  "platform_flag_score": number (0-10, 10=high risk),
  "platform_risks": [string],
  "quality_score": number (0-10, 10=excellent),
  "suggestions": [string]
}`;

    tools = [{ type: "x_search" }];
  } else {
    prompt = `Analyze this post draft for AI signals and platform flag patterns:

"${postText}"

AI DETECTION SIGNALS (score 0-10):
- Em-dashes (—) in casual posts (+2-3)
- Numbered lists in non-thread content (+2-3)
- Engagement bait closers: "What do you think?", "Would love your thoughts" (+2-3)
- Template phrases: "It's not X — it's Y", "Here's the thing:", "Let me be clear:" (+1-2)
- Perfect grammar throughout, no typos or fragments (+1-2)
- Formal vocabulary: "Certainly", "Indeed", "Furthermore" (+1 each)
- Hollow depth: sounds profound but says nothing specific (+1-2)
- Generic insights with no personal details or examples (+1-2)
- Missing human signals: no mess, no asides, no self-corrections (+1)

PLATFORM FLAG PATTERNS (score 0-10):
- Repetitive patterns that appear in multiple posts (high risk)
- Generic engagement bait questions (medium-high risk)
- Low-effort replies: "This!", "Great!", "So true!" (medium risk)
- Promotional spam or "DM me" (high risk)
- Template structures (high risk)

QUALITY SIGNALS (score 0-10):
- Specific details (names, dates, numbers, examples) (+)
- Authentic voice and genuine expression (+)
- Dense value worth re-reading (dwell time optimization) (+)
- Quotable standalone lines (+)

Return ONLY valid JSON (no markdown) with this schema:
{
  "post_text": "${postText.replace(/"/g, '\\"')}",
  "ai_detection_score": number (0-10, 10=definitely AI),
  "ai_signals": [string],
  "platform_flag_score": number (0-10, 10=high risk),
  "platform_risks": [string],
  "quality_score": number (0-10, 10=excellent),
  "suggestions": [string]
}`;
  }
}

const body = {
  model,
  input: [
    {
      role: "user",
      content: [{ type: "input_text", text: prompt }],
    },
  ],
  ...(tools.length ? { tools } : {}),
  store: false,
  temperature: 0,
};

const res = await fetch("https://api.x.ai/v1/responses", {
  method: "POST",
  headers: {
    "Content-Type": "application/json",
    Authorization: `Bearer ${apiKey}`,
  },
  body: JSON.stringify(body),
});

if (!res.ok) {
  const t = await res.text().catch(() => "");
  console.error(`xAI API error: ${res.status} ${res.statusText}`);
  console.error(t.slice(0, 4000));
  process.exit(1);
}

const data = await res.json();

const text =
  data.output_text ||
  data?.output
    ?.flatMap((o) => (Array.isArray(o?.content) ? o.content : []))
    ?.find((c) => c?.type === "output_text" && typeof c?.text === "string")
    ?.text ||
  null;

if (!text) {
  if (jsonOut) {
    console.log(JSON.stringify({ mode, query, raw: data }, null, 2));
  } else {
    console.log(JSON.stringify(data, null, 2));
  }
  process.exit(0);
}

let parsed;
try {
  parsed = JSON.parse(text);
} catch {
  parsed = null;
}

if (jsonOut || !parsed) {
  console.log(parsed ? JSON.stringify(parsed, null, 2) : text.trim());
  if (rawOut) {
    console.error("\n--- RAW RESPONSE (debug) ---\n");
    console.error(JSON.stringify(data, null, 2));
  }
  process.exit(0);
}

if (mode === "voice") {
  console.log(`Voice Analysis: ${parsed.handle}`);
  console.log(`Analyzed ${parsed.analyzed_posts} posts\n`);
  console.log("Voice:");
  console.log(`  Tone: ${parsed.voice?.tone}`);
  console.log(`  Personality: ${parsed.voice?.personality?.join(", ")}`);
  console.log(`  Perspective: ${parsed.voice?.perspective}`);
  console.log(`  Energy: ${parsed.voice?.energy_level}\n`);
  console.log("Top Topics:");
  for (const t of parsed.topics || []) console.log(`  - ${t}`);
  console.log("\nBest Posts:");
  for (const p of parsed.best_posts || []) {
    console.log(`  - ${p.url}`);
    console.log(`    "${p.text}"`);
    console.log(`    Why: ${p.why}`);
  }
} else if (mode === "trends") {
  console.log(`Trend Research: ${parsed.topic}\n`);
  console.log("Trends:");
  for (const t of parsed.trends || []) {
    console.log(`  - ${t.pattern}`);
    console.log(`    ${t.description}`);
  }
  console.log("\nPosting Angles:");
  for (const a of parsed.posting_angles || []) {
    console.log(`  - ${a.angle}`);
    console.log(`    Hook: ${a.hook}`);
    console.log(`    Audience: ${a.target_audience}`);
  }
  console.log("\nKey Hashtags:", parsed.hashtags?.join(", "));
  console.log("Key Accounts:", parsed.key_accounts?.join(", "));
} else if (mode === "post") {
  console.log("Post Safety Check\n");
  console.log(`AI Detection: ${parsed.ai_detection_score}/10`);
  if (parsed.ai_signals?.length) {
    console.log("Signals:");
    for (const s of parsed.ai_signals) console.log(`  - ${s}`);
  }
  console.log(`\nPlatform Flag Risk: ${parsed.platform_flag_score}/10`);
  if (parsed.platform_risks?.length) {
    console.log("Risks:");
    for (const r of parsed.platform_risks) console.log(`  - ${r}`);
  }
  console.log(`\nQuality Score: ${parsed.quality_score}/10`);
  if (parsed.suggestions?.length) {
    console.log("Suggestions:");
    for (const s of parsed.suggestions) console.log(`  - ${s}`);
  }
}

if (rawOut) {
  console.error("\n--- RAW RESPONSE (debug) ---\n");
  console.error(JSON.stringify(data, null, 2));
}
