# Search Patterns and Optimization

Query patterns, filters, and optimization tips for X and web search.

## X Search Patterns

### Basic Queries

**Single keyword:**
```bash
node scripts/grok_search.mjs "TypeScript" --x
```

**Multi-word phrase:**
```bash
node scripts/grok_search.mjs "AI agents deployment" --x
```

**Account mentions:**
```bash
node scripts/grok_search.mjs "@anthropic Claude" --x
```

### Date Filtering

**Recent posts (relative):**
```bash
# Last 7 days
node scripts/grok_search.mjs "product launch" --x --days 7

# Last 24 hours
node scripts/grok_search.mjs "breaking news" --x --days 1

# Last 30 days
node scripts/grok_search.mjs "trend analysis" --x --days 30
```

**Specific date range:**
```bash
node scripts/grok_search.mjs "conference" --x \
  --from 2026-01-15 --to 2026-01-20
```

**Date range best practices:**
- Recent data (1-7 days) for trending topics
- 30 days for pattern analysis
- Longer ranges may return less relevant results
- xAI's index freshness varies

### Handle Filtering

**Single account:**
```bash
node scripts/grok_search.mjs "launch" --x --handles AnthropicAI
```

**Multiple accounts:**
```bash
node scripts/grok_search.mjs "AI news" --x \
  --handles AnthropicAI,OpenAI,GoogleAI
```

**Exclude accounts:**
```bash
node scripts/grok_search.mjs "crypto" --x \
  --exclude spam_account1,spam_account2
```

**Combined filtering:**
```bash
node scripts/grok_search.mjs "startup advice" --x \
  --days 14 \
  --handles founder1,founder2,founder3 \
  --exclude promo_bot
```

### Query Optimization

**Specific beats generic:**
```bash
# ✓ Good
"Claude Sonnet 4.5 coding performance"

# ✗ Too generic
"AI"
```

**Include context:**
```bash
# ✓ Good
"TypeScript 5.7 new features"

# ✗ Missing context
"TypeScript features"
```

**Technical terms:**
```bash
# Technical audience
"transformer architecture attention mechanism"

# General audience
"how AI models work"
```

**Brand/product names:**
```bash
# Specific
"Claude Code CLI features"

# Generic
"AI coding tools"
```

### Topic Discovery

**Find experts:**
```bash
node scripts/grok_search.mjs "machine learning" --x --days 30
# Analyze top accounts from results
```

**Track hashtags:**
```bash
node scripts/grok_search.mjs "#BuildInPublic progress" --x --days 7
```

**Monitor conversations:**
```bash
node scripts/grok_search.mjs "your-brand OR your-product" --x --days 1
```

### Thread Discovery

Search often returns individual posts from threads. The URL will link to the post; navigate to view the full thread.

**Thread indicators in results:**
- Multiple posts from same author
- Sequential post timestamps
- "Thread:" or numbered posts in snippets

## Web Search Patterns

### General Queries

**Broad research:**
```bash
node scripts/grok_search.mjs "quantum computing basics" --web
```

**Current events:**
```bash
node scripts/grok_search.mjs "tech news January 2026" --web
```

**Technical documentation:**
```bash
node scripts/grok_search.mjs "FastAPI async patterns" --web
```

### Query Refinement

**Be specific:**
```bash
# ✓ Good
"Next.js 15 server actions best practices"

# ✗ Too broad
"Next.js tips"
```

**Include year for current info:**
```bash
"Rust async 2026"
"React hooks 2026"
```

**Add context:**
```bash
"Docker multi-stage builds production optimization"
```

### Use Cases

**Technical research:**
```bash
node scripts/grok_search.mjs "SQLite performance tuning large datasets" --web
```

**Best practices:**
```bash
node scripts/grok_search.mjs "TypeScript monorepo setup 2026" --web
```

**Comparisons:**
```bash
node scripts/grok_search.mjs "Bun vs Node.js performance benchmarks" --web
```

**Documentation:**
```bash
node scripts/grok_search.mjs "Anthropic Claude API streaming" --web
```

## Combined X + Web Research

Use both search modes for comprehensive research:

```bash
# 1. Check recent X discussions
node scripts/grok_search.mjs "new framework" --x --days 7

# 2. Get web documentation
node scripts/grok_search.mjs "new framework official docs" --web

# 3. Find expert opinions on X
node scripts/grok_search.mjs "new framework" --x \
  --handles expert1,expert2,expert3
```

## Output Handling

### JSON Schema

All searches return consistent JSON:

```json
{
  "query": string,
  "mode": "x" | "web",
  "results": [
    {
      "title": string | null,
      "url": string | null,
      "snippet": string | null,
      "author": string | null,      // X only
      "posted_at": string | null    // X only (ISO 8601)
    }
  ],
  "citations": [string]  // Unique URLs
}
```

### Result Limits

Default max results: 8

Override with `--max`:
```bash
node scripts/grok_search.mjs "query" --x --max 20
```

Citations cap: `max(12, results * 3)`

### Citation Quality

**X search citations:**
- Deduped by tweet ID
- Prefer `/@handle/status/id` over `/i/status/id`
- Include tweet URLs when available
- Order: result URLs first, then additional citations

**Web search citations:**
- Unique URLs from search results
- May include related pages
- Order matches relevance

## Advanced Patterns

### Monitoring

Track topics over time:

```bash
# Daily check
node scripts/grok_search.mjs "your-topic" --x --days 1

# Weekly digest
node scripts/grok_search.mjs "your-topic" --x --days 7
```

### Competitor Analysis

```bash
node scripts/grok_search.mjs "competitor-name" --x --days 30 \
  | jq '.results[] | {author, snippet}'
```

### Content Research

Before writing about a topic:

```bash
# 1. Recent discussions
node scripts/grok_search.mjs "topic" --x --days 7

# 2. Key voices
node scripts/grok_search.mjs "topic" --x \
  --handles expert1,expert2,expert3

# 3. Background research
node scripts/grok_search.mjs "topic overview 2026" --web
```

### Sentiment Analysis

Search for different perspectives:

```bash
# Positive
node scripts/grok_search.mjs "product love OR amazing OR game-changer" --x

# Negative
node scripts/grok_search.mjs "product disappointed OR broken OR bad" --x
```

## Troubleshooting

### No Results

**Possible causes:**
- Query too specific
- Date range too narrow
- Account handles misspelled or private
- Topic has no recent activity

**Solutions:**
- Broaden query terms
- Expand date range
- Remove handle filters
- Try related keywords

### Irrelevant Results

**Possible causes:**
- Query too generic
- Missing context
- Broad date range

**Solutions:**
- Add specific keywords
- Include brand/product names
- Narrow date range
- Use handle filters

### Missing Expected Posts

**X search limitations:**
- xAI's index may lag real-time
- Private/protected accounts excluded
- Deleted posts not available
- Very old posts may not be indexed

### Rate Limits

If hitting rate limits:
- Increase delay between requests
- Reduce max results per query
- Cache results locally
- Upgrade xAI plan

## Safe Posting Volumes

When creating content based on research, stay within platform safety limits:

**Daily Limits:**
- Original posts: 5-10 max
- Total posts + replies: 15-20 max
- Replies per hour: 3-5 max
- Self-promo posts: 1-2 max (10% ratio)

**Timing:**
- Space original posts 2-4 hours apart (author diversity penalty)
- Avoid burst activity (10+ in 30 min triggers spam detection)
- Don't reply to same thread multiple times quickly

**Reply Strategy:**
- Prioritize quality over quantity (1 reply loop = 75x value vs 1 like = 0.5x)
- Each reply must add genuine value
- Space replies across different threads
- Aim for reply loops (back-and-forth) when possible

**What Kills Reach:**
- Blocks: -148x (undoes ~148 likes worth of reach)
- "Not interested" clicks: negative penalty
- Repetitive patterns: spam filter trigger
- Volume violations: shadowban risk

## Best Practices

### Query Design

1. **Start broad, then narrow:**
   ```bash
   # First pass
   node scripts/grok_search.mjs "AI agents" --x --days 30

   # Refined
   node scripts/grok_search.mjs "AI agents architecture patterns" --x --days 7
   ```

2. **Use specific terminology:**
   - Technical terms for expert audiences
   - Plain language for general topics
   - Brand names for product discussions

3. **Include temporal context:**
   - Add year for current info
   - Use "latest" or "new" for recent updates
   - Specify version numbers for software

### Filter Usage

1. **Date filtering:**
   - Default to 7-30 days for most queries
   - 1-3 days for breaking news
   - Avoid very long ranges (90+ days)

2. **Handle filtering:**
   - Use for focused research
   - Combine 3-5 key accounts
   - Exclude known spam/low-quality

3. **Result limits:**
   - Default (8) for quick checks
   - 15-20 for thorough research
   - Higher counts increase API time

### Performance

1. **JSON output** (default) is fastest for programmatic use
2. **Text output** adds formatting overhead
3. **Raw output** doubles data transfer
4. Larger result sets increase processing time

### Integration

**With analysis tools:**
```bash
# Search + analyze
node scripts/grok_search.mjs "topic" --x --days 7 > results.json
node scripts/analyze.mjs trends "topic" --json > trends.json
```

**With chat:**
```bash
# Search then discuss
node scripts/grok_search.mjs "topic" --x | \
  jq -r '.citations[]' | head -5 | \
  xargs -I {} node scripts/chat.mjs "Summarize: {}"
```

**Pipeline processing:**
```bash
# Extract and process citations
node scripts/grok_search.mjs "topic" --x --links-only | \
  while read url; do
    echo "Processing $url"
    # Your processing here
  done
```
