# xAI Models Reference

Comprehensive guide to xAI models, capabilities, and selection.

## Model Overview

| Model | Speed | Quality | Vision | Context | Best For |
|-------|-------|---------|--------|---------|----------|
| grok-4-1-fast | Fast | Good | No | Large | Default choice |
| grok-4-fast | Fast | Good | No | Large | Alternative fast |
| grok-3 | Slower | Best | No | Large | Complex reasoning |
| grok-2-vision-1212 | Medium | Good | Yes | Large | Image analysis |

## Model Details

### grok-4-1-fast

**Capabilities:**
- Text generation
- Search (X + web)
- Analysis
- Chat

**Characteristics:**
- Fast response time
- Good quality for most tasks
- Reliable for structured output
- Handles JSON well

**Best for:**
- Search queries (default)
- Chat conversations
- Quick analysis
- Routine tasks
- Cost optimization

**Not ideal for:**
- Very complex reasoning
- Tasks requiring highest quality
- Image analysis

**Example usage:**
```bash
# Default for search
node scripts/grok_search.mjs "query" --x

# Default for chat
node scripts/chat.mjs "prompt"

# Default for analysis
node scripts/analyze.mjs voice @username
```

### grok-4-fast

**Capabilities:**
- Text generation
- Search (X + web)
- Analysis
- Chat

**Characteristics:**
- Similar to grok-4-1-fast
- Alternative fast model
- Good balance of speed/quality

**Best for:**
- Same use cases as grok-4-1-fast
- A/B testing model performance
- Fallback if grok-4-1-fast unavailable

**Not ideal for:**
- Very complex reasoning
- Tasks requiring highest quality
- Image analysis

**Example usage:**
```bash
node scripts/grok_search.mjs "query" --x --model grok-4-fast
```

### grok-3

**Capabilities:**
- Text generation
- Search (X + web)
- Analysis
- Chat
- Advanced reasoning

**Characteristics:**
- Slower response time
- Highest quality output
- Better at nuanced analysis
- More reliable for complex tasks

**Best for:**
- Complex voice analysis
- Detailed trend research
- Nuanced content analysis
- Multi-step reasoning
- When quality > speed

**Not ideal for:**
- Quick lookups
- Simple queries
- Real-time interactions
- Image analysis

**Example usage:**
```bash
# Deep voice analysis
node scripts/analyze.mjs voice @username --model grok-3 --days 90

# Complex trend research
node scripts/analyze.mjs trends "topic" --model grok-3

# Detailed reasoning
node scripts/chat.mjs --model grok-3 "Complex prompt requiring reasoning"
```

### grok-2-vision-1212

**Capabilities:**
- Text generation
- Image analysis
- Multi-modal chat
- Visual content description

**Characteristics:**
- Supports image inputs
- Medium speed (slower than grok-4)
- Good quality text and vision
- Handles multiple images

**Best for:**
- Screenshot analysis
- Image description
- Visual content review
- Diagram explanation
- UI/UX feedback

**Not ideal for:**
- Text-only tasks (use faster models)
- Complex reasoning without images
- Very large image batches

**Supported formats:**
- JPEG/JPG
- PNG
- WebP
- GIF

**Example usage:**
```bash
# Single image
node scripts/chat.mjs --model grok-2-vision-1212 \
  --image screenshot.png \
  "What's in this image?"

# Multiple images
node scripts/chat.mjs --model grok-2-vision-1212 \
  --image before.png --image after.png \
  "Compare these designs"

# Screenshot + analysis
node scripts/chat.mjs --model grok-2-vision-1212 \
  --image code.png \
  "Review this code for issues"
```

## Model Selection Guide

### By Task Type

**Search (X/web):**
- Default: `grok-4-1-fast`
- High quality: `grok-3`

**Chat:**
- Quick questions: `grok-4-1-fast`
- Deep discussion: `grok-3`
- With images: `grok-2-vision-1212`

**Analysis:**
- Voice patterns: `grok-4-1-fast` (basic), `grok-3` (deep)
- Trend research: `grok-4-1-fast` (overview), `grok-3` (detailed)
- Post safety: `grok-4-1-fast` (sufficient)

### By Priority

**Speed priority:**
1. `grok-4-1-fast`
2. `grok-4-fast`
3. `grok-2-vision-1212` (if vision needed)
4. `grok-3`

**Quality priority:**
1. `grok-3`
2. `grok-4-1-fast`
3. `grok-4-fast`
4. `grok-2-vision-1212`

**Vision required:**
- Only option: `grok-2-vision-1212`

### By Use Case

**Content research:**
```bash
# Quick overview
node scripts/grok_search.mjs "topic" --x --model grok-4-1-fast

# Deep dive
node scripts/analyze.mjs trends "topic" --model grok-3
```

**Voice analysis:**
```bash
# Quick check (30 days)
node scripts/analyze.mjs voice @user --model grok-4-1-fast

# Comprehensive (90 days)
node scripts/analyze.mjs voice @user --days 90 --model grok-3
```

**Technical Q&A:**
```bash
# Simple question
node scripts/chat.mjs --model grok-4-1-fast "What is X?"

# Complex explanation
node scripts/chat.mjs --model grok-3 "Explain X architecture in detail"
```

**Visual content:**
```bash
# Always use vision model
node scripts/chat.mjs --model grok-2-vision-1212 \
  --image diagram.png "Explain this architecture"
```

## Performance Characteristics

### Response Time

Approximate response times (varies by query complexity):

| Model | Simple Query | Complex Query | With Tools |
|-------|--------------|---------------|------------|
| grok-4-1-fast | 1-2s | 3-5s | 5-10s |
| grok-4-fast | 1-2s | 3-5s | 5-10s |
| grok-3 | 3-5s | 8-15s | 15-30s |
| grok-2-vision-1212 | 2-4s | 6-10s | 10-20s |

**Note:** Tool usage (x_search, web_search) adds significant time regardless of model.

### Output Quality

Quality metrics (subjective assessment):

| Model | Accuracy | Nuance | Reliability | Structured Output |
|-------|----------|--------|-------------|-------------------|
| grok-4-1-fast | Good | Good | High | Excellent |
| grok-4-fast | Good | Good | High | Excellent |
| grok-3 | Excellent | Excellent | Very High | Excellent |
| grok-2-vision-1212 | Good | Good | High | Good |

### Cost Considerations

xAI pricing varies by model (check current pricing at console.x.ai):

**Relative cost (approximate):**
- grok-4-1-fast: 1x (baseline)
- grok-4-fast: ~1x
- grok-3: ~2-3x
- grok-2-vision-1212: ~1.5-2x

**Cost optimization:**
- Use fast models for routine tasks
- Reserve grok-3 for complex analysis
- Batch similar queries when possible
- Cache results to avoid repeat calls

## Model Capabilities

### Structured Output

All models support JSON output with proper prompting:

```javascript
// Request
{
  "model": "grok-4-1-fast",
  "input": [{
    "role": "user",
    "content": [{
      "type": "input_text",
      "text": "Return ONLY valid JSON: {\"key\": \"value\"}"
    }]
  }],
  "temperature": 0
}
```

**JSON reliability:**
- grok-4-1-fast: Excellent
- grok-4-fast: Excellent
- grok-3: Excellent
- grok-2-vision-1212: Good

### Tool Usage

All models support tools (x_search, web_search):

```javascript
{
  "model": "grok-4-1-fast",
  "tools": [
    {
      "type": "x_search",
      "x_search": {
        "from_date": "2026-01-01",
        "to_date": "2026-01-31"
      }
    }
  ]
}
```

**Tool performance:**
- Fast models (grok-4): Good for most searches
- grok-3: Better at synthesizing search results
- Vision model: Can use tools, but slower

### Context Length

All models have large context windows:

- Input: Thousands of tokens
- Output: Thousands of tokens
- Exact limits not publicly specified

**Practical implications:**
- Voice analysis: Can process 50-100+ posts
- Trend research: Can synthesize many search results
- Chat: Can maintain long conversations

### Temperature Control

All models support temperature parameter:

```javascript
{
  "temperature": 0.0  // 0 = deterministic, 2 = creative
}
```

**Recommendations:**
- Search/analysis: `0` (deterministic)
- Chat: `0.3-0.7` (balanced)
- Creative writing: `1.0+` (varied)

## Model Updates

xAI regularly updates models. Check for new versions:

```bash
node scripts/models.mjs
```

**Naming pattern:**
- `grok-[version]-[variant]`
- `grok-[version]-vision-[date]`

**Tracking changes:**
- Monitor xAI announcements
- Test new models before switching defaults
- Keep fallback to stable versions

## Troubleshooting

### Model Not Found

If a model is unavailable:

```json
{
  "error": {
    "message": "Model not found",
    "type": "invalid_request_error"
  }
}
```

**Solutions:**
- Check available models: `node scripts/models.mjs`
- Verify model ID spelling
- Update to newer model version
- Fall back to `grok-4-1-fast`

### Poor Quality Output

If output quality is insufficient:

1. **Try grok-3:**
   ```bash
   node scripts/analyze.mjs voice @user --model grok-3
   ```

2. **Improve prompt:**
   - Add more context
   - Request structured format
   - Specify output requirements

3. **Adjust temperature:**
   - Lower for deterministic tasks
   - Higher for creative tasks

### Slow Response

If responses are too slow:

1. **Use faster model:**
   ```bash
   --model grok-4-1-fast
   ```

2. **Reduce query complexity:**
   - Shorter prompts
   - Fewer tool calls
   - Smaller date ranges

3. **Optimize tool usage:**
   - Narrow search filters
   - Reduce max results

## Best Practices

### Default Strategy

Use this decision tree:

```
Need vision?
├─ Yes → grok-2-vision-1212
└─ No → Need highest quality?
    ├─ Yes → grok-3
    └─ No → grok-4-1-fast
```

### When to Upgrade

Start with `grok-4-1-fast`, upgrade to `grok-3` if:
- Output lacks nuance
- Analysis is superficial
- Reasoning is flawed
- Quality matters more than speed

### When to Use Vision

Use `grok-2-vision-1212` for:
- Screenshots
- Diagrams
- UI mockups
- Code screenshots (prefer text when possible)
- Visual content that needs description

**Don't use vision for:**
- Text-only tasks (slower, no benefit)
- Simple lookups
- Routine chat

### Testing Models

Compare models for your use case:

```bash
# Test search quality
for model in grok-4-1-fast grok-3; do
  echo "Testing $model"
  node scripts/grok_search.mjs "complex query" --x --model $model
done

# Test analysis depth
node scripts/analyze.mjs voice @user --model grok-4-1-fast > fast.json
node scripts/analyze.mjs voice @user --model grok-3 > detailed.json
diff fast.json detailed.json
```

## Future Models

xAI releases new models periodically. Stay updated:

1. **Check announcements:** xAI blog, X account
2. **List models:** `node scripts/models.mjs`
3. **Test new models:** Add `--model <new-id>` to commands
4. **Update defaults:** Modify scripts if new models outperform current

**Backward compatibility:**
- Existing models remain available
- Old model IDs continue working
- New models may have different capabilities
- Test before updating production usage
