# xAI API Reference

Technical reference for xAI API endpoints used by this skill.

## Base URL

```
https://api.x.ai
```

## Authentication

All requests require Bearer token authentication:

```http
Authorization: Bearer xai-...
```

### API Key Configuration

Scripts check these locations in order:
1. `XAI_API_KEY` environment variable
2. `~/.clawdbot/clawdbot.json` → `env.XAI_API_KEY`
3. `~/.clawdbot/clawdbot.json` → `skills.entries.xai-plus.apiKey`
4. `~/.clawdbot/clawdbot.json` → `skills.entries["grok-search"].apiKey` (fallback)

Set via command:
```bash
clawdbot config set skills.entries.xai-plus.apiKey "xai-YOUR-KEY"
```

### Default Model Configuration

Scripts check these locations in order:
1. Command-line `--model` flag (highest priority)
2. `XAI_MODEL` environment variable
3. `~/.clawdbot/clawdbot.json` → `env.XAI_MODEL`
4. `~/.clawdbot/clawdbot.json` → `skills.entries.xai-plus.model`
5. Default: `grok-4-1-fast`

Set via command:
```bash
clawdbot config set skills.entries.xai-plus.model "grok-3"
```

## Endpoints

### POST /v1/responses

Unified endpoint for chat, search, and analysis using tools.

**Request:**

```json
{
  "model": "grok-4-1-fast",
  "input": [
    {
      "role": "user",
      "content": [
        {
          "type": "input_text",
          "text": "Your prompt"
        }
      ]
    }
  ],
  "tools": [],
  "store": false,
  "temperature": 0
}
```

**Response:**

```json
{
  "output": [
    {
      "role": "assistant",
      "content": [
        {
          "type": "output_text",
          "text": "Response text",
          "annotations": []
        }
      ]
    }
  ],
  "output_text": "Response text (convenience field)",
  "citations": []
}
```

**Input Content Types:**

```typescript
// Text input
{
  "type": "input_text",
  "text": string
}

// Image input (vision models only)
{
  "type": "input_image",
  "image_url": string  // data:image/jpeg;base64,...
}
```

### GET /v1/models

List available models.

**Response:**

```json
{
  "data": [
    {
      "id": "grok-4-1-fast",
      "object": "model",
      "owned_by": "xai"
    }
  ]
}
```

Alternative endpoint: `/v1/language-models`

## Tools

### x_search

Search X (Twitter) posts and threads.

**Configuration:**

```json
{
  "type": "x_search",
  "x_search": {
    "from_date": "2026-01-01",
    "to_date": "2026-01-31",
    "allowed_x_handles": ["username1", "username2"],
    "excluded_x_handles": ["spam1", "spam2"]
  }
}
```

**Parameters:**

| Parameter | Type | Required | Description |
|-----------|------|----------|-------------|
| `from_date` | string | No | Start date (YYYY-MM-DD) |
| `to_date` | string | No | End date (YYYY-MM-DD) |
| `allowed_x_handles` | string[] | No | Only search these accounts (no @) |
| `excluded_x_handles` | string[] | No | Exclude these accounts (no @) |

**Notes:**
- All date/handle parameters are optional
- Handles should NOT include @ symbol
- Date format must be YYYY-MM-DD
- Tool runs server-side; results depend on xAI's index freshness

### web_search

Search the web via Grok.

**Configuration:**

```json
{
  "type": "web_search"
}
```

No additional parameters supported.

## Models

### Available Models

| Model ID | Capabilities | Speed | Quality |
|----------|-------------|-------|---------|
| grok-4-1-fast | Text | Fast | Good |
| grok-4-fast | Text | Fast | Good |
| grok-3 | Text | Slower | Best |
| grok-2-vision-1212 | Text + Vision | Medium | Good |

### Model Selection

**Default (search, chat, analysis):**
```json
{
  "model": "grok-4-1-fast"
}
```

**High quality (complex reasoning):**
```json
{
  "model": "grok-3"
}
```

**Vision (image analysis):**
```json
{
  "model": "grok-2-vision-1212"
}
```

## Request Parameters

### Common Parameters

| Parameter | Type | Default | Description |
|-----------|------|---------|-------------|
| `model` | string | Required | Model ID to use |
| `input` | array | Required | Conversation messages |
| `tools` | array | `[]` | Tools to enable |
| `store` | boolean | `false` | Store conversation |
| `temperature` | number | `1.0` | Sampling temperature (0-2) |
| `max_tokens` | number | Model max | Max output tokens |

### Temperature Guidelines

- `0.0`: Deterministic (best for search, analysis)
- `0.3-0.7`: Balanced (good for most chat)
- `1.0+`: Creative (useful for brainstorming)

## Response Handling

### Text Extraction

The response may include text in multiple places:

```javascript
// Priority order:
const text =
  data.output_text ||  // Convenience field
  data?.output
    ?.flatMap(o => Array.isArray(o?.content) ? o.content : [])
    ?.find(c => c?.type === "output_text" && typeof c?.text === "string")
    ?.text ||
  null;
```

### Citations

Citations may appear in:

1. **Top-level citations array:**
```json
{
  "citations": ["https://..."]
}
```

2. **Annotations within content:**
```json
{
  "output": [
    {
      "content": [
        {
          "type": "output_text",
          "annotations": [
            {
              "url": "https://...",
              "web_citation": {
                "url": "https://..."
              }
            }
          ]
        }
      ]
    }
  ]
}
```

### Citation Deduplication

For X searches, prefer canonical URLs:
- ✓ `https://x.com/@handle/status/123`
- ✗ `https://x.com/i/status/123`

Use tweet ID to deduplicate, keeping the canonical format.

## Error Handling

### HTTP Status Codes

| Code | Meaning | Action |
|------|---------|--------|
| 200 | Success | Process response |
| 400 | Bad Request | Check request format |
| 401 | Unauthorized | Verify API key |
| 429 | Rate Limit | Retry with backoff |
| 500 | Server Error | Retry request |

### Error Response

```json
{
  "error": {
    "message": "Error description",
    "type": "error_type",
    "code": "error_code"
  }
}
```

### Common Errors

**Invalid API key:**
```json
{
  "error": {
    "message": "Invalid authentication credentials",
    "type": "authentication_error"
  }
}
```

**Rate limit:**
```json
{
  "error": {
    "message": "Rate limit exceeded",
    "type": "rate_limit_error"
  }
}
```

**Invalid model:**
```json
{
  "error": {
    "message": "Model not found",
    "type": "invalid_request_error"
  }
}
```

## Rate Limits

xAI enforces rate limits per API key. Exact limits depend on your plan.

**Best practices:**
- Use `temperature: 0` for deterministic tasks
- Batch related searches when possible
- Cache results when appropriate
- Handle 429 responses with exponential backoff

## API Changes

This reference reflects the xAI API as of January 2026. Check [docs.x.ai](https://docs.x.ai) for updates.

**Known variations:**
- Models endpoint may be `/v1/models` or `/v1/language-models`
- Response format may include `output_text` convenience field
- Citation format may vary between model versions
