# Analysis Prompts Reference

Structured prompts and scoring criteria for X content analysis.

## Voice Analysis

Analyze an account's writing patterns and voice.

### Input

```bash
node scripts/analyze.mjs voice @username [--days N]
```

### Prompt Structure

```
Analyze the voice and writing patterns of @{handle} on X.

Use x_search to find their recent posts (last {days} days).

Return ONLY valid JSON (no markdown) with this schema:
{
  "handle": "@{handle}",
  "analyzed_posts": number,
  "voice": {
    "tone": string,
    "personality": [string],
    "perspective": string,
    "energy_level": "high|medium|low"
  },
  "patterns": {
    "sentence_structure": [string],
    "vocabulary": [string],
    "formatting_quirks": [string],
    "recurring_phrases": [string]
  },
  "topics": [string],
  "best_posts": [
    {
      "url": string,
      "text": string,
      "why": string
    }
  ],
  "anti_patterns": [string]
}
```

### Output Schema

| Field | Type | Description |
|-------|------|-------------|
| `handle` | string | @username analyzed |
| `analyzed_posts` | number | Number of posts in analysis |
| `voice.tone` | string | Overall tone (casual, formal, technical, etc.) |
| `voice.personality` | string[] | Personality traits (curious, direct, playful, etc.) |
| `voice.perspective` | string | Writing perspective (practitioner, observer, educator, etc.) |
| `voice.energy_level` | enum | High, medium, or low energy |
| `patterns.sentence_structure` | string[] | Sentence patterns (short declarative, fragments, etc.) |
| `patterns.vocabulary` | string[] | Word choice patterns (technical, accessible, formal, etc.) |
| `patterns.formatting_quirks` | string[] | Formatting habits (line breaks, caps, punctuation) |
| `patterns.recurring_phrases` | string[] | Common phrases or expressions |
| `topics` | string[] | Main topics discussed |
| `best_posts` | object[] | Representative high-quality posts |
| `best_posts[].url` | string | Post URL |
| `best_posts[].text` | string | Post text |
| `best_posts[].why` | string | Why this post exemplifies their voice |
| `anti_patterns` | string[] | Things they never or rarely do |

### Interpretation

**Voice tone:**
- Describes overall feel of writing
- Can be multi-dimensional (e.g., "casual but technical")
- Look for consistency across posts

**Personality traits:**
- 3-5 key characteristics
- Observable from writing, not assumptions
- Should be evident in multiple posts

**Perspective:**
- How they position themselves
- Examples: expert sharing knowledge, learner documenting journey, observer commenting

**Energy level:**
- High: Exclamation marks, caps, intense language
- Medium: Balanced, varied expression
- Low: Understated, minimal punctuation

**Patterns:**
- Observable writing habits
- Should be specific and verifiable
- Help replicate the voice

**Anti-patterns:**
- Things they avoid or never do
- Useful for quality checks
- Help identify inauthenticity

### Use Cases

**Content creation:**
```bash
# Analyze target voice
node scripts/analyze.mjs voice @target --days 60 > target_voice.json

# Use output to inform writing style
```

**Account comparison:**
```bash
# Analyze multiple accounts
node scripts/analyze.mjs voice @account1 > voice1.json
node scripts/analyze.mjs voice @account2 > voice2.json

# Compare patterns
```

**Quality assurance:**
```bash
# Analyze own account periodically
node scripts/analyze.mjs voice @myaccount --days 90

# Check for voice drift over time
```

## Trend Research

Research trends and discussions about a topic.

### Input

```bash
node scripts/analyze.mjs trends "topic"
```

### Prompt Structure

```
Research trends and discussions about: {query}

Use x_search to find recent conversations (last 7 days).

Return ONLY valid JSON (no markdown) with this schema:
{
  "topic": "{query}",
  "trends": [
    {
      "pattern": string,
      "description": string,
      "example_posts": [string]
    }
  ],
  "perspectives": [
    {
      "viewpoint": string,
      "supporters": [string]
    }
  ],
  "hashtags": [string],
  "key_accounts": [string],
  "posting_angles": [
    {
      "angle": string,
      "hook": string,
      "target_audience": string
    }
  ]
}
```

### Output Schema

| Field | Type | Description |
|-------|------|-------------|
| `topic` | string | Topic researched |
| `trends` | object[] | Observed patterns in discussions |
| `trends[].pattern` | string | Trend description |
| `trends[].description` | string | Detailed explanation |
| `trends[].example_posts` | string[] | URLs demonstrating trend |
| `perspectives` | object[] | Different viewpoints on topic |
| `perspectives[].viewpoint` | string | Perspective description |
| `perspectives[].supporters` | string[] | Accounts holding this view |
| `hashtags` | string[] | Relevant hashtags |
| `key_accounts` | string[] | Influential accounts discussing topic |
| `posting_angles` | object[] | Suggested content angles |
| `posting_angles[].angle` | string | Content approach |
| `posting_angles[].hook` | string | Opening line/hook example |
| `posting_angles[].target_audience` | string | Who this angle appeals to |

### Interpretation

**Trends:**
- Patterns in how people discuss the topic
- Emerging themes or shifts in conversation
- Should be backed by multiple posts

**Perspectives:**
- Different viewpoints on the topic
- Not just pro/con, but nuanced positions
- Helps identify conversation landscape

**Key accounts:**
- Influential voices in the discussion
- Not just follower count, but engagement quality
- Good sources for further research

**Posting angles:**
- Actionable content ideas
- Based on gaps or opportunities in current discussion
- Should have clear target audience

### Use Cases

**Pre-writing research:**
```bash
# Research before creating content
node scripts/analyze.mjs trends "your topic" > trends.json

# Identify angles and gaps
jq '.posting_angles' trends.json
```

**Competitive analysis:**
```bash
# See how others discuss topic
node scripts/analyze.mjs trends "competitor product"

# Find differentiation opportunities
```

**Trend monitoring:**
```bash
# Weekly check on your domain
node scripts/analyze.mjs trends "your niche"

# Track conversation evolution
```

## Post Safety Check

Check a post for AI signals and platform flag patterns.

### Input

```bash
# Check draft text
node scripts/analyze.mjs post "Your post text"

# Check existing post
node scripts/analyze.mjs post --url "https://x.com/user/status/123"
```

### Prompt Structure (Draft Text)

```
Analyze this post draft for AI signals and platform flag patterns:

"{text}"

Check for:
1. AI detection signals (em-dashes, numbered lists, template phrases, perfect grammar, hollow depth)
2. Platform flag patterns (engagement bait, repetitive patterns, spam signals)
3. Overall authenticity and quality

Return ONLY valid JSON (no markdown) with this schema:
{
  "post_text": "{text}",
  "ai_detection_score": number (0-10, 10=definitely AI),
  "ai_signals": [string],
  "platform_flag_score": number (0-10, 10=high risk),
  "platform_risks": [string],
  "quality_score": number (0-10, 10=excellent),
  "suggestions": [string]
}
```

### Prompt Structure (Existing Post)

```
Analyze this X post for quality and safety: {url}

Use x_search to fetch the post content, then check for:
1. AI detection signals (em-dashes, numbered lists, template phrases, perfect grammar)
2. Platform flag patterns (engagement bait, repetitive patterns, spam signals)
3. Overall quality and authenticity

Return ONLY valid JSON (no markdown) with this schema:
{
  "post_url": "{url}",
  "post_text": string,
  "ai_detection_score": number (0-10, 10=definitely AI),
  "ai_signals": [string],
  "platform_flag_score": number (0-10, 10=high risk),
  "platform_risks": [string],
  "quality_score": number (0-10, 10=excellent),
  "suggestions": [string]
}
```

### Output Schema

| Field | Type | Description |
|-------|------|-------------|
| `post_text` | string | Post text analyzed |
| `post_url` | string | URL (if analyzing existing post) |
| `ai_detection_score` | number | 0-10 (10 = definitely AI) |
| `ai_signals` | string[] | Detected AI patterns |
| `platform_flag_score` | number | 0-10 (10 = high spam risk) |
| `platform_risks` | string[] | Platform flag patterns |
| `quality_score` | number | 0-10 (10 = excellent) |
| `suggestions` | string[] | Actionable improvements |

### X Algorithm Context

Understanding X's ranking and filtering mechanisms:

**Engagement Signal Weights (Relative to Like = 0.5x):**
- Like: 0.5x (baseline, low value)
- Reply to your post: 13-27x (high value)
- Reply loop (back-and-forth): 75x (extremely high value)
- Block: -148x (destroys reach, undoes ~148 likes)
- Mute/Report: Negative (less than block, but still harmful)

**Key Algorithm Behaviors:**
- **Dwell time** tracked separately from likes (posts worth re-reading rank higher)
- **Author diversity penalty**: Exponential decay when same author appears multiple times in feed
  - First post: full score
  - Each subsequent: reduced by decay factor
  - Implication: space posts 2-4 hours apart, quality > quantity
- **In-network (Thunder) vs Out-of-network (Phoenix)**: Out-of-network posts multiplied by weight factor <1.0
- **Negative signals** (blocks, "not interested", mutes) undo positive signals faster than you can build them

**Optimization Priorities:**
1. Get replies and respond fast (75x for loops)
2. Maximize dwell time (dense value, quotable lines)
3. Avoid blocks at all costs (-148x penalty)
4. Space posts to avoid author diversity decay
5. One meaningful reply > ten likes

### Scoring Criteria

**AI Detection Score (0-10):**
- 0-2: Human-like (natural, imperfect, authentic)
- 3-5: Borderline (some AI signals, but not definitive)
- 6-8: Likely AI (multiple strong signals)
- 9-10: Definitely AI (obvious patterns, no humanity)

**Platform Flag Score (0-10):**
- 0-2: Safe (natural engagement, no spam signals)
- 3-5: Low risk (minor issues, unlikely to flag)
- 6-8: Medium risk (patterns that may trigger filters)
- 9-10: High risk (will likely be flagged or suppressed)

**Quality Score (0-10):**
- 0-2: Poor (no value, generic, bad writing)
- 3-5: Below average (some issues, needs work)
- 6-8: Good (solid post, minor improvements possible)
- 9-10: Excellent (authentic, valuable, well-crafted)

### AI Detection Signals

Specific patterns that indicate AI generation (from X platform analysis):

**Dead Giveaways (+2-3 points each):**
- Em-dashes (—) in casual posts
- Numbered lists (1., 2., 3.) in non-thread content
- Engagement bait closers ("What do you think?", "Would love your thoughts", "Drop your take")
- Template structures: "It's not X — it's Y", "Here's the thing:", "Let me be clear:", "The real question is:"

**Structural Tells (+1-2 points each):**
- Perfect grammar throughout (no typos, fragments, or trailing thoughts)
- Every sentence complete and polished
- Reads like article/LinkedIn post, not a tweet
- Overly organized for casual context

**Vocabulary Red Flags (+1 point each):**
- Formal hedging: "Certainly", "Indeed", "I'd be happy to"
- Academic language: "It is important to note that"
- Corporate speak: "Leverage", "Utilize", "Facilitate"
- Robotic connectors: "Furthermore", "Additionally", "Moreover"
- Poetic but vague statements

**Content Issues (+1-2 points each):**
- Hollow depth (sounds profound, says nothing specific)
- Generic insights that apply to anything
- No specific details (names, dates, numbers, examples)
- Simulation of emotion without genuine expression
- Missing lived experience or personal context

**Missing Human Signals (-1 point each if present):**
- No typos or imperfections
- No fragments or incomplete thoughts
- No parenthetical asides
- No self-corrections mid-thought
- No "mess" in the writing

### Platform Flag Patterns

Patterns that trigger X's spam filters and algorithmic penalties (from platform analysis):

**Repetitive Patterns (High Risk):**
- Same opening phrases across multiple posts ("What are you building?", "Great point!")
- Template structures you reuse (openers, closers, question formats)
- Cookie-cutter replies that apply to any post
- Identical formatting patterns every time

**Engagement Bait (Medium-High Risk):**
- Generic questions begging for replies ("What do you think?", "Thoughts?")
- Low-effort questions adding nothing ("What kind?", "How so?")
- Ending every post with call for feedback
- Single-word low-effort replies ("This!", "Great!", "So true!")

**Promotional Spam (High Risk):**
- Mentioning your product in unrelated threads
- Subtle plugs woven into most replies
- "DM me" or invite code drops
- Self-promo in >10% of interactions

**Volume Red Flags (Algorithmic Penalties):**
- >5 replies per hour (triggers spam detection)
- >15-20 total posts+replies per day
- Burst activity (10+ replies in 30 minutes)
- Multiple replies to same thread in quick succession
- Original posts <2 hours apart (author diversity decay penalty)

**Structural Tells (Medium Risk):**
- Starting replies with "Hi!" + compliment + question + plug
- Jumping into every "what are you building" thread
- Copy-paste formats with minor word swaps
- Systematic engagement patterns

### Suggestions Format

Suggestions should be:
- Specific and actionable
- Prioritized (most important first)
- Tied to detected issues

**Good suggestions:**
- "Replace em-dash on line 2 with period"
- "Remove 'What do you think?' closer"
- "Add specific example from your experience"
- "Break up perfect grammar with fragment"

**Bad suggestions:**
- "Make it more human" (too vague)
- "Improve quality" (not actionable)
- "Fix the issues" (not specific)

### Use Cases

**Pre-publish check:**
```bash
# Check draft before posting
node scripts/analyze.mjs post "$(cat draft.txt)"

# Review scores and suggestions
```

**Batch checking:**
```bash
# Check multiple drafts
for file in drafts/*.txt; do
  echo "Checking $file"
  node scripts/analyze.mjs post "$(cat $file)" --json
done
```

**Existing post audit:**
```bash
# Check posted content
node scripts/analyze.mjs post --url "https://x.com/user/status/123"

# Learn from flagged posts
```

## General Analysis Tips

### Temperature Setting

All analysis scripts use `temperature: 0` for consistency:
- Deterministic output
- Reproducible results
- Reliable scoring

### Tool Usage

Voice and trend analysis use x_search tool:
- Date ranges filter results
- Handle filters focus analysis
- More recent = more relevant

### JSON Reliability

Request JSON explicitly in prompts:
```
Return ONLY valid JSON (no markdown) with this schema:
```

This ensures:
- Parseable output
- Structured data
- Agent-friendly format

### Model Selection

**Default (grok-4-1-fast):**
- Fast analysis
- Good quality
- Sufficient for most cases

**Upgrade to grok-3 when:**
- Need deeper insights
- Analyzing 90+ days of posts
- Complex trend research
- Quality over speed priority

### Error Handling

Analysis may fail if:
- Account is private/protected
- Too few posts in date range
- API rate limits hit
- Network issues

**Solutions:**
- Extend date range
- Verify account is public
- Retry with backoff
- Check API key validity

## Customization

### Modifying Prompts

To customize analysis behavior, edit the prompt in `scripts/analyze.mjs`:

**Example: Add sentiment analysis**
```javascript
const prompt = `Analyze the voice and writing patterns of @${handle} on X.

Use x_search to find their recent posts (last ${days} days).

// Add new field to schema:
{
  "sentiment": {
    "overall": "positive|neutral|negative",
    "consistency": number (0-10)
  }
}
```

### Adding Analysis Types

To add new analysis modes:

1. Add mode to valid modes array
2. Define prompt structure
3. Configure tools if needed
4. Handle output formatting

**Example skeleton:**
```javascript
else if (mode === "newmode") {
  prompt = `Your analysis prompt...`;
  tools = [/* tools if needed */];
}
```

### Custom Scoring

Adjust scoring weights by modifying prompt instructions:

```
ai_detection_score guidelines:
- Em-dash: +2 points
- Numbered list: +1 point
- Template phrase: +1 point each
- Perfect grammar: +1 point
- Engagement bait closer: +2 points
```

## Best Practices

1. **Run analysis before drafting**
   - Understand voice/trends first
   - Inform content strategy
   - Avoid common mistakes

2. **Check posts before publishing**
   - Catch AI signals
   - Identify platform risks
   - Improve quality

3. **Combine analysis types**
   - Voice + trends for comprehensive research
   - Post check + voice analysis for consistency
   - Trend research + competitor voice analysis

4. **Store results for comparison**
   - Track voice evolution
   - Monitor trend changes
   - Build historical dataset

5. **Use appropriate model**
   - Default for routine checks
   - grok-3 for deep analysis
   - Balance speed vs quality

## Advanced Usage

### Automated Quality Checks

```bash
#!/bin/bash
# Pre-publish check script

POST_FILE="$1"
POST_TEXT=$(cat "$POST_FILE")

# Run safety check
RESULT=$(node scripts/analyze.mjs post "$POST_TEXT" --json)

# Parse scores
AI_SCORE=$(echo "$RESULT" | jq -r '.ai_detection_score')
FLAG_SCORE=$(echo "$RESULT" | jq -r '.platform_flag_score')
QUALITY=$(echo "$RESULT" | jq -r '.quality_score')

# Decision logic
if [ "$AI_SCORE" -gt 5 ]; then
  echo "❌ AI detection too high: $AI_SCORE"
  exit 1
fi

if [ "$FLAG_SCORE" -gt 5 ]; then
  echo "❌ Platform risk too high: $FLAG_SCORE"
  exit 1
fi

if [ "$QUALITY" -lt 6 ]; then
  echo "⚠️  Quality below threshold: $QUALITY"
  exit 1
fi

echo "✓ Post passes quality checks"
```

### Periodic Voice Analysis

```bash
#!/bin/bash
# Weekly voice check

HANDLE="@myaccount"
DATE=$(date +%Y-%m-%d)
OUTPUT="voice_analysis_${DATE}.json"

node scripts/analyze.mjs voice $HANDLE --days 7 --json > "$OUTPUT"

# Track changes over time
echo "Voice analysis saved to $OUTPUT"
```

### Trend Monitoring

```bash
#!/bin/bash
# Daily trend check for your niche

TOPICS=("AI agents" "Claude Code" "automation")

for topic in "${TOPICS[@]}"; do
  echo "Researching: $topic"
  node scripts/analyze.mjs trends "$topic" --json > "trends_${topic// /_}.json"
done
```
