---
name: xai-plus
description: |
  Search X/Twitter and the web, chat with Grok models (text + vision), and analyze X content using xAI's API.
  Use when: searching X posts/threads, web research via Grok, chatting with Grok, analyzing voice patterns,
  researching trends, or checking post quality. Triggers: grok, xai, search x, search twitter, x search,
  ask grok, grok chat, analyze voice, x trends.
metadata:
  openclaw:
    emoji: "🔎"
    requires:
      bins: ["node"]
      env: ["XAI_API_KEY"]
    primaryEnv: "XAI_API_KEY"
---

# xAI Skill

Search X (Twitter), search the web, chat with Grok models (including vision), and analyze X content patterns using xAI's API.

## Features

- **X Search**: Search posts, threads, and accounts with date/handle filters
- **Web Search**: Search the web via Grok's web_search tool
- **Chat**: Text and vision (image analysis) with Grok models
- **Analysis**: Voice patterns, trend research, post safety checks
- **Models**: List available xAI models

## Setup

### API Key

Get your xAI API key from [console.x.ai](https://console.x.ai).

```bash
# Via clawdbot config (recommended)
clawdbot config set skills.entries.xai-plus.apiKey "xai-YOUR-KEY"

# Or environment variable
export XAI_API_KEY="xai-YOUR-KEY"
```

The scripts check these locations in order:
1. `XAI_API_KEY` env var
2. `~/.clawdbot/clawdbot.json` → `env.XAI_API_KEY`
3. `~/.clawdbot/clawdbot.json` → `skills.entries.xai-plus.apiKey`
4. `~/.clawdbot/clawdbot.json` → `skills.entries["grok-search"].apiKey` (fallback)

### Default Model (Optional)

Override the default model (`grok-4-1-fast`):

```bash
# Via config
clawdbot config set skills.entries.xai-plus.model "grok-3"

# Or environment variable
export XAI_MODEL="grok-3"
```

Model priority:
1. Command-line `--model` flag (highest priority)
2. `XAI_MODEL` env var
3. `~/.clawdbot/clawdbot.json` → `env.XAI_MODEL`
4. `~/.clawdbot/clawdbot.json` → `skills.entries.xai-plus.model`
5. Default: `grok-4-1-fast`

## Search

### X Search

Search X posts and threads with optional filters.

**Basic search:**
```bash
node {baseDir}/scripts/grok_search.mjs "query" --x
```

**With date filter:**
```bash
# Last 7 days
node {baseDir}/scripts/grok_search.mjs "Claude AI" --x --days 7

# Specific date range
node {baseDir}/scripts/grok_search.mjs "AI agents" --x --from 2026-01-01 --to 2026-01-31
```

**Filter by handles:**
```bash
# Only from specific accounts
node {baseDir}/scripts/grok_search.mjs "AI news" --x --handles @AnthropicAI,@OpenAI

# Exclude accounts
node {baseDir}/scripts/grok_search.mjs "GPT" --x --exclude @spam1,@spam2
```

**Output formats:**
```bash
# JSON (default, agent-friendly)
node {baseDir}/scripts/grok_search.mjs "query" --x

# Links only
node {baseDir}/scripts/grok_search.mjs "query" --x --links-only

# Human-readable text
node {baseDir}/scripts/grok_search.mjs "query" --x --text
```

**JSON output schema:**
```json
{
  "query": "search query",
  "mode": "x",
  "results": [
    {
      "title": "@handle",
      "url": "https://x.com/handle/status/123",
      "snippet": "Post text...",
      "author": "@handle",
      "posted_at": "2026-01-15T10:30:00Z"
    }
  ],
  "citations": ["https://x.com/..."]
}
```

### Web Search

Search the web via Grok.

```bash
node {baseDir}/scripts/grok_search.mjs "TypeScript best practices 2026" --web
```

**JSON output schema:**
```json
{
  "query": "search query",
  "mode": "web",
  "results": [
    {
      "title": "Page title",
      "url": "https://example.com/page",
      "snippet": "Description...",
      "author": null,
      "posted_at": null
    }
  ],
  "citations": ["https://example.com/..."]
}
```

### Search Options

| Flag | Description | Example |
|------|-------------|---------|
| `--x` | Search X/Twitter | Required for X search |
| `--web` | Search the web | Required for web search |
| `--days N` | Last N days (X only) | `--days 7` |
| `--from YYYY-MM-DD` | Start date (X only) | `--from 2026-01-01` |
| `--to YYYY-MM-DD` | End date (X only) | `--to 2026-01-31` |
| `--handles a,b` | Only these accounts (X only) | `--handles @user1,@user2` |
| `--exclude a,b` | Exclude accounts (X only) | `--exclude @spam` |
| `--max N` | Max results | `--max 20` |
| `--model ID` | Override model | `--model grok-3` |
| `--json` | JSON output (default) | - |
| `--links-only` | Just URLs | - |
| `--text` | Human-readable | - |
| `--raw` | Include debug output | - |

See [references/search-patterns.md](references/search-patterns.md) for advanced query patterns and optimization tips.

## Chat

### Text Chat

Ask Grok anything.

```bash
node {baseDir}/scripts/chat.mjs "What is quantum computing?"
```

**With model override:**
```bash
node {baseDir}/scripts/chat.mjs --model grok-3 "Explain transformers in ML"
```

**JSON output:**
```bash
node {baseDir}/scripts/chat.mjs --json "What is TypeScript?"
```

JSON schema:
```json
{
  "model": "grok-4-1-fast",
  "prompt": "What is TypeScript?",
  "text": "TypeScript is...",
  "citations": ["https://..."]
}
```

### Vision Chat

Analyze images with Grok.

```bash
node {baseDir}/scripts/chat.mjs --image ./screenshot.png "What's in this image?"
```

**Multiple images:**
```bash
node {baseDir}/scripts/chat.mjs --image ./pic1.jpg --image ./pic2.jpg "Compare these"
```

**Supported formats:** JPG, PNG, WebP, GIF

### Chat Options

| Flag | Description | Example |
|------|-------------|---------|
| `--model ID` | Model to use | `--model grok-2-vision-1212` |
| `--image PATH` | Attach image (can repeat) | `--image ./pic.jpg` |
| `--json` | JSON output | - |
| `--raw` | Include debug output | - |

See [references/models.md](references/models.md) for model comparison and capabilities.

## Analysis

Analyze X content for voice patterns, trends, and post quality.

### Voice Analysis

Analyze an account's voice and writing patterns.

```bash
node {baseDir}/scripts/analyze.mjs voice @username
```

**Custom date range:**
```bash
# Last 60 days
node {baseDir}/scripts/analyze.mjs voice @username --days 60
```

**JSON output schema:**
```json
{
  "handle": "@username",
  "analyzed_posts": 150,
  "voice": {
    "tone": "casual, technical",
    "personality": ["curious", "direct", "helpful"],
    "perspective": "practitioner sharing lessons",
    "energy_level": "medium"
  },
  "patterns": {
    "sentence_structure": ["short declarative", "occasional fragments"],
    "vocabulary": ["technical", "accessible"],
    "formatting_quirks": ["line breaks for emphasis", "minimal punctuation"],
    "recurring_phrases": ["here's the thing", "turns out"]
  },
  "topics": ["AI", "software engineering", "startups"],
  "best_posts": [
    {
      "url": "https://x.com/username/status/123",
      "text": "Post text...",
      "why": "Authentic voice, specific example"
    }
  ],
  "anti_patterns": ["never uses em-dashes", "avoids numbered lists"]
}
```

### Trend Research

Research trends and discussions about a topic.

```bash
node {baseDir}/scripts/analyze.mjs trends "AI agents"
```

**JSON output schema:**
```json
{
  "topic": "AI agents",
  "trends": [
    {
      "pattern": "Shift from chatbots to autonomous agents",
      "description": "Discussion focuses on...",
      "example_posts": ["https://x.com/..."]
    }
  ],
  "perspectives": [
    {
      "viewpoint": "Agents will replace most SaaS",
      "supporters": ["@user1", "@user2"]
    }
  ],
  "hashtags": ["#AIAgents", "#AutonomousAI"],
  "key_accounts": ["@researcher1", "@founder2"],
  "posting_angles": [
    {
      "angle": "Practical implementation challenges",
      "hook": "Everyone talks about AI agents. Nobody talks about...",
      "target_audience": "Engineers building with AI"
    }
  ]
}
```

### Post Safety Check

Check a draft post or existing post for AI signals and platform flag patterns.

**Check draft text:**
```bash
node {baseDir}/scripts/analyze.mjs post "Your draft post text here"
```

**Check existing post:**
```bash
node {baseDir}/scripts/analyze.mjs post --url "https://x.com/user/status/123"
```

**JSON output schema:**
```json
{
  "post_text": "Your post...",
  "ai_detection_score": 3,
  "ai_signals": [
    "Contains em-dash",
    "Ends with engagement bait question"
  ],
  "platform_flag_score": 2,
  "platform_risks": [
    "Generic question could trigger spam filter"
  ],
  "quality_score": 7,
  "suggestions": [
    "Replace em-dash with period or comma",
    "Remove 'What do you think?' closer",
    "Add specific personal detail"
  ]
}
```

**Scoring:**
- **AI Detection**: 0-10 (10 = definitely AI)
- **Platform Flag Risk**: 0-10 (10 = high spam risk)
- **Quality**: 0-10 (10 = excellent)

### Analysis Options

| Flag | Description | Example |
|------|-------------|---------|
| `--days N` | Date range for analysis | `--days 60` |
| `--url URL` | Analyze existing post | `--url https://x.com/...` |
| `--model ID` | Override model | `--model grok-3` |
| `--json` | JSON output | - |
| `--raw` | Include debug output | - |

See [references/analysis-prompts.md](references/analysis-prompts.md) for detailed prompt structures and scoring criteria.

## Models

List available xAI models.

```bash
node {baseDir}/scripts/models.mjs
```

**Output:**
```
grok-2-vision-1212
grok-3
grok-4-1-fast
grok-4-fast
```

**JSON output:**
```bash
node {baseDir}/scripts/models.mjs --json
```

Quick model comparison:

| Model | Speed | Quality | Vision | Best For |
|-------|-------|---------|--------|----------|
| grok-4-1-fast | Fast | Good | No | Default (search, chat, analysis) |
| grok-4-fast | Fast | Good | No | Alternative fast model |
| grok-3 | Slower | Best | No | Complex reasoning, detailed analysis |
| grok-2-vision-1212 | Medium | Good | Yes | Image analysis |

See [references/models.md](references/models.md) for detailed model capabilities and use cases.

## Advanced Usage

### Citation Deduplication

For X searches, the tool automatically deduplicates tweet URLs, preferring canonical `/@handle/status/id` over `/i/status/id` format.

### Custom Model Selection

Override the default model for any operation:

```bash
# Search with grok-3 for better quality
node {baseDir}/scripts/grok_search.mjs "complex query" --x --model grok-3

# Chat with vision model
node {baseDir}/scripts/chat.mjs --model grok-2-vision-1212 --image pic.jpg "Describe"

# Analysis with grok-3 for deeper insights
node {baseDir}/scripts/analyze.mjs voice @username --model grok-3
```

### Debugging

Add `--raw` to any command to see the full API response:

```bash
node {baseDir}/scripts/grok_search.mjs "query" --x --raw
```

## Reference Documentation

- [API Reference](references/api-reference.md) - xAI API endpoints and parameters
- [Search Patterns](references/search-patterns.md) - Query patterns, filters, optimization tips
- [Models](references/models.md) - Model comparison and capabilities
- [Analysis Prompts](references/analysis-prompts.md) - Structured prompts and scoring criteria
- [X Algorithm](references/x-algorithm.md) - Ranking, engagement weights, spam detection

## Examples

### Research a topic
```bash
# Find recent discussions
node {baseDir}/scripts/grok_search.mjs "Claude Sonnet 4.5" --x --days 3

# Get trend analysis
node {baseDir}/scripts/analyze.mjs trends "Claude Sonnet 4.5"
```

### Analyze voice before drafting
```bash
# Study the target account
node {baseDir}/scripts/analyze.mjs voice @target_account --days 30

# Check your draft
node {baseDir}/scripts/analyze.mjs post "Your draft here"
```

### Multi-modal research
```bash
# Search web for context
node {baseDir}/scripts/grok_search.mjs "TypeScript 5.7 features" --web

# Ask follow-up
node {baseDir}/scripts/chat.mjs "What are the key TypeScript 5.7 improvements?"

# Analyze screenshot
node {baseDir}/scripts/chat.mjs --image ./code.png "Review this code"
```

## Error Handling

Common errors and solutions:

**Missing API key:**
```
Missing XAI_API_KEY
```
→ Set `XAI_API_KEY` environment variable or add to `~/.clawdbot/clawdbot.json`

**Invalid mode:**
```
Must specify --web or --x
```
→ Add `--web` or `--x` flag to search command

**Image format error:**
```
Unsupported image type
```
→ Use JPG, PNG, WebP, or GIF format

**API error:**
```
xAI API error: 401 Unauthorized
```
→ Check API key is valid and active

## Tips

- Default model (`grok-4-1-fast`) is fast and good for most tasks
- Use `grok-3` for complex analysis or when quality matters more than speed
- X searches are limited by recency (xAI x_search tool constraints)
- Web searches work best with specific, current queries
- Voice analysis needs sufficient post history (30+ posts recommended)
- Post safety checks are advisory; use judgment for final decisions
- JSON output is best for agent/script consumption
- Text output is better for terminal/human reading

## Troubleshooting

**No results from X search:**
- Try broader query or longer date range
- Check if handles exist and are public
- Remove overly restrictive filters

**Voice analysis incomplete:**
- Increase `--days` to get more post history
- Check if account is public and active
- Verify handle is correct (with or without @)

**API rate limits:**
- xAI enforces rate limits per API key
- Spread requests over time if hitting limits
- Consider upgrading xAI plan for higher limits

## Content Writing Workflow

Use the analysis tools to improve your X content:

```bash
# Research before writing
node {baseDir}/scripts/analyze.mjs trends "your topic"
node {baseDir}/scripts/grok_search.mjs "your topic" --x --days 7

# Study voice patterns
node {baseDir}/scripts/analyze.mjs voice @target_account

# Check draft before posting
node {baseDir}/scripts/analyze.mjs post "$(cat draft.txt)"
```

Use the JSON output to:
- Research current discussions and posting angles
- Learn from successful voices in your niche
- Catch AI signals and platform flags before publishing
