#!/usr/bin/env python3
"""
Generate X/Twitter posts that match an analyzed voice profile.
Can work standalone or integrate with LLM APIs.
"""

import json
import sys
import subprocess
import argparse
from pathlib import Path
from typing import List, Dict, Any, Optional

def load_profile(profile_path: str) -> Dict[str, Any]:
    """Load voice profile from JSON"""
    with open(profile_path, 'r') as f:
        return json.load(f)

def quick_analyze(username: str) -> Dict[str, Any]:
    """Quick analysis if no profile exists"""
    analyze_script = Path(__file__).parent / "analyze_voice.py"
    temp_profile = "/tmp/voice-profile-temp.json"
    
    result = subprocess.run(
        ["python3", str(analyze_script), username, "--tweets", "30", "--output", temp_profile],
        capture_output=True,
        text=True
    )
    
    if result.returncode != 0:
        raise RuntimeError(f"Analysis failed: {result.stderr}")
    
    return load_profile(temp_profile)

def build_generation_prompt(profile: Dict[str, Any], topic: str, post_type: Optional[str] = None) -> str:
    """Build prompt for LLM to generate posts matching voice"""
    
    patterns = profile['patterns']
    
    # Extract key characteristics
    avg_length = patterns.get('avg_length', 100)
    topics = ', '.join(patterns.get('topics', []))
    phrases = ', '.join(patterns.get('signature_phrases', []))
    tone = patterns.get('tone', {})
    style = patterns.get('style', {})
    engagement = patterns.get('engagement_type', '')
    
    # Sample tweets for reference
    samples = '\n'.join(f"  - {tweet}" for tweet in profile.get('sample_tweets', [])[:5])
    
    prompt = f"""Analyze this Twitter account's voice and generate posts that EXACTLY match their style.

ACCOUNT: {profile['account']}
ANALYZED TWEETS: {profile['analyzed_tweets']}

VOICE CHARACTERISTICS:
• Average length: {avg_length} characters ({patterns.get('distribution', {})})
• Topics: {topics}
• Tone: {json.dumps(tone, indent=2)}
• Style: {json.dumps(style, indent=2)}
• Engagement type: {engagement}
• Signature phrases: {phrases}

SAMPLE TWEETS (for reference):
{samples}

CRITICAL STYLE RULES:
1. Match the length ({avg_length} ± 20 chars preferred)
2. Use their signature phrases naturally: {phrases}
3. Match their tone: {"self-deprecating" if tone.get('self_deprecating') else ""}{"sarcastic" if tone.get('sarcastic') else ""}{"edgy" if tone.get('edgy') else ""}
4. Match capitalization: {style.get('capitalization', 'mixed')}
5. Match emoji usage: {style.get('emoji_usage', 'moderate')}
6. Match punctuation: {style.get('punctuation', 'casual')}
7. Stay on-brand for topics: {topics}

TASK: Generate 3 different posts about "{topic}" that sound EXACTLY like this account.

Each post should:
- Sound authentic (not forced or tryhard)
- Use natural language for this person
- Match their humor style
- Feel like something they would actually tweet

Format as JSON array:
[
  {{
    "post": "the actual tweet text",
    "confidence": 0.85,
    "reasoning": "why this matches their voice"
  }}
]

Generate posts that would make someone think "{profile['account']} definitely wrote this."
"""
    
    return prompt

def generate_via_llm(prompt: str) -> List[Dict[str, Any]]:
    """
    Generate posts using available LLM.
    This is a placeholder - in real usage, Dale (the agent) will call this
    and use their own LLM access to generate.
    
    For standalone use, this could integrate with OpenAI/Anthropic APIs.
    """
    
    # When Dale uses this, they'll intercept here and use their LLM
    print("GENERATION_PROMPT:", file=sys.stderr)
    print(prompt, file=sys.stderr)
    print("\n" + "="*80, file=sys.stderr)
    print("NOTE: This script outputs the prompt for Dale to use with their LLM.", file=sys.stderr)
    print("Dale will generate the posts and return them.", file=sys.stderr)
    print("="*80 + "\n", file=sys.stderr)
    
    # Return empty so agent knows to handle this
    return []

def main():
    parser = argparse.ArgumentParser(description='Generate posts matching voice profile')
    parser.add_argument('--profile', help='Path to voice profile JSON')
    parser.add_argument('--account', help='Account to analyze (if no profile)')
    parser.add_argument('--topic', required=True, help='Topic for the posts')
    parser.add_argument('--type', help='Post type (hot-take, observation, meme, etc.)')
    parser.add_argument('--count', type=int, default=3, help='Number of posts to generate')
    parser.add_argument('--output', help='Output JSON file (optional)')
    
    args = parser.parse_args()
    
    # Load or create profile
    if args.profile:
        profile = load_profile(args.profile)
    elif args.account:
        print(f"No profile provided, analyzing {args.account}...", file=sys.stderr)
        profile = quick_analyze(args.account)
    else:
        print("Error: Must provide --profile or --account", file=sys.stderr)
        return 1
    
    # Build prompt
    prompt = build_generation_prompt(profile, args.topic, args.type)
    
    # Output prompt (for Dale to use)
    print("\n" + "="*80)
    print("GENERATION PROMPT FOR LLM:")
    print("="*80)
    print(prompt)
    print("="*80 + "\n")
    
    # In actual usage, Dale will see this prompt and generate posts
    # using their Claude access, then format the results
    
    return 0

if __name__ == '__main__':
    sys.exit(main())
