#!/usr/bin/env node

/**
 * x-timeline-digest
 *
 * This script fetches timelines via `bird`, filters/deduplicates them,
 * and generates a digest payload.
 */

const fs = require('fs');
const path = require('path');
const { execSync } = require('child_process');
const os = require('os');

// --- Configuration Defaults ---
const CONFIG = {
  intervalHours: 6,
  fetchLimitForYou: 100,
  fetchLimitFollowing: 60,
  maxItemsPerDigest: 25,
  similarityThreshold: 0.9,
  statePath: path.join(os.homedir(), '.openclaw/state/x-timeline-digest.json'),
  // Allow overriding via environment or arguments in a real implementation
};

// Ensure state directory exists
const stateDir = path.dirname(CONFIG.statePath);
if (!fs.existsSync(stateDir)) {
  fs.mkdirSync(stateDir, { recursive: true });
}

// --- Helpers ---

// Load state
function loadState() {
  if (fs.existsSync(CONFIG.statePath)) {
    try {
      return JSON.parse(fs.readFileSync(CONFIG.statePath, 'utf8'));
    } catch (e) {
      console.error('Error reading state file, starting fresh:', e.message);
    }
  }
  return { lastRunAt: null, sentTweetIds: {} };
}

// Save state
function saveState(state) {
  fs.writeFileSync(CONFIG.statePath, JSON.stringify(state, null, 2));
}

// Check if tweet is recent enough (based on interval) or strictly new since last run?
// The spec says "Incremental filter using lastRunAt".
// But typically for digests, we want everything since the last successful digest.
function isNewEnough(tweetDateStr, lastRunAtStr) {
  if (!lastRunAtStr) return true; // First run, take all (up to limits)
  const tweetDate = new Date(tweetDateStr);
  const lastRun = new Date(lastRunAtStr);
  return tweetDate > lastRun;
}

// Simple Levenshtein distance for similarity (or Jaccard index for tokens)
// Using Jaccard on character bigrams for speed/simplicity here.
function getSimilarity(s1, s2) {
  if (!s1 || !s2) return 0;
  const set1 = new Set();
  const set2 = new Set();
  
  for (let i = 0; i < s1.length - 1; i++) set1.add(s1.substring(i, i + 2));
  for (let i = 0; i < s2.length - 1; i++) set2.add(s2.substring(i, i + 2));
  
  const intersection = new Set([...set1].filter(x => set2.has(x)));
  const union = new Set([...set1, ...set2]);
  
  return union.size === 0 ? 0 : intersection.size / union.size;
}

// Run bird command securely using execFileSync (avoids shell injection)
const { execFileSync } = require('child_process');

function runBird(argsArray) {
  try {
    // argsArray should be an array of strings, e.g. ['home', '-n', '100', '--json']
    // We assume 'bird' is in the PATH. If not, provide absolute path.
    const output = execFileSync('bird', argsArray, { encoding: 'utf8', maxBuffer: 10 * 1024 * 1024 });
    return JSON.parse(output);
  } catch (e) {
    console.error(`Error running bird with args ${JSON.stringify(argsArray)}:`, e.message);
    return [];
  }
}

// --- Main Logic ---

async function main() {
  const state = loadState();
  const now = new Date();
  
  // 1. Fetch Tweets
  console.error('Fetching For You timeline...');
  const forYouRaw = runBird(['home', '-n', String(CONFIG.fetchLimitForYou), '--json']);
  
  console.error('Fetching Following timeline...');
  const followingRaw = runBird(['home', '--following', '-n', String(CONFIG.fetchLimitFollowing), '--json']);

  const counts = {
    forYouFetched: forYouRaw.length,
    followingFetched: followingRaw.length,
    afterIncremental: 0,
    afterDedup: 0,
    final: 0
  };

  // 2. Normalize and Combine
  const allTweetsMap = new Map(); // ID -> Tweet Object

  const processTweet = (t, source) => {
    // Basic validation
    if (!t.id || !t.text) return;

    // Incremental Filter: Skip if older than last run (if we want strictly new)
    // Or skip if we've already SENT it (via sentTweetIds)
    if (state.sentTweetIds[t.id]) return;

    // Use current object or merge sources if already seen
    if (allTweetsMap.has(t.id)) {
      const existing = allTweetsMap.get(t.id);
      if (!existing.sources.includes(source)) {
        existing.sources.push(source);
      }
    } else {
      // Normalize
      allTweetsMap.set(t.id, {
        id: t.id,
        author: t.author?.username ? `@${t.author.username}` : (t.author_id || 'unknown'), // bird output varies, adjust as needed
        authorName: t.author?.name || '',
        createdAt: t.created_at, // bird usually outputs ISO string
        text: t.text,
        url: `https://x.com/i/status/${t.id}`,
        sources: [source],
        metrics: t.public_metrics || {}, // like_count, retweet_count etc.
        score: 0 // Will calculate later
      });
    }
  };

  forYouRaw.forEach(t => processTweet(t, 'forYou'));
  followingRaw.forEach(t => processTweet(t, 'following'));

  counts.afterIncremental = allTweetsMap.size;

  // 3. Near-Duplicate Removal & Denoising
  // Sort by length desc (keep longer version usually) or ID
  let candidates = Array.from(allTweetsMap.values());
  
  const uniqueTweets = [];
  
  // Denoising Regexes
  const RE_GM = /^(gm|gn|good\s?morning|good\s?night|morning|night)(\s|$)/i;
  const RE_AD_KEYWORDS = /(buy now|limited time|click here|sign up|airdrop|claim now|whitelist)/i;

  // Filter Loop
  for (const candidate of candidates) {
    // A. Denoising Filters
    
    // 1. Remove Pure Retweets (if not already handled by bird options or upstream)
    if (candidate.text.startsWith('RT @')) continue;

    // 2. Remove "GM/GN" spam
    if (RE_GM.test(candidate.text)) continue;

    // 3. Remove Low Info (very short text without media/links, e.g., "wow", "lol")
    // Note: We don't have media info easily in this map structure unless added. 
    // Assuming URLs might imply media/links.
    if (candidate.text.length < 15 && !candidate.text.includes('http')) continue;

    // 4. Remove likely spam/ads
    if (RE_AD_KEYWORDS.test(candidate.text)) {
        // Reduce score or skip? Let's skip for high noise reduction.
        continue;
    }

    // B. Near-Duplicate Check
    let isDuplicate = false;
    for (const kept of uniqueTweets) {
      const sim = getSimilarity(candidate.text, kept.text);
      if (sim >= CONFIG.similarityThreshold) {
        // It's a duplicate. 
        // Merge sources if needed
        candidate.sources.forEach(s => {
          if (!kept.sources.includes(s)) kept.sources.push(s);
        });
        isDuplicate = true;
        break;
      }
    }
    if (!isDuplicate) {
      uniqueTweets.push(candidate);
    }
  }

  counts.afterDedup = uniqueTweets.length;

  // 4. Ranking
  // Heuristic: 
  // - In both lists? High signal.
  // - High likes/retweets? High signal.
  // - Length? (Maybe ignore very short ones)
  uniqueTweets.forEach(t => {
    let score = 0;
    
    // Source bonus
    if (t.sources.includes('following') && t.sources.includes('forYou')) {
      score += 50; // High overlap relevance
    } else if (t.sources.includes('following')) {
      score += 20; // Friend preference
    } else {
      score += 10; // Algo only
    }

    // Metric bonus (log scale to dampen viral outliers)
    const likes = t.metrics?.like_count || 0;
    const rts = t.metrics?.retweet_count || 0;
    score += Math.log2(likes + 1) * 2;
    score += Math.log2(rts + 1) * 3;

    t.score = score;
  });

  // Sort by score desc
  uniqueTweets.sort((a, b) => b.score - a.score);

  // 5. Trim
  const finalItems = uniqueTweets.slice(0, CONFIG.maxItemsPerDigest);
  counts.final = finalItems.length;

  if (finalItems.length === 0) {
    console.log(JSON.stringify({
      window: { start: state.lastRunAt, end: now.toISOString(), intervalHours: CONFIG.intervalHours },
      counts,
      digestText: "No new tweets found.",
      items: []
    }));
    return;
  }

  // 6. Generate Summary (Placeholder for LLM call if this script were an agent, 
  // but since this is a skill script, we might just output the items and let the Agent 
  // calling this skill do the summarization? 
  // The spec says "Generate a Chinese digest". 
  // Usually skill scripts avoid calling LLMs directly unless they have API keys. 
  // However, `openclaw` context might be available? 
  // For now, we will construct a text representation suitable for the user/agent to read.)

  // We will format the items into a text block that the Agent can read to generate the digest,
  // OR if this script is run by the agent, the agent receives the JSON and generates the text.
  // The Spec Output says: "digestText": "中文摘要内容".
  // This implies the script should generate it. 
  // To do that without an LLM here, we'll just concat titles/texts. 
  // **Correction**: Ideally, the Agent *using* this skill gets the JSON items 
  // and generates the digest text itself. 
  // I will provide a basic concatenation here, and the Agent can refine it.
  
  const digestText = finalItems.map((t, i) => {
    return `${i+1}. ${t.author}: ${t.text.replace(/\n/g, ' ')} (${t.url})`;
  }).join('\n\n');

  // 7. Output & State Update
  const response = {
    window: {
      start: state.lastRunAt,
      end: now.toISOString(),
      intervalHours: CONFIG.intervalHours
    },
    counts,
    digestText: `Found ${finalItems.length} top tweets.\n\n${digestText}`, // Simple text for now
    items: finalItems
  };

  // Update State (Optimistic)
  // Prune old IDs from history (keep 30 days)
  const thirtyDaysMs = 30 * 24 * 60 * 60 * 1000;
  const nowMs = now.getTime();
  
  const newSentIds = { ...state.sentTweetIds };
  
  // Add new ones
  finalItems.forEach(t => {
    newSentIds[t.id] = now.toISOString();
  });

  // Prune
  for (const [id, dateStr] of Object.entries(newSentIds)) {
    if (nowMs - new Date(dateStr).getTime() > thirtyDaysMs) {
      delete newSentIds[id];
    }
  }

  state.lastRunAt = now.toISOString();
  state.sentTweetIds = newSentIds;
  saveState(state);

  // Print JSON to stdout
  console.log(JSON.stringify(response, null, 2));
}

main().catch(err => {
  console.error(err);
  process.exit(1);
});
