#!/usr/bin/env bash
# worktree-agents/scripts/push_and_pr.sh
# 推送分支并在 GitHub 上创建 PR
#
# 用法: push_and_pr.sh <repo_dir> <gh_token> <github_owner_repo> <agent_name> <worktree_path> <branch> <base_branch>
# 输出: PR URL

set -euo pipefail

REPO_DIR="$1"
GH_TOKEN="$2"
OWNER_REPO="$3"
AGENT_NAME="$4"
WORKTREE_PATH="$5"
BRANCH="$6"
BASE_BRANCH="$7"

cd "$WORKTREE_PATH"

# 用 GIT_ASKPASS 环境变量传 token，避免写入 remote URL（防止 token 泄漏到 git log/remote 输出）
export GIT_ASKPASS_TOKEN="$GH_TOKEN"

# 创建临时 askpass 脚本
ASKPASS_SCRIPT=$(mktemp /tmp/git-askpass-XXXXXX.sh)
cat > "$ASKPASS_SCRIPT" << 'EOF'
#!/usr/bin/env bash
echo "$GIT_ASKPASS_TOKEN"
EOF
chmod +x "$ASKPASS_SCRIPT"

# 确保 remote URL 是干净的 https（不含 token）
git remote set-url origin "https://github.com/${OWNER_REPO}.git"

# 推送
GIT_ASKPASS="$ASKPASS_SCRIPT" git push origin "$BRANCH" 2>&1

# 删除临时 askpass 脚本
rm -f "$ASKPASS_SCRIPT"

# 修复：用 printf 生成换行正确的 PR body（原来 $'...\n...' 在 jq --arg 里是字面 \n）
PR_BODY=$(printf "## %s Output\n\nGenerated by OpenClaw multi-agent worktree orchestrator.\n\nWorktree: \`%s\`\nBranch: \`%s\`" \
  "$AGENT_NAME" "$WORKTREE_PATH" "$BRANCH")

RESPONSE=$(curl -s -X POST \
  -H "Authorization: Bearer $GH_TOKEN" \
  -H "Accept: application/vnd.github+json" \
  "https://api.github.com/repos/$OWNER_REPO/pulls" \
  -d "$(jq -n \
    --arg title "[$AGENT_NAME] automated task" \
    --arg head "$BRANCH" \
    --arg base "$BASE_BRANCH" \
    --arg body "$PR_BODY" \
    '{title: $title, head: $head, base: $base, body: $body}')")

PR_URL=$(echo "$RESPONSE" | jq -r '.html_url // empty')

if [ -z "$PR_URL" ]; then
  echo "[ERROR] PR 创建失败，API 返回：" >&2
  echo "$RESPONSE" >&2
  exit 1
fi

echo "$PR_URL"
