#!/bin/bash
# workspace-status.sh — Quick workspace health check
# Run from workspace root or pass path as argument

WORKSPACE="${1:-$HOME/.openclaw/workspace}"

echo "=== Workspace Status: $WORKSPACE ==="
echo ""

# Check if workspace exists
if [ ! -d "$WORKSPACE" ]; then
    echo "❌ Workspace not found: $WORKSPACE"
    exit 1
fi

cd "$WORKSPACE"

echo "## Required Files"
for file in AGENTS.md SOUL.md USER.md IDENTITY.md TOOLS.md; do
    if [ -f "$file" ]; then
        lines=$(wc -l < "$file")
        echo "✅ $file ($lines lines)"
    else
        echo "❌ $file MISSING"
    fi
done

echo ""
echo "## Optional Files"
for file in HEARTBEAT.md MEMORY.md BOOT.md BOOTSTRAP.md; do
    if [ -f "$file" ]; then
        lines=$(wc -l < "$file")
        echo "✅ $file ($lines lines)"
    else
        echo "⬜ $file (not present)"
    fi
done

echo ""
echo "## Directories"
for dir in memory skills docs canvas; do
    if [ -d "$dir" ]; then
        count=$(find "$dir" -type f | wc -l)
        echo "✅ $dir/ ($count files)"
    else
        echo "⬜ $dir/ (not present)"
    fi
done

echo ""
echo "## Memory Files (last 7 days)"
if [ -d "memory" ]; then
    find memory -name "*.md" -mtime -7 -exec basename {} \; 2>/dev/null | sort -r | head -7
else
    echo "No memory directory"
fi

echo ""
echo "## Git Status"
if [ -d ".git" ]; then
    if git diff --quiet && git diff --cached --quiet; then
        echo "✅ Working tree clean"
    else
        echo "⚠️  Uncommitted changes:"
        git status --short
    fi
    
    untracked=$(git ls-files --others --exclude-standard | wc -l)
    if [ "$untracked" -gt 0 ]; then
        echo "⚠️  $untracked untracked files"
    fi
else
    echo "⬜ Not a git repository"
fi

echo ""
echo "## File Sizes (lines)"
wc -l AGENTS.md SOUL.md USER.md IDENTITY.md TOOLS.md HEARTBEAT.md MEMORY.md 2>/dev/null | grep -v total

echo ""
echo "## Potential Issues"
# Check for rogue files
for file in README.md NOTES.md TODO.md; do
    if [ -f "$file" ]; then
        echo "⚠️  Found $file — may duplicate bootstrap file purposes"
    fi
done

# Check for secrets
if grep -r -l "sk-" --include="*.md" . 2>/dev/null | grep -v node_modules; then
    echo "⚠️  Possible API key found in .md files"
fi

if [ -f ".env" ]; then
    echo "⚠️  .env file in workspace (should be gitignored)"
fi

echo ""
echo "=== End Status ==="
