# OpenClaw Workspace Conventions

Source: OpenClaw docs (`/docs/concepts/agent-workspace.md`, `/docs/concepts/memory.md`)

## Workspace Location

- Default: `~/.openclaw/workspace`
- Override via `agents.defaults.workspace` in `~/.openclaw/openclaw.json`
- Profile-specific: `~/.openclaw/workspace-<profile>` when `OPENCLAW_PROFILE` is set

**Important:** Only ONE active workspace. Multiple workspaces cause auth/state drift.

## What Lives WHERE

### In Workspace (`~/.openclaw/workspace/`)

| File | Purpose | Loaded |
|------|---------|--------|
| `AGENTS.md` | Operating instructions, memory workflow, behavior rules | Every session |
| `SOUL.md` | Persona, tone, boundaries | Every session |
| `USER.md` | User profile, preferences | Every session |
| `IDENTITY.md` | Agent name, vibe, emoji, external handles | Every session |
| `TOOLS.md` | Environment-specific tool notes | Every session |
| `HEARTBEAT.md` | Short checklist for heartbeat runs | On heartbeat |
| `BOOT.md` | Startup checklist on gateway restart | On boot |
| `BOOTSTRAP.md` | One-time first-run ritual | Once, then delete |
| `MEMORY.md` | Curated long-term memory | Main session only |
| `memory/YYYY-MM-DD.md` | Daily logs | Read today + yesterday |
| `memory/*.md` | Reference docs (deep dives, projects) | On demand |
| `skills/` | Workspace-specific skills | On demand |
| `canvas/` | Canvas UI files | On demand |

**Note:** Only `MEMORY.md` and `memory/**/*.md` are vector-indexed.

### NOT in Workspace (lives in `~/.openclaw/`)

- `openclaw.json` — Config
- `credentials/` — OAuth tokens, API keys
- `agents/<agentId>/sessions/` — Session transcripts
- `skills/` — Managed/installed skills
- `memory/<agentId>.sqlite` — Vector index (SQLite with embeddings)

**Never commit these to git.**

## File Specifications

### AGENTS.md
- **Purpose:** Operating instructions for the agent
- **Contains:** Rules, priorities, "how to behave", memory workflow instructions
- **Size:** < 500 lines ideal, < 1000 max
- **Anti-patterns:**
  - Personal memories (use MEMORY.md or daily files)
  - Skill-specific procedures (use skills/)
  - Environment config (use TOOLS.md)

### SOUL.md
- **Purpose:** Persona definition
- **Contains:** Identity philosophy, tone guidelines, boundaries, emotional character
- **Size:** < 200 lines ideal
- **Anti-patterns:**
  - Task lists
  - Technical instructions
  - Memories about what happened

### USER.md
- **Purpose:** User profile
- **Contains:** Name, pronouns, timezone, preferences, how to address them
- **Size:** < 100 lines ideal
- **Anti-patterns:**
  - Agent memories
  - System configuration
  - Relationship history (that goes in MEMORY.md)

### IDENTITY.md
- **Purpose:** Agent's external identity
- **Contains:** Name, nickname, emoji, avatar, external handles (GitHub, Twitter, wallets)
- **Size:** < 50 lines ideal
- **Anti-patterns:**
  - Philosophy about identity (that's SOUL.md)
  - Instructions

### TOOLS.md
- **Purpose:** Environment-specific notes
- **Contains:** Camera names, SSH hosts, voice preferences, device nicknames
- **Size:** Variable
- **Anti-patterns:**
  - Skill instructions (use skills/)
  - Operating procedures (use AGENTS.md)

### HEARTBEAT.md
- **Purpose:** Checklist for periodic heartbeat runs
- **Contains:** Short, actionable items to check
- **Size:** < 100 lines (token burn concern)
- **Anti-patterns:**
  - Full procedures (just reference them)
  - Documentation

**Heartbeat Response Contract:**
- Reply `HEARTBEAT_OK` when nothing needs attention
- `HEARTBEAT_OK` at start/end of reply is stripped; if remaining content ≤ 300 chars, reply is dropped
- For alerts, do NOT include `HEARTBEAT_OK` — just return the alert text
- If HEARTBEAT.md is empty (only headers/blank lines), heartbeat is skipped to save API calls

### MEMORY.md
- **Purpose:** Curated long-term memory
- **Contains:** Distilled lessons, key people, important projects, lasting context
- **Loaded:** Main/private session ONLY (never in group chats)
- **Anti-patterns:**
  - Raw daily logs (use memory/*.md)
  - Duplicate of daily file content
  - Sensitive credentials

### memory/YYYY-MM-DD.md
- **Purpose:** Daily log
- **Contains:** What happened today, raw notes, session summaries
- **Naming:** Strict `YYYY-MM-DD.md` format
- **Workflow:** Read today + yesterday on session start
- **Anti-patterns:**
  - Curated insights without also putting in MEMORY.md
  - Very old files never reviewed

### memory/*.md (reference docs)
- **Purpose:** Searchable reference material
- **Contains:** Deep dives, project docs, technical references
- **Why here:** `memory/` is vector-indexed for semantic search. Files elsewhere are NOT searchable.
- **Examples:** `api-integration-notes.md`, `project-alpha.md`
- **Note:** Daily logs use `YYYY-MM-DD.md` naming; reference docs use descriptive names

## Git Conventions

**⚠️ IMPORTANT: This workspace is PRIVATE. Never push to GitHub or any public remote.**

The workspace contains personal memory, identity, and potentially sensitive context. Git is for LOCAL version control and backup only.

### Do Track (locally)
- All bootstrap files (AGENTS.md, SOUL.md, etc.)
- memory/*.md
- skills/

### Do NOT Track (gitignore)
```gitignore
.DS_Store
.env
**/*.key
**/*.pem
**/secrets*
```

### Never Do
- Push to GitHub or any public repository
- Commit API keys, OAuth tokens, passwords
- Commit anything from ~/.openclaw/
- Share raw chat dumps with sensitive content

## Memory Workflow

1. **During sessions:** Write notable things to `memory/YYYY-MM-DD.md`
2. **Periodically:** Review daily files, distill insights to `MEMORY.md`
3. **On session start:** Read today + yesterday's daily files
4. **In main session:** Also read MEMORY.md
5. **In group chats:** Do NOT read MEMORY.md (security)

### Automatic Memory Flush (Pre-Compaction)

OpenClaw automatically triggers a **silent agent turn** before context compaction:

- Fires when session approaches ~180k tokens (configurable via `compaction.reserveTokensFloor` + `softThresholdTokens`)
- Agent receives system prompt: "Session nearing compaction. Store durable memories now."
- Agent should write lasting notes to `memory/YYYY-MM-DD.md`, then reply `NO_REPLY`
- One flush per compaction cycle (tracked in sessions.json)

Config (`agents.defaults.compaction.memoryFlush`):
```json5
{
  enabled: true,
  softThresholdTokens: 4000,
  systemPrompt: "Session nearing compaction. Store durable memories now.",
  prompt: "Write any lasting notes to memory/YYYY-MM-DD.md; reply with NO_REPLY if nothing to store."
}
```

### Vector Search

Only `MEMORY.md` and `memory/**/*.md` are indexed by default.

**Extra paths:** To index files outside `memory/`, add them to config:
```json5
agents: {
  defaults: {
    memorySearch: {
      extraPaths: ["../team-docs", "/srv/shared-notes"]
    }
  }
}
```

**Session memory (experimental):** Enable `memorySearch.experimental.sessionMemory = true` to also index session transcripts. This allows `memory_search` to recall conversations.

See [Memory docs](/concepts/memory) for full details on vector search, hybrid search, and session memory.

## Skill Conventions

- Workspace skills in `<workspace>/skills/` override managed skills
- Each skill needs `SKILL.md` with frontmatter (`name`, `description`)
- Keep SKILL.md lean (< 500 lines)
- Use `references/` for detailed docs
- Use `scripts/` for executable code
- Use `assets/` for templates/files used in output

## Common Anti-Patterns

1. **Scope creep** — File grows beyond its purpose
2. **Duplication** — Same info in multiple files
3. **Wrong location** — Memories in SOUL.md, instructions in IDENTITY.md
4. **Bloat** — Bootstrap files too large (loaded every session)
5. **Rogue files** — README.md, NOTES.md that duplicate bootstrap purposes
6. **Secrets in workspace** — API keys should be in ~/.openclaw/credentials/
7. **Stale daily files** — Never reviewed, insights never curated
