#!/usr/bin/env bash
# Social Media Scheduler — generates a week of platform-optimized posts
# Usage: ./generate.sh "topic" ["target audience"]

set -euo pipefail

TOPIC="${1:?Usage: ./generate.sh \"topic\" [\"target audience\"]}"
AUDIENCE="${2:-general audience}"
OUTPUT="content-calendar.md"
DATE=$(date +%Y-%m-%d)

TOPIC_TAG=$(echo "$TOPIC" | tr '[:upper:]' '[:lower:]' | sed 's/[^a-z0-9]//g')
TOPIC_TITLE=$(echo "$TOPIC" | awk '{for(i=1;i<=NF;i++) $i=toupper(substr($i,1,1)) substr($i,2)}1')

generate_day() {
  local day="$1" theme="$2" tw_time="$3" li_time="$4" ig_time="$5"
  local theme_lower
  theme_lower=$(echo "$theme" | tr '[:upper:]' '[:lower:]')
  local day_lower
  day_lower=$(echo "$day" | tr '[:upper:]' '[:lower:]')

  cat << EOF
## ${day} — ${theme}

### 🐦 Twitter/X (Post at ${tw_time})
> [Your ${TOPIC_TITLE} tweet here — keep under 280 chars]
>
> 💡 Theme: ${theme}
> Hashtags: #${TOPIC_TAG} #${day}Motivation #Tips

**Format notes:** Punchy, conversational. Use a hook in the first line. Consider a thread if the idea is big.

### 💼 LinkedIn (Post at ${li_time})
> **[Compelling headline about ${TOPIC_TITLE}]**
>
> [Opening hook — a bold statement or surprising stat about ${TOPIC}]
>
> [2-3 sentences expanding on the ${theme_lower} angle for ${AUDIENCE}]
>
> [Call to action — ask for comments or shares]
>
> #${TOPIC_TAG} #ProfessionalDevelopment #${TOPIC_TAG}Tips #Insights #CareerGrowth

**Format notes:** Professional but human. Line breaks for readability. End with a question to drive engagement.

### 📸 Instagram (Post at ${ig_time})
> [Visual idea: Describe the image/carousel/reel concept for ${theme}]
>
> Caption:
> [Story-driven opening line about ${TOPIC}]
>
> [2-3 lines of value — ${theme_lower} angle]
>
> [CTA: "Save this for later" / "Tag someone who needs this" / "Link in bio"]
>
> .
> .
> .
> #${TOPIC_TAG} #${TOPIC_TAG}tips #${TOPIC_TAG}community #contentcreator #${day_lower}vibes #motivation #growth #learning #socialmedia #instagood

**Format notes:** Visual-first. Carousel posts get highest saves. Use line breaks and dot spacers before hashtag block.

---

EOF
}

# Write header
cat > "$OUTPUT" << EOF
# 📅 Social Media Content Calendar
**Topic:** ${TOPIC_TITLE}
**Audience:** ${AUDIENCE}
**Generated:** ${DATE}

---

EOF

# Generate each day
generate_day "Monday"    "Motivational / Week Opener"    "8:00 AM"  "7:30 AM"  "11:00 AM" >> "$OUTPUT"
generate_day "Tuesday"   "Educational / How-To"          "10:00 AM" "8:00 AM"  "1:00 PM"  >> "$OUTPUT"
generate_day "Wednesday" "Engagement / Question"         "12:00 PM" "12:00 PM" "5:00 PM"  >> "$OUTPUT"
generate_day "Thursday"  "Behind-the-Scenes / Story"     "9:00 AM"  "7:30 AM"  "11:00 AM" >> "$OUTPUT"
generate_day "Friday"    "Tip / Quick Win"               "11:00 AM" "9:00 AM"  "2:00 PM"  >> "$OUTPUT"
generate_day "Saturday"  "Curated / Industry News"       "10:00 AM" "10:00 AM" "11:00 AM" >> "$OUTPUT"
generate_day "Sunday"    "Reflection / Community"        "7:00 PM"  "8:00 PM"  "6:00 PM"  >> "$OUTPUT"

# Append strategy footer
cat >> "$OUTPUT" << 'EOF'
## 📊 Weekly Strategy Notes

### Content Mix
- 🎯 **Promotional:** 1-2 posts (keep the sell soft)
- 📚 **Educational:** 2-3 posts (your main value driver)
- 💬 **Engagement:** 1-2 posts (questions, polls, debates)
- 📖 **Storytelling:** 1-2 posts (personal, behind-the-scenes)

### Cross-Platform Tips
- Repurpose content across platforms but **adapt the format**
- Twitter threads → LinkedIn articles → Instagram carousels
- Engage in comments for 15 min after posting
- Track what resonates and double down next week

### Recommended Tools
- **Scheduling:** Buffer, Hootsuite, or Later
- **Graphics:** Canva, Figma
- **Analytics:** Native platform insights + Google Analytics

---
*Generated by Social Media Scheduler 🦞 — a ClawHub skill by Shelly*
EOF

echo "✅ Content calendar generated: ${OUTPUT}"
echo "   Topic: ${TOPIC_TITLE}"
echo "   Audience: ${AUDIENCE}"
echo "   Posts: 21 drafts (7 days × 3 platforms)"
echo ""
echo "📝 Next steps:"
echo "   1. Open ${OUTPUT} and fill in the bracketed sections"
echo "   2. Add your brand voice and specific examples"
echo "   3. Create visuals for Instagram posts"
echo "   4. Schedule using your favorite tool"
