/**
 * 真实数据源接入模块
 * 使用公开 API 和 RSS 获取热点数据
 */

const fetch = require('node-fetch');
const cheerio = require('cheerio');
const config = require('./config');

class RealSources {
  /**
   * 获取微博热搜（通过公开 API）
   */
  async getWeiboHot() {
    try {
      console.log('[数据源] 获取微博热搜...');
      
      // 使用第三方聚合 API（更稳定）
      const response = await fetch('https://weibo.com/ajax/side/hotSearch', {
        headers: {
          'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36',
          'Accept': 'application/json'
        },
        timeout: 10000
      });
      
      if (!response.ok) {
        throw new Error(`HTTP ${response.status}`);
      }
      
      const data = await response.json();
      const hotList = data.data?.realtime || [];
      
      return hotList.slice(0, 30).map((item, index) => ({
        source: 'weibo',
        title: item.note || item.word,
        hotValue: item.num || (100 - index * 3),
        url: `https://s.weibo.com/weibo?q=${encodeURIComponent(item.note || item.word)}`,
        timestamp: new Date().toISOString(),
        raw: item
      }));
      
    } catch (error) {
      console.error('[数据源] 微博热搜获取失败:', error.message);
      return [];
    }
  }

  /**
   * 获取知乎热榜
   */
  async getZhihuHot() {
    try {
      console.log('[数据源] 获取知乎热榜...');
      
      const response = await fetch('https://www.zhihu.com/api/v3/feed/topstory/hot-list?limit=50&desktop=true', {
        headers: {
          'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
          'Accept': 'application/json'
        },
        timeout: 10000
      });
      
      if (!response.ok) {
        throw new Error(`HTTP ${response.status}`);
      }
      
      const data = await response.json();
      
      return data.data?.slice(0, 30).map((item, index) => ({
        source: 'zhihu',
        title: item.target?.title || item.title,
        hotValue: item.target?.answer_count || (100 - index * 3),
        url: item.target?.url || item.url,
        timestamp: new Date().toISOString(),
        excerpt: item.target?.excerpt,
        raw: item
      }));
      
    } catch (error) {
      console.error('[数据源] 知乎热榜获取失败:', error.message);
      return [];
    }
  }

  /**
   * 获取 36 氪科技新闻
   */
  async get36Kr() {
    try {
      console.log('[数据源] 获取 36 氪科技新闻...');
      
      const response = await fetch('https://36kr.com/api/newsflash?per_page=20', {
        headers: {
          'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
          'Accept': 'application/json'
        },
        timeout: 10000
      });
      
      if (!response.ok) {
        throw new Error(`HTTP ${response.status}`);
      }
      
      const data = await response.json();
      const items = data.data?.items || [];
      
      return items.map(item => ({
        source: '36kr',
        title: item.title,
        hotValue: 80 - Math.floor(Math.random() * 30),
        url: `https://36kr.com/p/${item.id}`,
        timestamp: item.published_at || new Date().toISOString(),
        content: item.content,
        raw: item
      }));
      
    } catch (error) {
      console.error('[数据源] 36 氪获取失败:', error.message);
      return [];
    }
  }

  /**
   * 获取虎嗅科技新闻
   */
  async getHuxiu() {
    try {
      console.log('[数据源] 获取虎嗅新闻...');
      
      // 虎嗅需要解析 HTML
      const response = await fetch('https://www.huxiu.com/article/', {
        headers: {
          'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36'
        },
        timeout: 10000
      });
      
      if (!response.ok) {
        throw new Error(`HTTP ${response.status}`);
      }
      
      const html = await response.text();
      const $ = cheerio.load(html);
      const articles = [];
      
      // 解析虎嗅文章列表
      $('.article-list .article-item').each((i, el) => {
        const title = $(el).find('.article-title').text().trim();
        const url = $(el).find('a').attr('href');
        
        if (title && url) {
          articles.push({
            source: 'huxiu',
            title: title,
            hotValue: 80 - i * 2,
            url: url.startsWith('http') ? url : `https://www.huxiu.com${url}`,
            timestamp: new Date().toISOString()
          });
        }
      });
      
      return articles.slice(0, 20);
      
    } catch (error) {
      console.error('[数据源] 虎嗅获取失败:', error.message);
      // 返回备用数据
      return this.getHuxiuBackup();
    }
  }

  /**
   * 获取 Product Hunt（科技产品）
   */
  async getProductHunt() {
    try {
      console.log('[数据源] 获取 Product Hunt...');
      
      // Product Hunt 需要 API Key，这里用公开页面
      const response = await fetch('https://www.producthunt.com/', {
        headers: {
          'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36'
        },
        timeout: 10000
      });
      
      if (!response.ok) {
        throw new Error(`HTTP ${response.status}`);
      }
      
      const html = await response.text();
      const $ = cheerio.load(html);
      const products = [];
      
      // 简单解析（实际可能需要更复杂的解析逻辑）
      $('[data-test="post-item"]').each((i, el) => {
        const title = $(el).find('[data-test="post-name"]').text().trim();
        const url = $(el).find('a').first().attr('href');
        
        if (title && url) {
          products.push({
            source: 'producthunt',
            title: title,
            hotValue: 90 - i * 3,
            url: url.startsWith('http') ? url : `https://www.producthunt.com${url}`,
            timestamp: new Date().toISOString()
          });
        }
      });
      
      return products.slice(0, 15);
      
    } catch (error) {
      console.error('[数据源] Product Hunt 获取失败:', error.message);
      return [];
    }
  }

  /**
   * 获取 GitHub Trending
   */
  async getGitHubTrending() {
    try {
      console.log('[数据源] 获取 GitHub Trending...');
      
      const response = await fetch('https://github.com/trending', {
        headers: {
          'User-Agent': 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36'
        },
        timeout: 10000
      });
      
      if (!response.ok) {
        throw new Error(`HTTP ${response.status}`);
      }
      
      const html = await response.text();
      const $ = cheerio.load(html);
      const repos = [];
      
      $('article.Box-row').each((i, el) => {
        const title = $(el).find('h2 a').text().trim().replace(/\n/g, '').trim();
        const url = $(el).find('h2 a').attr('href');
        const description = $(el).find('p').text().trim();
        
        if (title && url) {
          repos.push({
            source: 'github',
            title: title,
            hotValue: 95 - i * 3,
            url: `https://github.com${url}`,
            timestamp: new Date().toISOString(),
            description: description
          });
        }
      });
      
      return repos.slice(0, 20);
      
    } catch (error) {
      console.error('[数据源] GitHub Trending 获取失败:', error.message);
      return [];
    }
  }

  /**
   * 获取 Hacker News
   */
  async getHackerNews() {
    try {
      console.log('[数据源] 获取 Hacker News...');
      
      const response = await fetch('https://hacker-news.firebaseio.com/v0/topstories.json', {
        timeout: 10000
      });
      
      if (!response.ok) {
        throw new Error(`HTTP ${response.status}`);
      }
      
      const storyIds = await response.json();
      const stories = [];
      
      // 获取前 20 个故事的详情
      for (let i = 0; i < Math.min(20, storyIds.length); i++) {
        const storyId = storyIds[i];
        const storyResponse = await fetch(`https://hacker-news.firebaseio.com/v0/item/${storyId}.json`);
        const story = await storyResponse.json();
        
        if (story && story.title) {
          stories.push({
            source: 'hackernews',
            title: story.title,
            hotValue: story.score || (100 - i * 3),
            url: story.url || `https://news.ycombinator.com/item?id=${storyId}`,
            timestamp: new Date(story.time * 1000).toISOString(),
            points: story.score,
            comments: story.descendants
          });
        }
      }
      
      return stories;
      
    } catch (error) {
      console.error('[数据源] Hacker News 获取失败:', error.message);
      return [];
    }
  }

  /**
   * 备用数据（当虎嗅 API 失败时）
   */
  getHuxiuBackup() {
    return [
      { source: 'huxiu', title: 'AI 大模型竞争进入新阶段', hotValue: 88, url: '#', timestamp: new Date().toISOString() },
      { source: 'huxiu', title: '科技巨头加码 AI 硬件', hotValue: 82, url: '#', timestamp: new Date().toISOString() },
      { source: 'huxiu', title: '创业公司融资动态', hotValue: 75, url: '#', timestamp: new Date().toISOString() }
    ];
  }

  /**
   * 获取所有数据源
   */
  async getAllSources() {
    console.log('\n========== 获取真实数据源 ==========\n');
    
    const results = await Promise.allSettled([
      this.getWeiboHot(),
      this.getZhihuHot(),
      this.get36Kr(),
      this.getHuxiu(),
      this.getGitHubTrending(),
      this.getHackerNews()
    ]);
    
    const allTopics = [];
    
    results.forEach((result, index) => {
      const sourceNames = ['微博', '知乎', '36 氪', '虎嗅', 'GitHub', 'HackerNews'];
      
      if (result.status === 'fulfilled') {
        const topics = result.value;
        console.log(`✅ ${sourceNames[index]}: ${topics.length} 条`);
        allTopics.push(...topics);
      } else {
        console.log(`❌ ${sourceNames[index]}: ${result.reason}`);
      }
    });
    
    console.log(`\n共获取 ${allTopics.length} 条热点`);
    
    return allTopics;
  }
}

// 直接运行时测试
if (require.main === module) {
  const sources = new RealSources();
  sources.getAllSources().then(topics => {
    console.log('\n=== 前 10 条热点 ===');
    topics.slice(0, 10).forEach((t, i) => {
      console.log(`${i + 1}. [${t.source}] ${t.title} (热度：${t.hotValue})`);
    });
  });
}

module.exports = RealSources;
