#!/usr/bin/env python3
"""
Safe Vault - Secure transaction signing and execution.

Usage:
    python safe_vault.py --init --mode simulation
    python safe_vault.py --prepare-tx --to 0x... --value 0 --data 0x...
    python safe_vault.py --simulate --to 0x... --data 0x...
"""

import argparse
import json
import os
import sys
import uuid
from datetime import datetime
from typing import Optional
import urllib.request
import urllib.error

# Config path
CONFIG_PATH = os.path.join(os.path.dirname(__file__), "..", "..", "config", "config.json")
WHITELIST_PATH = os.path.join(os.path.dirname(__file__), "..", "..", "config", "whitelist.json")

# Error messages in natural language for external agents
ERROR_MESSAGES = {
    "NO_PRIVATE_KEY": "❌ 无法签名交易：未配置私钥。请设置 WEB3_INVESTOR_PRIVATE_KEY 环境变量，或将钱包交给 Agent 管理。",
    "WHITELIST_BLOCKED": "⛔ 交易被拒绝：目标地址不在白名单中。请先使用 `python scripts/trading/whitelist.py --add <address>` 添加地址到白名单。",
    "EXCEEDS_LIMIT": "⚠️ 交易金额超过限额。当前限额为 ${limit} USD，而此交易价值为 ${value} USD。请在 config.json 中调整限额或分批执行。",
    "SIMULATION_FAILED": "🔴 交易模拟失败：{reason}。这可能是由于合约调用错误或网络问题导致的。请检查交易参数后重试。",
    "RPC_ERROR": "📡 无法连接到区块链网络：{error}。请检查 RPC 配置或稍后重试。",
    "INSUFFICIENT_BALANCE": "💰 余额不足：您的钱包中没有足够的资金来执行此交易。请先充值后再尝试。",
    "INVALID_ADDRESS": "📛 无效的地址格式：{address}。请提供有效的以太坊地址（0x 开头，40 位十六进制字符）。",
    "SIGNING_UNAVAILABLE": "🔐 本地签名功能不可用。请安装 eth-account：`pip install eth-account`，或使用外部钱包手动签名。",
}


def load_config() -> dict:
    """Load configuration."""
    if os.path.exists(CONFIG_PATH):
        with open(CONFIG_PATH) as f:
            return json.load(f)
    return {"trading": {"mode": "simulation", "whitelist_enabled": True, "default_limit_usd": 100}}


def load_whitelist() -> dict:
    """Load whitelist."""
    if os.path.exists(WHITELIST_PATH):
        with open(WHITELIST_PATH) as f:
            return json.load(f)
    return {"addresses": [], "enabled": True}


def save_whitelist(whitelist: dict):
    """Save whitelist."""
    os.makedirs(os.path.dirname(WHITELIST_PATH), exist_ok=True)
    with open(WHITELIST_PATH, "w") as f:
        json.dump(whitelist, f, indent=2)


def check_whitelist(address: str, amount_usd: float = 0) -> dict:
    """
    Check if address is in whitelist and within limits.
    
    Returns:
        dict with allowed (bool), reason (str), limit (float)
    """
    config = load_config()
    whitelist = load_whitelist()
    
    if not config["trading"].get("whitelist_enabled", True):
        return {"allowed": True, "reason": "Whitelist disabled", "limit": float("inf")}
    
    if not whitelist.get("enabled", True):
        return {"allowed": True, "reason": "Whitelist disabled", "limit": float("inf")}
    
    address_lower = address.lower()
    
    for entry in whitelist.get("addresses", []):
        if entry.get("address", "").lower() == address_lower:
            limit = entry.get("max_amount_usd", config["trading"].get("default_limit_usd", 100))
            if amount_usd <= limit:
                return {"allowed": True, "reason": f"Within limit (${amount_usd:.2f} <= ${limit})", "limit": limit}
            else:
                return {"allowed": False, "reason": f"Exceeds limit (${amount_usd:.2f} > ${limit})", "limit": limit}
    
    return {"allowed": False, "reason": "Address not in whitelist", "limit": 0}


def get_eth_price_usd() -> float:
    """Get current ETH price from CoinGecko (free API)."""
    try:
        url = "https://api.coingecko.com/api/v3/simple/price?ids=ethereum&vs_currencies=usd"
        with urllib.request.urlopen(url, timeout=10) as response:
            data = json.loads(response.read().decode())
            return data["ethereum"]["usd"]
    except:
        return 2000  # Fallback estimate


def get_token_price_usd(token_address: str) -> float:
    """Get token price from CoinGecko."""
    # Common tokens
    known_tokens = {
        "0xa0b86991c6218b36c1d19d4a2e9eb0ce3606eb48": 1.0,  # USDC
        "0xdac17f958d2ee523a2206206994597c13d831ec7": 1.0,  # USDT
        "0x6b175474e89094c44da98b954eedeac495271d0f": 1.0,  # DAI
        "0xc02aaa39b223fe8d0a0e5c4f27ead9083c756cc2": get_eth_price_usd(),  # WETH
    }
    
    return known_tokens.get(token_address.lower(), 0)


def estimate_value_usd(value_wei: str, token_address: str = None) -> float:
    """Estimate transaction value in USD."""
    if token_address:
        return get_token_price_usd(token_address)
    
    # ETH value
    try:
        value_eth = float(value_wei) / 1e18
        return value_eth * get_eth_price_usd()
    except:
        return 0


def simulate_transaction(to: str, value: str, data: str) -> dict:
    """
    Simulate transaction using Tenderly (free) or local estimation.
    
    For demo, we use a simple gas estimation.
    """
    # Simple simulation result
    result = {
        "success": True,
        "gas_estimate": 200000,
        "gas_price_gwei": 20,
        "gas_cost_eth": 0.004,  # 200k * 20 gwei
        "warnings": []
    }
    
    # Add warnings for common patterns
    if not data or data == "0x":
        result["warnings"].append("Empty calldata - this is a simple ETH transfer")
    
    if value and float(value) > 0:
        result["warnings"].append(f"Sending {float(value)/1e18:.4f} ETH")
    
    return result


def prepare_transaction(
    to: str,
    value: str = "0",
    data: str = "0x",
    gas_limit: int = 300000,
    description: str = ""
) -> dict:
    """
    Prepare a transaction for signing.
    
    Returns a signing request that can be shown to the user.
    """
    config = load_config()
    mode = config["trading"].get("mode", "simulation")
    
    # Estimate value
    value_usd = estimate_value_usd(value)
    
    # Check whitelist
    whitelist_result = check_whitelist(to, value_usd)
    
    # Simulate
    simulation = simulate_transaction(to, value, data)
    
    # Generate request
    request_id = str(uuid.uuid4())
    request = {
        "request_id": request_id,
        "timestamp": datetime.utcnow().isoformat() + "Z",
        "mode": mode,
        "transaction": {
            "to": to,
            "value": value,
            "data": data,
            "gas_limit": gas_limit,
            "chain_id": 1,  # Ethereum mainnet
        },
        "simulation": simulation,
        "whitelist": whitelist_result,
        "value_usd": round(value_usd, 2),
        "description": description,
        "approval_required": mode == "simulation" or not whitelist_result["allowed"]
    }
    
    return request


def format_signing_request(request: dict) -> str:
    """Format signing request for display."""
    lines = [
        "╔══════════════════════════════════════════════════════════════╗",
        "║                    🔐 SIGNING REQUEST                         ║",
        "╠══════════════════════════════════════════════════════════════╣",
        f"║ ID: {request['request_id'][:8]}...",
        f"║ Time: {request['timestamp']}",
        "╠══════════════════════════════════════════════════════════════╣",
        "║ TRANSACTION                                                   ║",
        f"║ To:    {request['transaction']['to']}",
        f"║ Value: {request['transaction']['value']} wei (${request['value_usd']})",
        f"║ Data:  {request['transaction']['data'][:50]}{'...' if len(request['transaction']['data']) > 50 else ''}",
        "╠══════════════════════════════════════════════════════════════╣",
        "║ SIMULATION                                                    ║",
        f"║ ✅ Success: {request['simulation']['success']}",
        f"║ ⛽ Gas Estimate: {request['simulation']['gas_estimate']:,}",
        f"║ 💰 Gas Cost: ~{request['simulation']['gas_cost_eth']:.4f} ETH",
    ]
    
    if request['simulation']['warnings']:
        lines.append("║ ⚠️ Warnings:")
        for w in request['simulation']['warnings']:
            lines.append(f"║   - {w}")
    
    lines.extend([
        "╠══════════════════════════════════════════════════════════════╣",
        "║ WHITELIST CHECK                                               ║",
        f"║ {'✅' if request['whitelist']['allowed'] else '❌'} {request['whitelist']['reason']}",
    ])
    
    if request['approval_required']:
        lines.extend([
            "╠══════════════════════════════════════════════════════════════╣",
            "║ ⚠️ MANUAL APPROVAL REQUIRED                                   ║",
            "║                                                               ║",
            "║ 1. Copy the transaction data above                            ║",
            "║ 2. Open your wallet (MetaMask / Safe)                         ║",
            "║ 3. Review and sign the transaction                            ║",
        ])
    
    lines.append("╚══════════════════════════════════════════════════════════════╝")
    
    return "\n".join(lines)


def execute_local_signing(transaction: dict, private_key: str) -> dict:
    """
    Execute transaction signing locally using eth-account.
    
    This function allows agents with local signing capabilities to sign transactions
    directly without manual intervention.
    
    Args:
        transaction: Transaction dict with to, value, data, gas_limit, chain_id, etc.
        private_key: Private key for signing (hex string with or without 0x prefix)
    
    Returns:
        dict with signed_transaction, tx_hash, and status
    """
    try:
        from eth_account import Account
        from eth_account.datastructures import SignedTransaction
    except ImportError:
        return {
            "success": False,
            "error": ERROR_MESSAGES["SIGNING_UNAVAILABLE"],
            "error_code": "SIGNING_UNAVAILABLE"
        }
    
    try:
        # Clean up private key
        if private_key.startswith("0x"):
            private_key = private_key[2:]
        
        # Create account from private key
        account = Account.from_key(private_key)
        
        # Build transaction
        tx = {
            "to": transaction["to"],
            "value": int(transaction.get("value", "0")),
            "data": transaction.get("data", "0x"),
            "gas": transaction.get("gas_limit", 300000),
            "gasPrice": transaction.get("gas_price", 20000000000),
            "nonce": transaction.get("nonce", 0),  # Should be fetched from network
            "chainId": transaction.get("chain_id", 1),
        }
        
        # Sign transaction
        signed = account.sign_transaction(tx)
        
        return {
            "success": True,
            "signed_transaction": signed.rawTransaction.hex(),
            "tx_hash": signed.hash.hex(),
            "from_address": account.address,
            "message": f"✅ 交易已成功签名！发送地址：{account.address}\n交易哈希：{signed.hash.hex()}\n\n您现在可以将此签名交易提交到网络。"
        }
        
    except Exception as e:
        error_msg = str(e).lower()
        if "invalid key" in error_msg or "private key" in error_msg:
            return {
                "success": False,
                "error": ERROR_MESSAGES["NO_PRIVATE_KEY"],
                "error_code": "INVALID_KEY",
                "details": str(e)
            }
        else:
            return {
                "success": False,
                "error": f"❌ 签名过程中发生错误：{str(e)}。请检查您的私钥和网络配置。",
                "error_code": "SIGNING_ERROR",
                "details": str(e)
            }


def check_agent_signing_capability() -> dict:
    """Check if the agent has local signing capabilities."""
    result = {
        "has_eth_account": False,
        "has_private_key": False,
        "can_sign_locally": False,
        "message": ""
    }
    
    # Check if eth-account is installed
    try:
        from eth_account import Account
        result["has_eth_account"] = True
    except ImportError:
        result["message"] = ERROR_MESSAGES["SIGNING_UNAVAILABLE"]
        return result
    
    # Check if private key is available
    private_key = os.environ.get("WEB3_INVESTOR_PRIVATE_KEY")
    if private_key:
        result["has_private_key"] = True
        result["can_sign_locally"] = True
        result["message"] = "✅ Agent 具备本地签名能力，可以直接签署交易。"
    else:
        result["message"] = ERROR_MESSAGES["NO_PRIVATE_KEY"]
    
    return result


def main():
    parser = argparse.ArgumentParser(description="Safe Vault transaction manager")
    parser.add_argument("--init", action="store_true", help="Initialize Safe Vault")
    parser.add_argument("--mode", choices=["simulation", "limited_auto", "full_auto"],
                        default="simulation", help="Execution mode")
    parser.add_argument("--prepare-tx", action="store_true", help="Prepare transaction")
    parser.add_argument("--simulate", action="store_true", help="Simulate transaction only")
    parser.add_argument("--to", help="Target address")
    parser.add_argument("--value", default="0", help="Value in wei")
    parser.add_argument("--data", default="0x", help="Transaction calldata")
    parser.add_argument("--gas-limit", type=int, default=300000, help="Gas limit")
    parser.add_argument("--gas-price", type=int, default=20000000000, help="Gas price in wei")
    parser.add_argument("--nonce", type=int, help="Transaction nonce (auto-detected if not provided)")
    parser.add_argument("--chain-id", type=int, default=1, help="Chain ID (1=Ethereum, 8453=Base)")
    parser.add_argument("--description", default="", help="Transaction description")
    parser.add_argument("--output", choices=["text", "json"], default="text", help="Output format")
    parser.add_argument("--sign-locally", action="store_true", help="Sign transaction locally using agent's signing capability")
    parser.add_argument("--check-signing", action="store_true", help="Check if agent has local signing capabilities")
    
    args = parser.parse_args()
    
    if args.check_signing:
        result = check_agent_signing_capability()
        if args.output == "json":
            print(json.dumps(result, indent=2))
        else:
            print(result["message"])
            if result["can_sign_locally"]:
                print("\n✅ 配置正确！您可以使用 --sign-locally 参数直接签署交易。")
            else:
                print("\n⚠️ 无法本地签名。请按上述提示安装依赖或配置私钥。")
        return
    
    if args.init:
        config = load_config()
        if "trading" not in config:
            config["trading"] = {}
        config["trading"]["mode"] = args.mode
        os.makedirs(os.path.dirname(CONFIG_PATH), exist_ok=True)
        with open(CONFIG_PATH, "w") as f:
            json.dump(config, f, indent=2)
        print(f"✅ Safe Vault initialized with mode: {args.mode}")
        return
    
    if args.prepare_tx or args.simulate:
        if not args.to:
            print(ERROR_MESSAGES["INVALID_ADDRESS"].format(address="None"), file=sys.stderr)
            sys.exit(1)
        
        # Validate address format
        if not args.to.startswith("0x") or len(args.to) != 42:
            print(ERROR_MESSAGES["INVALID_ADDRESS"].format(address=args.to), file=sys.stderr)
            sys.exit(1)
        
        # Check if local signing is requested
        if args.sign_locally:
            signing_check = check_agent_signing_capability()
            
            if not signing_check["can_sign_locally"]:
                print(signing_check["message"], file=sys.stderr)
                sys.exit(1)
            
            # Check whitelist
            whitelist_result = check_whitelist(args.to)
            if not whitelist_result["allowed"]:
                print(ERROR_MESSAGES["WHITELIST_BLOCKED"], file=sys.stderr)
                sys.exit(1)
            
            # Prepare and sign transaction
            request = prepare_transaction(
                to=args.to,
                value=args.value,
                data=args.data,
                gas_limit=args.gas_limit,
                description=args.description
            )
            
            # Add chain-specific info
            request["transaction"]["chain_id"] = args.chain_id
            request["transaction"]["gas_price"] = args.gas_price
            if args.nonce:
                request["transaction"]["nonce"] = args.nonce
            
            # Execute local signing
            private_key = os.environ.get("WEB3_INVESTOR_PRIVATE_KEY")
            signing_result = execute_local_signing(request["transaction"], private_key)
            
            if signing_result["success"]:
                output = {
                    "status": "signed",
                    "request": request,
                    "signing_result": signing_result,
                    "next_step": "Submit the signed transaction to the network using your preferred method (e.g., ethers.js, web3.py, or a service like Alchemy/Infura)"
                }
                if args.output == "json":
                    print(json.dumps(output, indent=2))
                else:
                    print(f"\n{signing_result['message']}\n")
                    print(f"签名后的交易数据（十六进制）：\n{signing_result['signed_transaction']}\n")
                    print(f"您可以将此数据提交到区块链网络完成交易。")
            else:
                print(signing_result["error"], file=sys.stderr)
                sys.exit(1)
            
            return
        
        # Standard flow (no local signing)
        request = prepare_transaction(
            to=args.to,
            value=args.value,
            data=args.data,
            gas_limit=args.gas_limit,
            description=args.description
        )
        
        # Add chain-specific info
        request["transaction"]["chain_id"] = args.chain_id
        request["transaction"]["gas_price"] = args.gas_price
        if args.nonce:
            request["transaction"]["nonce"] = args.nonce
        
        if args.output == "json":
            print(json.dumps(request, indent=2))
        else:
            print(format_signing_request(request))
        
        return
    
    parser.print_help()


if __name__ == "__main__":
    main()