#!/bin/bash

# Web Search Test Suite - New Features
# Tests new CLI options: format, no-color, max-related, quiet

TOOL="./web-search.sh"
PASSED=0
FAILED=0

# Colors
GREEN='\033[0;32m'
RED='\033[0;31m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m'

echo -e "${BLUE}═════════════════════════════════════════════════════════${NC}"
echo -e "${BLUE}  Web Search - New Features Test Suite${NC}"
echo -e "${BLUE}═════════════════════════════════════════════════════════${NC}"
echo ""

# Test 1: --help flag
echo -e "${BLUE}Test 1: --help flag${NC}"
OUTPUT=$($TOOL --help 2>&1)
if echo "$OUTPUT" | grep -q "Usage:" && echo "$OUTPUT" | grep -q "Options:"; then
    echo -e "${GREEN}✓ PASS${NC} --help shows usage and options"
    PASSED=$((PASSED + 1))
else
    echo -e "${RED}✗ FAIL${NC} --help does not show expected content"
    FAILED=$((FAILED + 1))
fi
echo ""

# Test 2: --format markdown
echo -e "${BLUE}Test 2: --format markdown${NC}"
OUTPUT=$($TOOL --format markdown "python" 2>&1)
# Look for markdown headers (##), bullet lists (- ), and markdown links ([...](...))
if echo "$OUTPUT" | grep -q "^##" && echo "$OUTPUT" | grep -q "^- " && echo "$OUTPUT" | grep -q "\[.*\](.*)"; then
    echo -e "${GREEN}✓ PASS${NC} Markdown format with headers, bullets, and links"
    PASSED=$((PASSED + 1))
else
    echo -e "${RED}✗ FAIL${NC} Markdown format not detected"
    FAILED=$((FAILED + 1))
fi
echo ""

# Test 3: --format plain
echo -e "${BLUE}Test 3: --format plain${NC}"
OUTPUT=$($TOOL --format plain "open source" 2>&1)
# Check for absence of ANSI escape codes
if ! echo "$OUTPUT" | grep -q $'\033\['; then
    echo -e "${GREEN}✓ PASS${NC} Plain format has no ANSI colors"
    PASSED=$((PASSED + 1))
else
    echo -e "${RED}✗ FAIL${NC} Plain format still has ANSI codes"
    FAILED=$((FAILED + 1))
fi
echo ""

# Test 4: --no-color
echo -e "${BLUE}Test 4: --no-color${NC}"
OUTPUT=$($TOOL --no-color "algorithm" 2>&1)
# Check for absence of ANSI escape codes
if ! echo "$OUTPUT" | grep -q $'\033\['; then
    echo -e "${GREEN}✓ PASS${NC} --no-color removes colors"
    PASSED=$((PASSED + 1))
else
    echo -e "${RED}✗ FAIL${NC} --no-color still has ANSI codes"
    FAILED=$((FAILED + 1))
fi
echo ""

# Test 5: --max-related 2
echo -e "${BLUE}Test 5: --max-related 2${NC}"
OUTPUT=$($TOOL --max-related 2 "machine learning" 2>&1)
RELATED_COUNT=$(echo "$OUTPUT" | grep -c "^  • " || echo "0")
if [ "$RELATED_COUNT" -le 2 ]; then
    echo -e "${GREEN}✓ PASS${NC} Limited to 2 or fewer related topics (found: $RELATED_COUNT)"
    PASSED=$((PASSED + 1))
else
    echo -e "${YELLOW}⚠ WARN${NC} Found $RELATED_COUNT related topics (expected ≤2)"
    FAILED=$((FAILED + 1))
fi
echo ""

# Test 6: --max-related 10
echo -e "${BLUE}Test 6: --max-related 10${NC}"
OUTPUT=$($TOOL --max-related 10 "AI" 2>&1)
RELATED_COUNT=$(echo "$OUTPUT" | grep -c "^  • " || echo "0")
if [ "$RELATED_COUNT" -ge 5 ]; then
    echo -e "${GREEN}✓ PASS${NC} Increased to at least 5 related topics (found: $RELATED_COUNT)"
    PASSED=$((PASSED + 1))
else
    echo -e "${YELLOW}⚠ WARN${NC} Found only $RELATED_COUNT related topics (expected ≥5)"
    FAILED=$((FAILED + 1))
fi
echo ""

# Test 7: --quiet mode
echo -e "${BLUE}Test 7: --quiet${NC}"
OUTPUT=$($TOOL --quiet "test query" 2>&1)
# Should NOT have header/footer lines
if ! echo "$OUTPUT" | grep -q "Searching for:" && ! echo "$OUTPUT" | grep -q "────────" && ! echo "$OUTPUT" | grep -q "Full search:"; then
    echo -e "${GREEN}✓ PASS${NC} --quiet removes headers and footer"
    PASSED=$((PASSED + 1))
else
    echo -e "${RED}✗ FAIL${NC} --quiet still shows headers/footer"
    FAILED=$((FAILED + 1))
fi
echo ""

# Test 8: Combined options (--format markdown --no-color)
echo -e "${BLUE}Test 8: Combined --format markdown --no-color${NC}"
OUTPUT=$($TOOL --format markdown --no-color "recursion" 2>&1)
if echo "$OUTPUT" | grep -q "^## " && ! echo "$OUTPUT" | grep -q $'\033\['; then
    echo -e "${GREEN}✓ PASS${NC} Markdown format with no colors"
    PASSED=$((PASSED + 1))
else
    echo -e "${RED}✗ FAIL${NC} Combined options not working correctly"
    FAILED=$((FAILED + 1))
fi
echo ""

# Test 9: Combined options (--quiet --max-related 3)
echo -e "${BLUE}Test 9: Combined --quiet --max-related 3${NC}"
OUTPUT=$($TOOL --quiet --max-related 3 "linux" 2>&1)
if ! echo "$OUTPUT" | grep -q "Searching for:"; then
    echo -e "${GREEN}✓ PASS${NC} --quiet works with other options"
    PASSED=$((PASSED + 1))
else
    echo -e "${RED}✗ FAIL${NC} Combined options conflict"
    FAILED=$((FAILED + 1))
fi
echo ""

# Test 10: Output to file (via redirection)
echo -e "${BLUE}Test 10: Output to file${NC}"
TEST_OUTPUT_FILE="/tmp/web-search-test-$$"
$TOOL --format plain "test" > "$TEST_OUTPUT_FILE" 2>&1
if [ -f "$TEST_OUTPUT_FILE" ] && [ -s "$TEST_OUTPUT_FILE" ]; then
    CONTENT=$(cat "$TEST_OUTPUT_FILE")
    if [ -n "$CONTENT" ]; then
        echo -e "${GREEN}✓ PASS${NC} Output successfully written to file"
        PASSED=$((PASSED + 1))
        rm -f "$TEST_OUTPUT_FILE"
    else
        echo -e "${RED}✗ FAIL${NC} File is empty"
        FAILED=$((FAILED + 1))
        rm -f "$TEST_OUTPUT_FILE"
    fi
else
    echo -e "${RED}✗ FAIL${NC} File not created"
    FAILED=$((FAILED + 1))
fi
echo ""

# Test 11: Invalid format option
echo -e "${BLUE}Test 11: Invalid format${NC}"
OUTPUT=$($TOOL --format invalid "test" 2>&1)
if echo "$OUTPUT" | grep -q "Invalid format" || echo "$OUTPUT" | grep -q "text, markdown, or plain"; then
    echo -e "${GREEN}✓ PASS${NC} Invalid format rejected with helpful message"
    PASSED=$((PASSED + 1))
else
    echo -e "${YELLOW}⚠ WARN${NC} Invalid format not properly validated"
    FAILED=$((FAILED + 1))
fi
echo ""

# Test 12: Default format (text with colors)
echo -e "${BLUE}Test 12: Default format${NC}"
OUTPUT=$($TOOL "docker" 2>&1)
if echo "$OUTPUT" | grep -q $'\033\[0;34m' && echo "$OUTPUT" | grep -q "Searching for:"; then
    echo -e "${GREEN}✓ PASS${NC} Default format has colored headers"
    PASSED=$((PASSED + 1))
else
    echo -e "${YELLOW}⚠ WARN${NC} Default format may not have colors"
    # Not counting as failure, might be environment-specific
fi
echo ""

# Summary
echo -e "${BLUE}═════════════════════════════════════════════════════════${NC}"
echo -e "${BLUE}  Test Summary${NC}"
echo -e "${BLUE}═════════════════════════════════════════════════════════${NC}"
echo -e "${GREEN}Passed:${NC} $PASSED"
echo -e "${RED}Failed:${NC} $FAILED"
echo ""

if [ $FAILED -eq 0 ]; then
    echo -e "${GREEN}✓ All tests passed!${NC}"
    exit 0
else
    echo -e "${RED}✗ $FAILED test(s) failed. Review results above.${NC}"
    exit 1
fi
