#!/bin/bash

# Web Search Tool
# Uses DuckDuckGo Instant Answer API (no API key required)
# Usage: ./web-search.sh [options] "your search query"

set -e

API_BASE="https://api.duckduckgo.com"

# Default values
MAX_RELATED=5
OUTPUT_FORMAT="text"
QUIET_MODE=false
USE_COLOR=true

# Colors for output
GREEN='\033[0;32m'
BLUE='\033[0;34m'
YELLOW='\033[1;33m'
RED='\033[0;31m'
NC='\033[0m' # No Color

# Function to display usage
usage() {
    if [ "$USE_COLOR" = true ]; then
        echo -e "${YELLOW}Web Search Tool${NC}"
    else
        echo "Web Search Tool"
    fi
    echo "Usage: $0 [options] \"your search query\""
    echo ""
    echo "Options:"
    echo "  -h, --help              Show this help message"
    echo "  --format <format>       Output format: text, markdown, plain (default: text)"
    echo "  --no-color              Disable colored output"
    echo "  --max-related <N>      Number of related topics to show (default: 5)"
    echo "  --quiet                Minimal output (just results, no headers/footer)"
    echo ""
    echo "Output to file:"
    echo "  $0 \"query\" > output.txt"
    echo ""
    echo "Examples:"
    echo "  $0 \"open source AI models\""
    echo "  $0 --format markdown \"AI research\""
    echo "  $0 --max-related 10 \"machine learning\""
    echo "  $0 --quiet \"what is 2+2\""
    echo "  $0 --format markdown --no-color \"search query\" > results.md"
    echo ""
    echo "For more information, visit: https://clawhub.ai/skills/web-search-instant"
    exit 0
}

# Parse command-line arguments
QUERY_ARGS=()
while [[ $# -gt 0 ]]; do
    case $1 in
        -h|--help)
            usage
            ;;
        --format)
            OUTPUT_FORMAT="$2"
            shift 2
            ;;
        --no-color)
            USE_COLOR=false
            shift
            ;;
        --max-related)
            MAX_RELATED="$2"
            shift 2
            ;;
        --quiet)
            QUIET_MODE=true
            shift
            ;;
        *)
            QUERY_ARGS+=("$1")
            shift
            ;;
    esac
done

# Validate format option
case "$OUTPUT_FORMAT" in
    text|markdown|plain)
        ;;
    *)
        echo "Error: Invalid format '$OUTPUT_FORMAT'. Use: text, markdown, or plain" >&2
        exit 1
        ;;
esac

# Disable colors for markdown/plain format or if --no-color
if [ "$OUTPUT_FORMAT" = "plain" ] || [ "$OUTPUT_FORMAT" = "markdown" ] || [ "$USE_COLOR" = false ]; then
    GREEN=""
    BLUE=""
    YELLOW=""
    RED=""
    NC=""
fi

# Check if query is provided
if [ ${#QUERY_ARGS[@]} -eq 0 ]; then
    usage
fi

# Get the search query (join all arguments with spaces)
QUERY="${QUERY_ARGS[*]}"

# URL encode the query
ENCODED_QUERY=$(echo "$QUERY" | jq -sRr @uri 2>/dev/null || python3 -c "import urllib.parse, sys; print(urllib.parse.quote(sys.argv[1]))" "$QUERY")

if [ -z "$ENCODED_QUERY" ]; then
    if [ "$USE_COLOR" = true ]; then
        echo -e "${RED}Error: Failed to encode query${NC}" >&2
    else
        echo "Error: Failed to encode query" >&2
    fi
    exit 1
fi

# Build the API URL
API_URL="${API_BASE}/?q=${ENCODED_QUERY}&format=json&no_html=1&skip_disambig=0"

# Only show header if not quiet mode
if [ "$QUIET_MODE" = false ]; then
    if [ "$USE_COLOR" = true ]; then
        echo -e "${BLUE}Searching for:${NC} ${QUERY}"
        echo -e "${BLUE}───────────────────────────────────────${NC}"
    else
        echo "Searching for: ${QUERY}"
        echo "───────────────────────────────────────"
    fi
    echo ""
fi

# Fetch results
if command -v curl &> /dev/null; then
    RESPONSE=$(curl -sL --max-time 10 "$API_URL")
elif command -v wget &> /dev/null; then
    RESPONSE=$(wget -qO- --timeout=10 "$API_URL")
else
    if [ "$USE_COLOR" = true ]; then
        echo -e "${RED}Error: Neither curl nor wget is installed${NC}" >&2
    else
        echo "Error: Neither curl nor wget is installed" >&2
    fi
    exit 1
fi

# Check if we got a response
if [ -z "$RESPONSE" ]; then
    if [ "$USE_COLOR" = true ]; then
        echo -e "${RED}Error: No response from DuckDuckGo API${NC}" >&2
    else
        echo "Error: No response from DuckDuckGo API" >&2
    fi
    echo "This could be a network issue or the API is temporarily unavailable."
    exit 1
fi

# Parse JSON using jq if available, or fallback to basic text extraction
if command -v jq &> /dev/null; then
    # Check for Abstract (direct answer)
    ABSTRACT=$(echo "$RESPONSE" | jq -r '.Abstract // empty' | tr -d '\r\n')
    
    # Check for AbstractText (alternative field)
    if [ -z "$ABSTRACT" ]; then
        ABSTRACT=$(echo "$RESPONSE" | jq -r '.AbstractText // empty' | tr -d '\r\n')
    fi
    
    # Check for AbstractSource
    ABSTRACT_SOURCE=$(echo "$RESPONSE" | jq -r '.AbstractSource // empty' | tr -d '\r\n')
    
    # Check for AbstractURL
    ABSTRACT_URL=$(echo "$RESPONSE" | jq -r '.AbstractURL // empty' | tr -d '\r\n')
    
    # Check for RelatedTopics (for additional results)
    RELATED_TOPICS=$(echo "$RESPONSE" | jq -r '.RelatedTopics[]?.Text // empty' 2>/dev/null | head -5)
    
    # Check for Answer
    ANSWER=$(echo "$RESPONSE" | jq -r '.Answer // empty' | tr -d '\r\n')
    
    # Check for AnswerType
    ANSWER_TYPE=$(echo "$RESPONSE" | jq -r '.AnswerType // empty' | tr -d '\r\n')
    
    # Check for Definition
    DEFINITION=$(echo "$RESPONSE" | jq -r '.Definition // empty' | tr -d '\r\n')
    
    # Check for Heading
    HEADING=$(echo "$RESPONSE" | jq -r '.Heading // empty' | tr -d '\r\n')
else
    # Fallback: Extract data using grep and sed (less reliable)
    echo -e "${YELLOW}Warning: jq not found. Using basic parsing (install jq for better results).${NC}" >&2

    # Extract JSON values more carefully
    ABSTRACT=$(echo "$RESPONSE" | grep -oP '(?<="Abstract":")[^"]*' | head -1 | tr -d '\\r\\n')
    ABSTRACT_SOURCE=$(echo "$RESPONSE" | grep -oP '(?<="AbstractSource":")[^"]*' | head -1)
    ABSTRACT_URL=$(echo "$RESPONSE" | grep -oP '(?<="AbstractURL":")[^"]*' | head -1)
    ANSWER=$(echo "$RESPONSE" | grep -oP '(?<="Answer":")[^"]*' | head -1 | tr -d '\\r\\n')
    DEFINITION=$(echo "$RESPONSE" | grep -oP '(?<="Definition":")[^"]*' | head -1 | tr -d '\\r\\n')
    HEADING=$(echo "$RESPONSE" | grep -oP '(?<="Heading":")[^"]*' | head -1)

    # Extract RelatedTopics (basic extraction)
    RELATED_TOPICS=$(echo "$RESPONSE" | grep -oP '(?<="Text":")[^"]*' | head -5)
fi

# Display results
RESULT_COUNT=0

# Display Answer (direct answers for calculations, conversions, etc.)
if [ -n "$ANSWER" ] && [ "$ANSWER" != "" ]; then
    if [ "$OUTPUT_FORMAT" = "markdown" ]; then
        echo "## Answer"
        echo ""
        echo "$ANSWER"
        echo ""
    elif [ "$USE_COLOR" = true ]; then
        echo -e "${GREEN}▸ Answer:${NC}"
        echo -e "  $ANSWER"
        echo ""
    else
        echo "Answer:"
        echo "  $ANSWER"
        echo ""
    fi
    RESULT_COUNT=$((RESULT_COUNT + 1))
fi

# Display Abstract (main result)
if [ -n "$ABSTRACT" ] && [ "$ABSTRACT" != "" ]; then
    if [ "$OUTPUT_FORMAT" = "markdown" ]; then
        echo "## Result"
        echo ""
        if [ -n "$HEADING" ] && [ "$HEADING" != "" ]; then
            echo "**$HEADING**"
            echo ""
        fi
        echo "$ABSTRACT"
        echo ""
        if [ -n "$ABSTRACT_SOURCE" ] && [ "$ABSTRACT_SOURCE" != "" ]; then
            echo "*Source: $ABSTRACT_SOURCE*"
        fi
        if [ -n "$ABSTRACT_URL" ] && [ "$ABSTRACT_URL" != "" ]; then
            echo "*URL: $ABSTRACT_URL*"
        fi
        echo ""
    elif [ "$USE_COLOR" = true ]; then
        echo -e "${GREEN}▸ Result:${NC}"
        if [ -n "$HEADING" ] && [ "$HEADING" != "" ]; then
            echo -e "  ${BLUE}$HEADING${NC}"
        fi
        echo -e "  $ABSTRACT"
        if [ -n "$ABSTRACT_SOURCE" ] && [ "$ABSTRACT_SOURCE" != "" ]; then
            echo -e "  ${YELLOW}Source:${NC} $ABSTRACT_SOURCE"
        fi
        if [ -n "$ABSTRACT_URL" ] && [ "$ABSTRACT_URL" != "" ]; then
            echo -e "  ${YELLOW}URL:${NC} $ABSTRACT_URL"
        fi
        echo ""
    else
        echo "Result:"
        if [ -n "$HEADING" ] && [ "$HEADING" != "" ]; then
            echo "  $HEADING"
        fi
        echo "  $ABSTRACT"
        if [ -n "$ABSTRACT_SOURCE" ] && [ "$ABSTRACT_SOURCE" != "" ]; then
            echo "  Source: $ABSTRACT_SOURCE"
        fi
        if [ -n "$ABSTRACT_URL" ] && [ "$ABSTRACT_URL" != "" ]; then
            echo "  URL: $ABSTRACT_URL"
        fi
        echo ""
    fi
    RESULT_COUNT=$((RESULT_COUNT + 1))
fi

# Display Definition
if [ -n "$DEFINITION" ] && [ "$DEFINITION" != "" ]; then
    if [ "$OUTPUT_FORMAT" = "markdown" ]; then
        echo "## Definition"
        echo ""
        echo "$DEFINITION"
        echo ""
    elif [ "$USE_COLOR" = true ]; then
        echo -e "${GREEN}▸ Definition:${NC}"
        echo -e "  $DEFINITION"
        echo ""
    else
        echo "Definition:"
        echo "  $DEFINITION"
        echo ""
    fi
    RESULT_COUNT=$((RESULT_COUNT + 1))
fi

# Display Related Topics
if [ -n "$RELATED_TOPICS" ] && [ "$RELATED_TOPICS" != "" ]; then
    if [ "$OUTPUT_FORMAT" = "markdown" ]; then
        echo "## Related"
        echo ""
    elif [ "$USE_COLOR" = true ]; then
        echo -e "${GREEN}▸ Related:${NC}"
    else
        echo "Related:"
    fi
    echo "$RELATED_TOPICS" | while read -r topic; do
        if [ -n "$topic" ] && [ "$topic" != "" ]; then
            # Remove HTML links if present
            topic_clean=$(echo "$topic" | sed -E 's/<a[^>]*href="([^"]*)"[^>]*>([^<]*)<\/a>/\2 (\1)/g' | sed 's/<[^>]*>//g')
            if [ "$OUTPUT_FORMAT" = "markdown" ]; then
                echo "- $topic_clean"
            else
                echo -e "  • $topic_clean"
            fi
        fi
    done | head -n "$MAX_RELATED"
    echo ""
    RESULT_COUNT=$((RESULT_COUNT + 1))
fi

# If no results found
if [ $RESULT_COUNT -eq 0 ]; then
    if [ "$OUTPUT_FORMAT" = "markdown" ]; then
        echo "## No results found"
        echo ""
        echo "Try:"
        echo "- Rephrasing your query"
        echo "- Using more specific terms"
        echo "- Checking spelling"
        echo ""
        echo "For full web results: https://duckduckgo.com/?q=${ENCODED_QUERY}"
    elif [ "$USE_COLOR" = true ]; then
        echo -e "${YELLOW}No direct results found.${NC}"
        echo ""
        echo -e "${BLUE}Try:${NC}"
        echo "  • Rephrasing your query"
        echo "  • Using more specific terms"
        echo "  • Checking spelling"
        echo ""
        echo -e "${BLUE}For full web results, visit:${NC}"
        echo "  https://duckduckgo.com/?q=${ENCODED_QUERY}"
    else
        echo "No direct results found."
        echo ""
        echo "Try:"
        echo "  - Rephrasing your query"
        echo "  - Using more specific terms"
        echo "  - Checking spelling"
        echo ""
        echo "For full web results: https://duckduckgo.com/?q=${ENCODED_QUERY}"
    fi
fi

# Only show footer if not quiet mode
if [ "$QUIET_MODE" = false ]; then
    if [ "$OUTPUT_FORMAT" = "markdown" ]; then
        echo ""
        echo "---"
        echo ""
        echo "[Full search](https://duckduckgo.com/?q=${ENCODED_QUERY})"
    elif [ "$USE_COLOR" = true ]; then
        echo -e "${BLUE}───────────────────────────────────────${NC}"
        echo -e "${BLUE}Full search:${NC} https://duckduckgo.com/?q=${ENCODED_QUERY}"
    else
        echo "───────────────────────────────────────"
        echo "Full search: https://duckduckgo.com/?q=${ENCODED_QUERY}"
    fi
fi

# Note: For file output, use shell redirection:
# ./web-search.sh "query" > output.txt
