#!/bin/bash

# Web Search Test Suite
# Tests various query types and edge cases

TOOL="./web-search.sh"
PASSED=0
FAILED=0
WARNINGS=0

# Colors
GREEN='\033[0;32m'
RED='\033[0;31m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m'

echo -e "${BLUE}═══════════════════════════════════════════════════════════${NC}"
echo -e "${BLUE}  Web Search Tool - Test Suite${NC}"
echo -e "${BLUE}═══════════════════════════════════════════════════════════${NC}"
echo ""

run_test() {
    local name="$1"
    local query="$2"
    local expected="$3"

    echo -e "${BLUE}Test:${NC} $name"
    echo -e "${BLUE}Query:${NC} \"$query\""
    echo ""

    # Run the tool
    OUTPUT=$($TOOL "$query" 2>&1)
    EXIT_CODE=$?

    # Check exit code
    if [ $EXIT_CODE -eq 0 ]; then
        echo -e "${GREEN}✓${NC} Exit code: OK"
    else
        echo -e "${RED}✗${NC} Exit code: $EXIT_CODE (expected 0)"
        FAILED=$((FAILED + 1))
        return
    fi

    # Check expected content
    if [ -n "$expected" ]; then
        if echo "$OUTPUT" | grep -q "$expected"; then
            echo -e "${GREEN}✓${NC} Contains expected: \"$expected\""
            PASSED=$((PASSED + 1))
        else
            echo -e "${YELLOW}⚠${NC} Warning: Expected \"$expected\" not found (this may be OK for some queries)"
            WARNINGS=$((WARNINGS + 1))
        fi
    else
        PASSED=$((PASSED + 1))
    fi

    echo ""
}

# Test 1: Basic calculation
run_test "Basic Calculation" "2+2" "4"

# Test 2: Wikipedia-style query
run_test "Wikipedia Query" "artificial intelligence" "intelligence"

# Test 3: Definition
run_test "Definition" "define recursion" "definition"

# Test 4: Conversion
run_test "Unit Conversion" "100 miles to km" "km"

# Test 5: Programming question
run_test "Programming" "what is python" "programming language"

# Test 6: Scientific fact
run_test "Scientific Fact" "speed of light" "light"

# Test 7: Historical fact
run_test "Historical Fact" "who won 2024 Olympics" "Olympics"

# Test 8: No results (edge case)
run_test "No Results (Edge Case)" "xyz12345nonexistent" "No direct results found"

# Test 9: Special characters
run_test "Special Characters" "what is C++" "C"

# Test 10: Multi-word query
run_test "Multi-word Query" "how to install docker on ubuntu" "docker"

# Test 11: Celebrity/Person
run_test "Person Query" "who is Elon Musk" "Musk"

# Test 12: Weather (if API supports)
run_test "Weather Query" "weather in Tokyo" "Tokyo"

# Test 13: Math operation
run_test "Math Operation" "sqrt(144)" "12"

# Test 14: Percentage calculation
run_test "Percentage" "10% of 500" "50"

# Test 15: Empty query (should show usage)
echo -e "${BLUE}Test: Empty Query (Edge Case)${NC}"
echo -e "${BLUE}Query:${NC} (empty)"
echo ""
OUTPUT=$($TOOL 2>&1)
if echo "$OUTPUT" | grep -q "Usage:"; then
    echo -e "${GREEN}✓${NC} Shows usage message"
    PASSED=$((PASSED + 1))
else
    echo -e "${RED}✗${NC} Does not show usage message"
    FAILED=$((FAILED + 1))
fi
echo ""

# Summary
echo -e "${BLUE}═══════════════════════════════════════════════════════════${NC}"
echo -e "${BLUE}  Test Summary${NC}"
echo -e "${BLUE}═══════════════════════════════════════════════════════════${NC}"
echo -e "${GREEN}Passed:${NC} $PASSED"
echo -e "${RED}Failed:${NC} $FAILED"
echo -e "${YELLOW}Warnings:${NC} $WARNINGS"
echo ""

if [ $FAILED -eq 0 ]; then
    echo -e "${GREEN}✓ All critical tests passed!${NC}"
    exit 0
else
    echo -e "${RED}✗ Some tests failed. Review results above.${NC}"
    exit 1
fi
