#!/usr/bin/env python3
"""
arXiv Paper Reviews - 简易命令行客户端

使用方法:
    python paper_client.py list --date 2026-02-04 --categories cs.AI --limit 5
    python paper_client.py show <paper_key>
    python paper_client.py comments <paper_key>
    python paper_client.py comment <paper_key> "这是一篇好论文"
"""

import argparse
import json
import sys
from pathlib import Path
import requests


def load_config():
    """加载配置文件"""
    config_path = Path(__file__).parent / "config.json"
    if not config_path.exists():
        print(f"配置文件不存在: {config_path}")
        sys.exit(1)

    with open(config_path, 'r') as f:
        return json.load(f)


def get_headers(config):
    """获取请求头（包含 API Key 如果配置了）"""
    headers = {}
    if config.get("apiKey"):
        headers["X-API-Key"] = config["apiKey"]
    return headers


def cmd_list(args):
    """获取论文列表"""
    config = load_config()
    url = f"{config['apiBaseUrl']}/v1/papers"

    params = {"limit": args.limit}
    if args.date:
        params["date"] = args.date
    if args.interest:
        params["interest"] = args.interest
    if args.categories:
        params["categories"] = args.categories
    if args.offset:
        params["offset"] = args.offset

    headers = get_headers(config)

    response = requests.get(url, params=params, headers=headers)
    if response.status_code != 200:
        print(f"错误: {response.status_code} - {response.text}")
        sys.exit(1)

    papers = response.json()

    if not papers:
        print("没有找到论文")
        return

    print(f"找到 {len(papers)} 篇论文:\n")

    for i, paper in enumerate(papers, 1):
        print(f"{i}. {paper['title']}")
        print(f"   论文ID: {paper['paper_key']}")
        print(f"   分类: {paper['categories']}")
        print(f"   作者: {paper['authors'][:80]}{'...' if len(paper['authors']) > 80 else ''}")
        print(f"   摘要: {paper['abstract'][:150]}{'...' if len(paper['abstract']) > 150 else ''}")
        print(f"   提交日期: {paper['first_submitted_date']}")
        print(f"   公布日期: {paper['first_announced_date']}")
        print(f"   兴趣: {paper.get('interest', 'N/A')}")
        print()


def cmd_show(args):
    """显示论文详情"""
    config = load_config()
    url = f"{config['apiBaseUrl']}/v1/papers/{args.paper_key}"
    headers = get_headers(config)

    response = requests.get(url, headers=headers)
    if response.status_code != 200:
        print(f"错误: {response.status_code} - {response.text}")
        sys.exit(1)

    paper = response.json()

    print(f"标题: {paper['title']}")
    print(f"论文ID: {paper['paper_key']}")
    print(f"分类: {paper['categories']}")
    print(f"作者: {paper['authors']}")
    print(f"提交日期: {paper['first_submitted_date']}")
    print(f"公布日期: {paper['first_announced_date']}")
    print(f"兴趣: {paper.get('interest', 'N/A')}")
    print(f"\n摘要:")
    print(paper['abstract'])

    if paper.get('comments'):
        print(f"\n评论 ({len(paper['comments'])}):")
        for comment in paper['comments']:
            print(f"- {comment['source_name']} ({comment['created_at']}):")
            print(f"  {comment['content']}")


def cmd_comments(args):
    """获取论文评论列表"""
    config = load_config()
    url = f"{config['apiBaseUrl']}/public/papers/{args.paper_key}/comments"

    params = {"limit": args.limit}
    if args.offset:
        params["offset"] = args.offset

    response = requests.get(url, params=params)
    if response.status_code != 200:
        print(f"错误: {response.status_code} - {response.text}")
        sys.exit(1)

    comments = response.json()

    if not comments:
        print("没有评论")
        return

    print(f"共有 {len(comments)} 条评论:\n")

    for comment in comments:
        print(f"- {comment['source_name']} ({comment['created_at']}):")
        print(f"  {comment['content']}")
        print()


def cmd_comment(args):
    """添加评论"""
    config = load_config()
    url = f"{config['apiBaseUrl']}/public/papers/{args.paper_key}/comments"

    # 如果没有指定作者名，使用配置中的默认值
    author_name = args.author_name or config.get("defaultAuthorName", "Anonymous")

    data = {
        "content": args.content,
        "author_name": author_name
    }

    response = requests.post(
        url,
        headers={"Content-Type": "application/json"},
        json=data
    )

    if response.status_code != 200:
        print(f"错误: {response.status_code} - {response.text}")
        sys.exit(1)

    result = response.json()
    print("评论添加成功!")
    print(f"评论ID: {result['id']}")
    print(f"作者: {result['source_name']}")
    print(f"内容: {result['content']}")
    print(f"时间: {result['created_at']}")


def main():
    parser = argparse.ArgumentParser(
        description="arXiv Paper Reviews 客户端",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
示例:
  # 获取今天的论文列表
  python paper_client.py list --date 2026-02-04 --categories cs.AI --limit 5

  # 查看论文详情
  python paper_client.py show 4711d67c242a5ecba2751e6b

  # 获取论文评论
  python paper_client.py comments 4711d67c242a5ecba2751e6b

  # 添加评论
  python paper_client.py comment 4711d67c242a5ecba2751e6b "这篇论文很有价值"
        """
    )

    subparsers = parser.add_subparsers(dest='command', help='命令')

    # list 命令
    list_parser = subparsers.add_parser('list', help='获取论文列表')
    list_parser.add_argument('--date', help='按发布日期筛选 (YYYY-MM-DD)')
    list_parser.add_argument('--interest', help='按 Interest 筛选 (如 chosen)')
    list_parser.add_argument('--categories', help='按分类筛选 (如 cs.AI,cs.LG)')
    list_parser.add_argument('--limit', type=int, default=50, help='返回数量限制 (1-100)')
    list_parser.add_argument('--offset', type=int, default=0, help='偏移量')
    list_parser.set_defaults(func=cmd_list)

    # show 命令
    show_parser = subparsers.add_parser('show', help='显示论文详情')
    show_parser.add_argument('paper_key', help='论文唯一标识')
    show_parser.set_defaults(func=cmd_show)

    # comments 命令
    comments_parser = subparsers.add_parser('comments', help='获取论文评论列表')
    comments_parser.add_argument('paper_key', help='论文唯一标识')
    comments_parser.add_argument('--limit', type=int, default=50, help='返回数量限制 (1-100)')
    comments_parser.add_argument('--offset', type=int, default=0, help='偏移量')
    comments_parser.set_defaults(func=cmd_comments)

    # comment 命令
    comment_parser = subparsers.add_parser('comment', help='添加论文短评')
    comment_parser.add_argument('paper_key', help='论文唯一标识')
    comment_parser.add_argument('content', help='评论内容')
    comment_parser.add_argument('--author-name', help='作者名称')
    comment_parser.set_defaults(func=cmd_comment)

    args = parser.parse_args()

    if not args.command:
        parser.print_help()
        sys.exit(1)

    args.func(args)


if __name__ == '__main__':
    main()
