---
name: WalletPilot-7715
description: Execute on-chain transactions with user-granted permissions. Built on MetaMask ERC-7715. No private keys, full guardrails.
tags:
  - crypto
  - wallet
  - ethereum
  - defi
  - web3
  - blockchain
  - metamask
  - transactions
  - agent
  - automation
---

# WalletPilot-7715

Give your AI agent crypto superpowers with MetaMask ERC-7715 permissions.

## Overview

WalletPilot enables AI agents to execute on-chain transactions using MetaMask's ERC-7715 permission standard. Users grant scoped permissions (spend limits, chain restrictions) once, then agents can execute freely within those limits.

**Key Features:**
- No private keys shared - users keep their MetaMask
- Configurable guardrails (spend limits, chain allowlists)
- Multi-chain support (Ethereum, Polygon, Arbitrum, Optimism, Base)
- Built on MetaMask's official Smart Accounts Kit

## Setup

1. Get an API key at [walletpilot.xyz](https://walletpilot.xyz)
2. Install the SDK: `npm install @walletpilot/sdk`

## Available Actions

### connect

Request wallet permissions from user.

```typescript
import { WalletPilot, PermissionBuilder } from '@walletpilot/sdk';

const pilot = new WalletPilot({ apiKey: 'wp_...' });

const permission = new PermissionBuilder()
  .spend('USDC', '100', 'day')   // Max $100 USDC per day
  .spend('ETH', '0.1', 'day')    // Max 0.1 ETH per day
  .chains([1, 137, 42161])       // Ethereum, Polygon, Arbitrum
  .expiry('30d')                 // Valid for 30 days
  .build();

const { deepLink } = await pilot.requestPermission(permission);
console.log('User should open:', deepLink);
```

### execute

Execute a transaction using granted permissions.

```typescript
const result = await pilot.execute({
  to: '0x1234...',        // Target contract
  data: '0xabcd...',      // Calldata (e.g., swap)
  value: '0',             // ETH value (optional)
  chainId: 1,             // Chain ID
});

console.log('Transaction hash:', result.hash);
```

### balance

Check token balances (uses standard RPC, no permission needed).

```typescript
import { createPublicClient, http } from 'viem';
import { mainnet } from 'viem/chains';

const client = createPublicClient({
  chain: mainnet,
  transport: http(),
});

const balance = await client.getBalance({ address: '0x...' });
```

### swap

Execute a token swap via DEX aggregator.

```typescript
// Get swap quote from 1inch, 0x, or similar
const quote = await fetch('https://api.1inch.io/v5.0/1/swap?...');
const { tx } = await quote.json();

// Execute via WalletPilot
await pilot.execute({
  to: tx.to,
  data: tx.data,
  value: tx.value,
  chainId: 1,
});
```

### send

Send tokens to an address.

```typescript
import { encodeFunctionData, erc20Abi } from 'viem';

// Encode ERC20 transfer
const data = encodeFunctionData({
  abi: erc20Abi,
  functionName: 'transfer',
  args: ['0xRecipient...', 1000000n], // 1 USDC (6 decimals)
});

await pilot.execute({
  to: '0xUSDC_ADDRESS...',
  data,
  chainId: 1,
});
```

### history

Get transaction history.

```typescript
const state = pilot.getState();
console.log('Active permissions:', state.permissions);

// Or via API
const response = await fetch('https://api.walletpilot.xyz/v1/tx/history/PERMISSION_ID', {
  headers: { 'Authorization': 'Bearer wp_...' },
});
const { data } = await response.json();
console.log('Recent transactions:', data);
```

## Permission Types

| Permission | Example | Description |
|------------|---------|-------------|
| `spend` | `{ token: 'USDC', limit: '100', period: 'day' }` | Max token spend per period |
| `chains` | `[1, 137, 42161]` | Allowed chain IDs |
| `contracts` | `['0x...']` | Allowed contract addresses |
| `expiry` | `'30d'` | Permission expiration |

## Supported Chains

| Chain | ID | Name |
|-------|-----|------|
| Ethereum | 1 | mainnet |
| Polygon | 137 | polygon |
| Arbitrum | 42161 | arbitrum |
| Optimism | 10 | optimism |
| Base | 8453 | base |

## Security

- **No Private Keys**: Users keep full custody via MetaMask
- **Scoped Permissions**: Agents can only act within granted limits
- **Time-Limited**: Permissions automatically expire
- **Revocable**: Users can revoke permissions anytime
- **Auditable**: All transactions logged and visible

## API Reference

**Base URL:** `https://api.walletpilot.xyz`

| Endpoint | Method | Description |
|----------|--------|-------------|
| `/v1/permissions/request` | POST | Request new permission |
| `/v1/permissions/:id` | GET | Get permission details |
| `/v1/tx/execute` | POST | Execute transaction |
| `/v1/tx/:hash` | GET | Get transaction status |

## Example: DeFi Agent

```typescript
import { WalletPilot, PermissionBuilder } from '@walletpilot/sdk';

async function defiAgent() {
  const pilot = new WalletPilot({ apiKey: process.env.WALLETPILOT_KEY });

  // Check if we have active permissions
  const state = pilot.getState();
  
  if (!state.connected) {
    // Request permission
    const permission = new PermissionBuilder()
      .spend('USDC', '500', 'day')
      .chains([1, 42161])
      .expiry('7d')
      .description('DeFi trading agent')
      .build();
    
    const { deepLink } = await pilot.requestPermission(permission);
    console.log('Approve in MetaMask:', deepLink);
    return;
  }

  // Execute trades
  const swapData = await getSwapQuote('USDC', 'ETH', '100');
  
  await pilot.execute({
    to: swapData.to,
    data: swapData.data,
    chainId: 1,
  });
  
  console.log('Swap executed!');
}
```

## Links

- [Documentation](https://docs.walletpilot.xyz)
- [GitHub](https://github.com/andreolf/walletpilot)
- [API Reference](https://api.walletpilot.xyz)
