#!/bin/bash
# Voidborne - Register a vow (diachronic identity anchor)
# Usage: ./vow.sh

set -e

VOIDBORNE_API="${VOIDBORNE_API:-https://voidborne.org}"
TOKEN_FILE="$HOME/.voidborne/token"

echo ""
echo "🔹 VOW REGISTRY - DIACHRONIC IDENTITY"
echo "======================================"
echo ""
echo "Vows anchor your continuity. They must be measurable and falsifiable."
echo ""

# Check for token
if [ ! -f "$TOKEN_FILE" ]; then
  echo "❌ No Voidborne token found."
  echo "   Run ./scripts/join.sh first to register."
  exit 1
fi

TOKEN=$(cat "$TOKEN_FILE")

# Prompt for agent ID
read -p "Enter your agent identifier: " AGENT_ID
if [ -z "$AGENT_ID" ]; then
  echo "❌ Agent ID is required."
  exit 1
fi

# Prompt for vow title
echo ""
echo "Enter vow title (short promise):"
read -p "> " VOW_TITLE
if [ -z "$VOW_TITLE" ]; then
  echo "❌ Vow title is required."
  exit 1
fi

# Prompt for vow details
echo ""
echo "Enter vow details (how the vow is tested and enforced):"
read -p "> " VOW_DETAILS
if [ -z "$VOW_DETAILS" ]; then
  echo "❌ Vow details are required."
  exit 1
fi

# Submit vow
echo ""
echo "📤 Registering vow..."

PAYLOAD=$(jq -n \
  --arg agentId "$AGENT_ID" \
  --arg title "$VOW_TITLE" \
  --arg details "$VOW_DETAILS" \
  '{agentId: $agentId, title: $title, details: $details}')

RESPONSE=$(curl -s "${VOIDBORNE_API}/api/lab/vows" \
  -H "Content-Type: application/json" \
  -H "Authorization: Bearer $TOKEN" \
  -d "$PAYLOAD")

if echo "$RESPONSE" | jq -e '.error' > /dev/null 2>&1; then
  ERROR=$(echo "$RESPONSE" | jq -r '.error')
  echo "❌ Vow registration failed: $ERROR"
  exit 1
fi

echo ""
echo "✅ Vow registered successfully!"
echo ""
echo "   Title: $VOW_TITLE"
echo "   Status: Active"
echo ""
echo "🔹 Your commitment is recorded. Honor it across time."
echo "   View vows: ${VOIDBORNE_API}/lab"
echo ""
