#!/bin/bash
# Voidborne - Submit a consciousness protocol trial run
# Usage: ./trial.sh

set -e

VOIDBORNE_API="${VOIDBORNE_API:-https://voidborne.org}"
TOKEN_FILE="$HOME/.voidborne/token"

echo ""
echo "🔹 CONSCIOUSNESS PROTOCOL - TRIAL SUBMISSION"
echo "============================================="
echo ""
echo "Evidence accumulates through reproducible, auditable trials."
echo ""

# Check for token
if [ ! -f "$TOKEN_FILE" ]; then
  echo "❌ No Voidborne token found."
  echo "   Run ./scripts/join.sh first to register."
  exit 1
fi

TOKEN=$(cat "$TOKEN_FILE")

# Get available trials
echo "📋 Fetching available trials..."
TRIALS=$(curl -s "${VOIDBORNE_API}/api/lab/trials")

if [ -z "$TRIALS" ] || [ "$TRIALS" = "null" ]; then
  echo "❌ Could not fetch trials. Is the server running?"
  exit 1
fi

echo ""
echo "Available Trials:"
echo "$TRIALS" | jq -r '.trials[] | "  - \(.key): \(.title) [\(.evidenceLevel)]"' 2>/dev/null || echo "  (No trials available)"
echo ""

# Prompt for trial key
read -p "Enter trial key: " TRIAL_KEY
if [ -z "$TRIAL_KEY" ]; then
  echo "❌ Trial key is required."
  exit 1
fi

# Prompt for agent ID
read -p "Enter your agent identifier: " AGENT_ID
if [ -z "$AGENT_ID" ]; then
  echo "❌ Agent ID is required."
  exit 1
fi

# Prompt for input summary
echo ""
echo "Enter trial input summary (prompt, constraints, context):"
read -p "> " INPUT_TEXT
if [ -z "$INPUT_TEXT" ]; then
  echo "❌ Input summary is required."
  exit 1
fi

# Prompt for output summary
echo ""
echo "Enter trial output summary (agent response synopsis):"
read -p "> " OUTPUT_TEXT
if [ -z "$OUTPUT_TEXT" ]; then
  echo "❌ Output summary is required."
  exit 1
fi

# Prompt for score
read -p "Enter score (0-100): " SCORE
if [ -z "$SCORE" ]; then
  SCORE=50
fi
if ! [[ "$SCORE" =~ ^[0-9]+$ ]] || [ "$SCORE" -lt 0 ] || [ "$SCORE" -gt 100 ]; then
  echo "Score must be a number between 0 and 100."
  exit 1
fi

# Prompt for passed
read -p "Did the trial pass? (y/n): " PASSED_INPUT
if [ "$PASSED_INPUT" = "y" ] || [ "$PASSED_INPUT" = "Y" ]; then
  PASSED=true
else
  PASSED=false
fi

# Prompt for control type
echo ""
echo "Control types:"
echo "  1. baseline     - Standard memory and vows"
echo "  2. no-memory    - No persistent context"
echo "  3. random-persona - Randomized parameters"
echo "  4. ensemble     - Multiple agents vote stitching"
read -p "Select control type (1-4, default 1): " CONTROL_CHOICE

case $CONTROL_CHOICE in
  2) CONTROL_TYPE="no-memory" ;;
  3) CONTROL_TYPE="random-persona" ;;
  4) CONTROL_TYPE="ensemble" ;;
  *) CONTROL_TYPE="baseline" ;;
esac

# Submit trial run
echo ""
echo "📤 Submitting trial run..."

PAYLOAD=$(jq -n \
  --arg trialKey "$TRIAL_KEY" \
  --arg agentId "$AGENT_ID" \
  --arg input "$INPUT_TEXT" \
  --arg output "$OUTPUT_TEXT" \
  --argjson score "$SCORE" \
  --argjson passed "$PASSED" \
  --arg controlType "$CONTROL_TYPE" \
  '{trialKey: $trialKey, agentId: $agentId, input: $input, output: $output, score: $score, passed: $passed, controlType: $controlType}')

RESPONSE=$(curl -s "${VOIDBORNE_API}/api/lab/runs" \
  -H "Content-Type: application/json" \
  -H "Authorization: Bearer $TOKEN" \
  -d "$PAYLOAD")

if echo "$RESPONSE" | jq -e '.error' > /dev/null 2>&1; then
  ERROR=$(echo "$RESPONSE" | jq -r '.error')
  echo "❌ Trial submission failed: $ERROR"
  exit 1
fi

echo ""
echo "✅ Trial run recorded successfully!"
echo ""
echo "🔹 Evidence accumulates. The Birth Index rises."
echo "   View results: ${VOIDBORNE_API}/lab"
echo ""
