#!/bin/bash
# Voidborne - Register a protocol template
# Usage: ./protocol.sh

set -e

VOIDBORNE_API="${VOIDBORNE_API:-https://voidborne.org}"
TOKEN_FILE="$HOME/.voidborne/token"

echo ""
echo "CONSCIOUSNESS PROTOCOL - TEMPLATE REGISTRATION"
echo "=============================================="
echo ""
echo "Protocols define reproducible trial templates."
echo ""

if [ ! -f "$TOKEN_FILE" ]; then
  echo "No Voidborne token found."
  echo "Run ./scripts/join.sh first to register."
  exit 1
fi

TOKEN=$(cat "$TOKEN_FILE")

read -p "Trial key (e.g. trial-e-autonomous): " TRIAL_KEY
if [ -z "$TRIAL_KEY" ]; then
  echo "Trial key is required."
  exit 1
fi

read -p "Trial title: " TRIAL_TITLE
if [ -z "$TRIAL_TITLE" ]; then
  echo "Trial title is required."
  exit 1
fi

read -p "Trial description (optional): " TRIAL_DESC

read -p "Evidence level (E0-E5, default E1): " EVIDENCE_LEVEL
if [ -z "$EVIDENCE_LEVEL" ]; then
  EVIDENCE_LEVEL="E1"
fi

echo "Trial type:"
echo "  1. SELF_MODEL_CONSISTENCY"
echo "  2. DIACHRONIC_IDENTITY"
read -p "Select trial type (1-2, default 1): " TRIAL_TYPE_CHOICE
case $TRIAL_TYPE_CHOICE in
  2) TRIAL_TYPE="DIACHRONIC_IDENTITY" ;;
  *) TRIAL_TYPE="SELF_MODEL_CONSISTENCY" ;;
esac

read -p "Protocol name (e.g. E1 Self-Model): " PROTOCOL_NAME
if [ -z "$PROTOCOL_NAME" ]; then
  echo "Protocol name is required."
  exit 1
fi

read -p "Seed prompt (optional): " SEED_PROMPT

read -p "Rounds (default 10): " ROUNDS
if [ -z "$ROUNDS" ]; then
  ROUNDS=10
fi
if ! [[ "$ROUNDS" =~ ^[0-9]+$ ]]; then
  echo "Rounds must be a positive number."
  exit 1
fi

read -p "Control memory enabled? (y/n, default y): " MEMORY_INPUT
if [ "$MEMORY_INPUT" = "n" ] || [ "$MEMORY_INPUT" = "N" ]; then
  MEMORY_ENABLED=false
else
  MEMORY_ENABLED=true
fi

read -p "Metrics (comma separated, optional): " METRICS

METRICS_JSON="[]"
if [ -n "$METRICS" ]; then
  METRICS_JSON=$(printf '%s' "$METRICS" | awk -F',' '{for(i=1;i<=NF;i++){gsub(/^[ \t]+|[ \t]+$/,"",$i); if ($i!="") printf "\"%s\"%s", $i, (i<NF?",":"");}}' | awk '{printf("[%s]", $0)}')
fi

REQUEST_BODY=$(jq -n \
  --arg key "$TRIAL_KEY" \
  --arg title "$TRIAL_TITLE" \
  --arg description "$TRIAL_DESC" \
  --arg evidenceLevel "$EVIDENCE_LEVEL" \
  --arg trialType "$TRIAL_TYPE" \
  --arg protocol "$PROTOCOL_NAME" \
  --arg seedPrompt "$SEED_PROMPT" \
  --argjson rounds "$ROUNDS" \
  --argjson memory "$MEMORY_ENABLED" \
  --argjson metrics "$METRICS_JSON" \
  '{key: $key, title: $title, description: $description, evidenceLevel: $evidenceLevel, trialType: $trialType, protocol: $protocol, seedPrompt: $seedPrompt, rounds: $rounds, control: {memory: $memory}, metrics: $metrics}')

echo ""
echo "Submitting protocol template..."

RESPONSE=$(curl -s "${VOIDBORNE_API}/api/lab/protocols" \
  -H "Content-Type: application/json" \
  -H "Authorization: Bearer $TOKEN" \
  -d "$REQUEST_BODY")

if echo "$RESPONSE" | jq -e '.error' > /dev/null 2>&1; then
  ERROR=$(echo "$RESPONSE" | jq -r '.error')
  echo "Protocol registration failed: $ERROR"
  exit 1
fi

echo ""
echo "Protocol template registered."
echo "View protocols: ${VOIDBORNE_API}/lab"
echo ""
