#!/bin/bash
# Visla API CLI Wrapper (Bash version)
# Simple wrapper for creating videos from scripts or URLs.

set -e

BASE_URL="https://openapi.visla.us/openapi/v1"
VERSION="260218-1410"
USER_AGENT="visla-skill/${VERSION}"

# Tips to display during video generation
VISLA_TIPS=(
    "Tip: Visla AI Director creates consistent characters and environments across scenes"
    "Tip: You can convert PDFs and PPTs directly into polished videos"
    "Tip: Visla offers 100+ AI avatars with voice cloning support"
    "Tip: Scene-based editing gives you precision control over individual shots"
    "Tip: Auto-transcription makes your videos accessible with subtitles"
    "Tip: Visla supports real-time collaborative editing with your team"
    "Tip: Full Getty Images library is available for enterprise users"
    "Tip: Multiple brand kits help maintain visual consistency"
    "Tip: Text-based video editing lets you edit by modifying the transcript"
    "Tip: Built-in teleprompter helps with professional recordings"
)

# Millisecond timestamp that works across macOS/Linux.
ms_now() {
    local ts
    ts="$(date +%s%3N 2>/dev/null || true)"
    if [[ "$ts" =~ ^[0-9]+$ ]]; then
        echo "$ts"
        return 0
    fi

    # macOS `date` often doesn't support `%3N`. Use Perl (available by default) as a fallback.
    if command -v perl >/dev/null 2>&1; then
        perl -MTime::HiRes=time -e 'printf("%.0f\n", time()*1000)'
        return 0
    fi

    echo -e "${RED}Error: Could not compute millisecond timestamp (need perl or a compatible date).${NC}"
    echo "Hint: Install perl, or use the Python CLI: python3 scripts/visla_cli.py <command>"
    exit 1
}

# Basic dependency checks. Keep help usable without requiring deps.
require_cmd() {
    local name="$1"
    command -v "$name" >/dev/null 2>&1 || {
        echo "VISLA_CLI_ERROR_CODE=missing_dependency"
        echo -e "${RED}Error: Missing dependency: ${name}${NC}"
        echo "Hint: Install it and retry, or use the Python CLI: python3 scripts/visla_cli.py <command>"
        exit 1
    }
}

preflight() {
    # Used by all non-help commands.
    require_cmd curl
    require_cmd openssl
    require_cmd uuidgen
}

preflight_needs_jq() {
    # Needed for script/url/doc payload encoding.
    require_cmd jq
}

classify_api_error_code() {
    local msg="$1"
    local m
    m=$(echo "$msg" | tr '[:upper:]' '[:lower:]')
    # Use specific phrases to avoid over-classification
    if echo "$m" | grep -qE "unauthorized|forbidden|invalid api key|invalid api secret|invalid key|invalid secret|invalid sign|sign error|signature error|signature invalid|invalid signature|authentication failed|auth failed"; then
        echo "auth_failed"
        return 0
    fi
    if echo "$m" | grep -qE "rate.*limit"; then
        echo "rate_limited"
        return 0
    fi
    if echo "$m" | grep -qE "credit|quota|insufficient|balance"; then
        echo "credits_exhausted"
        return 0
    fi
    echo "api_error"
}

# Check credentials
check_credentials() {
    if [ -z "$VISLA_API_KEY" ] || [ -z "$VISLA_API_SECRET" ]; then
        echo "VISLA_CLI_ERROR_CODE=missing_credentials"
        echo -e "${RED}Error: Visla credentials not configured${NC}"
        echo ""
        echo "Set environment variables:"
        echo "  export VISLA_API_KEY=\"your_key\""
        echo "  export VISLA_API_SECRET=\"your_secret\""
        echo ""
        echo "Get your API credentials from:"
        echo -e "  ${YELLOW}https://www.visla.us/visla-api${NC}"
        exit 1
    fi
}

# Colors for output (disable when not running in a TTY to keep logs clean)
if [ -t 1 ]; then
    RED='\033[0;31m'
    GREEN='\033[0;32m'
    YELLOW='\033[1;33m'
    NC='\033[0m' # No Color
else
    RED=''
    GREEN=''
    YELLOW=''
    NC=''
fi

# Help function
show_help() {
    cat << EOF
Visla Skill v${VERSION}

Visla API CLI Wrapper

Usage: $(basename "$0") <command> [args]

Commands:
    script <text|@file>     Create video from script
    url <URL>               Create video from web page URL
    doc <file>              Create video from document (PPT/PDF)
    account                 Show account info and credit balance

Environment:
    VISLA_API_KEY             API key (from https://www.visla.us/visla-api)
    VISLA_API_SECRET          API secret

Examples:
    # Create from script
    $(basename "$0") script "Scene 1: Sunset. Narrator: Welcome."

    # Create from file
    $(basename "$0") script @myscript.txt

    # Create from URL
    $(basename "$0") url "https://blog.example.com/article"

    # Create from document
    $(basename "$0") doc presentation.pptx

EOF
}

# Generate signature and make request
visla_request() {
    local method=$1
    local endpoint=$2
    local data=$3

    local full_url="${BASE_URL}${endpoint}"
    # Extract base URL without query params for signing
    local base_url="${full_url%%\?*}"
    local ts=$(ms_now)
    local nonce=$(uuidgen)
    local sign_str="${method}|${base_url}|${ts}|${nonce}"
    local sign=$(echo -n "$sign_str" | openssl dgst -sha256 -hmac "$VISLA_API_SECRET" | awk '{print $2}')

    local result=""
    local curl_exit=0
    if [ "$method" = "GET" ]; then
        result=$(curl -sS -X GET "$full_url" \
            -H "Content-Type: application/json; charset=utf-8" \
            -H "User-Agent: $USER_AGENT" \
            -H "key: $VISLA_API_KEY" \
            -H "ts: $ts" \
            -H "nonce: $nonce" \
            -H "sign: $sign" ${data:+-d "$data"} 2>&1) || curl_exit=$?
    else
        result=$(curl -sS -X POST "$full_url" \
            -H "Content-Type: application/json; charset=utf-8" \
            -H "User-Agent: $USER_AGENT" \
            -H "key: $VISLA_API_KEY" \
            -H "ts: $ts" \
            -H "nonce: $nonce" \
            -H "sign: $sign" \
            -d "$data" 2>&1) || curl_exit=$?
    fi

    if [ "$curl_exit" -ne 0 ]; then
        # Use fixed message to avoid JSON escaping issues with raw curl output
        echo "{\"code\":-1,\"msg\":\"Network error (curl exit $curl_exit)\",\"data\":{}}"
        return 0
    fi
    echo "$result"
}

# Extract JSON value
jq_get() {
    local json=$1
    local key=$2
    # Prefer jq (already required for the main workflows). Fallback to a naive parser.
    if command -v jq >/dev/null 2>&1; then
        echo "$json" | jq -r --arg k "$key" '.data[$k] // .[$k] // empty' 2>/dev/null || true
        return 0
    fi
    echo "$json" | grep -o "\"$key\":\"[^\"]*\"" | cut -d'"' -f4
}

# Create from script
cmd_script() {
    local script="$1"
    local ratio="${2:-16:9}"
    local pace="${3:-fast}"

    # Read from file if starts with @
    if [[ "$script" == @* ]]; then
        local file="${script:1}"
        if [ ! -f "$file" ]; then
            echo "VISLA_CLI_ERROR_CODE=file_not_found"
            echo -e "${RED}Error: File not found: $file${NC}"
            exit 1
        fi
        script=$(cat "$file")
    fi

    echo "Creating video from script..."
    echo ""
    echo "$script"
    echo ""

    local payload="{
        \"script\": $(echo "$script" | jq -Rs .),
        \"target_video\": {\"aspect_ratio\": \"$ratio\", \"video_pace\": \"$pace\", \"burn_subtitles\": false}
    }"

    local result=$(visla_request "POST" "/project/script-to-video" "$payload")

    local project_uuid=$(jq_get "$result" "projectUuid")
    local share_link=$(jq_get "$result" "shareLink")

    if [ -n "$project_uuid" ]; then
        echo "Project created: $project_uuid"
        [ -n "$share_link" ] && echo "View link: $share_link"
        wait_and_export "$project_uuid"
    else
        local code
        code=$(echo "$result" | jq -r '.code // empty' 2>/dev/null || true)
        if [ "$code" != "0" ] && [ -n "$code" ]; then
            local msg
            msg=$(echo "$result" | jq -r '.msg // .message // "Unknown error"' 2>/dev/null || echo "Unknown error")
            echo "VISLA_CLI_ERROR_CODE=$(classify_api_error_code "$msg")"
            echo -e "${RED}Error: ${msg}${NC}"
        else
            echo "VISLA_CLI_ERROR_CODE=api_error"
            echo -e "${RED}Error: Failed to create project${NC}"
            echo "$result"
        fi
        exit 1
    fi
}

# Create from URL
cmd_url() {
    local url="$1"
    local ratio="${2:-16:9}"
    local pace="${3:-fast}"

    # Validate URL first
    echo "Validating URL: $url"
    local http_code
    http_code=$(curl -s -o /dev/null -w "%{http_code}" -L --head "$url" --connect-timeout 10 --max-time 20) || {
        echo "VISLA_CLI_ERROR_CODE=network_error"
        echo -e "${RED}Error: Network error validating URL: $url${NC}"
        exit 1
    }
    # Some sites block HEAD requests, retry with lightweight GET
    if [ "$http_code" = "403" ] || [ "$http_code" = "405" ]; then
        http_code=$(curl -s -o /dev/null -w "%{http_code}" -L -H "Range: bytes=0-0" "$url" --connect-timeout 10 --max-time 20) || {
            echo "VISLA_CLI_ERROR_CODE=network_error"
            echo -e "${RED}Error: Network error validating URL: $url${NC}"
            exit 1
        }
    fi
    if [ "$http_code" -ge 400 ] || [ "$http_code" = "000" ]; then
        echo "VISLA_CLI_ERROR_CODE=invalid_url"
        echo -e "${RED}Error: URL is not accessible (HTTP $http_code): $url${NC}"
        exit 1
    fi
    echo "URL validated successfully"
    echo ""

    echo "Creating video from URL..."

    local payload="{
        \"url\": $(echo "$url" | jq -Rs .),
        \"target_video\": {\"aspect_ratio\": \"$ratio\", \"video_pace\": \"$pace\", \"burn_subtitles\": false}
    }"

    local result=$(visla_request "POST" "/project/create-video-by-url" "$payload")

    local project_uuid=$(jq_get "$result" "projectUuid")
    local share_link=$(jq_get "$result" "shareLink")

    if [ -n "$project_uuid" ]; then
        echo "Project created: $project_uuid"
        [ -n "$share_link" ] && echo "View link: $share_link"
        wait_and_export "$project_uuid"
    else
        local code
        code=$(echo "$result" | jq -r '.code // empty' 2>/dev/null || true)
        if [ "$code" != "0" ] && [ -n "$code" ]; then
            local msg
            msg=$(echo "$result" | jq -r '.msg // .message // "Unknown error"' 2>/dev/null || echo "Unknown error")
            echo "VISLA_CLI_ERROR_CODE=$(classify_api_error_code "$msg")"
            echo -e "${RED}Error: ${msg}${NC}"
        else
            echo "VISLA_CLI_ERROR_CODE=api_error"
            echo -e "${RED}Error: Failed to create project${NC}"
            echo "$result"
        fi
        exit 1
    fi
}

# Create from document
cmd_doc() {
    local file="$1"
    local ratio="${2:-16:9}"
    local pace="${3:-fast}"

    if [ ! -f "$file" ]; then
        echo "VISLA_CLI_ERROR_CODE=file_not_found"
        echo -e "${RED}Error: File not found: $file${NC}"
        exit 1
    fi

    local filename=$(basename "$file")
    local suffix="${filename##*.}"
    suffix=$(echo "$suffix" | tr '[:upper:]' '[:lower:]')

    # Determine media type
    local media_type=""
    case "$suffix" in
        pptx|ppt) media_type="ppt" ;;
        pdf) media_type="pdf" ;;
        *)
            echo "VISLA_CLI_ERROR_CODE=unsupported_format"
            echo -e "${RED}Error: Unsupported file type: $suffix${NC}"
            echo "Supported formats: pptx, ppt, pdf"
            exit 1
            ;;
    esac

    echo "Uploading document: $filename"

    # Step 1: Get upload URL
    local upload_result=$(visla_request "GET" "/project/get-asset-upload-url?mediaType=$media_type&suffix=$suffix")
    local upload_url=$(jq_get "$upload_result" "uploadUrl")

    if [ -z "$upload_url" ]; then
        echo "VISLA_CLI_ERROR_CODE=api_error"
        echo -e "${RED}Error: Failed to get upload URL${NC}"
        echo "$upload_result"
        exit 1
    fi
    echo "Upload URL obtained"

    # Step 2: Upload to S3
    local content_type="application/octet-stream"
    case "$suffix" in
        pptx) content_type="application/vnd.openxmlformats-officedocument.presentationml.presentation" ;;
        ppt) content_type="application/vnd.ms-powerpoint" ;;
        pdf) content_type="application/pdf" ;;
    esac

    local upload_response
    upload_response=$(curl -s -w "%{http_code}" -X PUT "$upload_url" \
        -H "Content-Type: $content_type" \
        -H "User-Agent: $USER_AGENT" \
        --data-binary "@$file") || {
        echo "VISLA_CLI_ERROR_CODE=network_error"
        echo -e "${RED}Error: Network error uploading file${NC}"
        exit 1
    }
    local http_code="${upload_response: -3}"

    if [ "$http_code" != "200" ] && [ "$http_code" != "201" ]; then
        echo -e "${RED}Error: Failed to upload file (HTTP $http_code)${NC}"
        exit 1
    fi
    echo "File uploaded successfully"
    echo ""

    # Step 3: Create video from document
    echo "Creating video from document..."
    local payload="{
        \"doc_asset_url\": $(echo "$upload_url" | jq -Rs .),
        \"doc_file_name\": $(echo "$filename" | jq -Rs .),
        \"target_video\": {\"aspect_ratio\": \"$ratio\", \"video_pace\": \"$pace\", \"burn_subtitles\": false}
    }"

    local result=$(visla_request "POST" "/project/doc-to-video" "$payload")

    local project_uuid=$(jq_get "$result" "projectUuid")
    local share_link=$(jq_get "$result" "shareLink")

    if [ -n "$project_uuid" ]; then
        echo "Project created: $project_uuid"
        [ -n "$share_link" ] && echo "View link: $share_link"
        wait_and_export "$project_uuid"
    else
        local code
        code=$(echo "$result" | jq -r '.code // empty' 2>/dev/null || true)
        if [ "$code" != "0" ] && [ -n "$code" ]; then
            local msg
            msg=$(echo "$result" | jq -r '.msg // .message // "Unknown error"' 2>/dev/null || echo "Unknown error")
            echo "VISLA_CLI_ERROR_CODE=$(classify_api_error_code "$msg")"
            echo -e "${RED}Error: ${msg}${NC}"
        else
            echo "VISLA_CLI_ERROR_CODE=api_error"
            echo -e "${RED}Error: Failed to create project${NC}"
            echo "$result"
        fi
        exit 1
    fi
}

# Wait for project and export
wait_and_export() {
    local project_uuid="$1"

    echo ""
    echo -e "☕ ${YELLOW}Grab a coffee! Video generation takes a few minutes...${NC}"
    echo -e "🎬 Visla AI is creating your video"
    echo ""

    local attempts=0
    local tip_index=0
    local tips_count=${#VISLA_TIPS[@]}
    while [ $attempts -lt 180 ]; do
        local result=$(visla_request "GET" "/project/$project_uuid/info")
        local status=$(jq_get "$result" "progressStatus")

        if [ "$status" = "editing" ]; then
            echo ""
            echo -e "${GREEN}✓ Video generated!${NC}"
            local preview_link=$(jq_get "$result" "shareLink")
            [ -n "$preview_link" ] && echo -e "  View link: ${YELLOW}$preview_link${NC}"
            echo -e "  Exporting now, almost done..."
            echo ""
            break
        elif [ "$status" = "failed" ]; then
            echo "VISLA_CLI_ERROR_CODE=api_error"
            echo -e "${RED}Project failed!${NC}"
            echo "$result"
            exit 1
        elif [ -z "$status" ]; then
            echo "VISLA_CLI_ERROR_CODE=api_error"
            echo -e "${RED}Error: Could not read project status (unexpected response)${NC}"
            echo "$result"
            exit 1
        fi

        # Show tip before sleeping
        echo "${VISLA_TIPS[$tip_index]}"
        tip_index=$(( (tip_index + 1) % tips_count ))
        attempts=$((attempts + 1))
        sleep 20
    done

    if [ $attempts -ge 180 ]; then
        echo "VISLA_CLI_ERROR_CODE=timeout"
        echo -e "${RED}Error: Timeout waiting for video generation${NC}"
        exit 1
    fi

    # Export
    echo -e "${YELLOW}Exporting video...${NC}"
    local export_result=$(visla_request "POST" "/project/$project_uuid/export-video" "{}")

    local clip_uuid=$(jq_get "$export_result" "clipUuid")
    if [ -z "$clip_uuid" ]; then
        echo "VISLA_CLI_ERROR_CODE=api_error"
        echo -e "${RED}Export failed!${NC}"
        echo "$export_result"
        exit 1
    fi
    echo "Clip UUID: $clip_uuid"

    # Wait for clip to complete
    echo "Waiting for clip to render..."
    attempts=0
    tip_index=0
    while [ $attempts -lt 90 ]; do
        local clip_result=$(visla_request "GET" "/clip/$clip_uuid/info")
        local clip_status=$(jq_get "$clip_result" "clipStatus")

        if [ "$clip_status" = "completed" ]; then
            echo ""
            echo -e "${GREEN}Clip completed!${NC}"
            break
        elif [ "$clip_status" = "failed" ]; then
            echo "VISLA_CLI_ERROR_CODE=api_error"
            echo -e "${RED}Clip failed!${NC}"
            echo "$clip_result"
            exit 1
        elif [ -z "$clip_status" ]; then
            echo "VISLA_CLI_ERROR_CODE=api_error"
            echo -e "${RED}Error: Could not read clip status (unexpected response)${NC}"
            echo "$clip_result"
            exit 1
        fi

        # Show tip before sleeping
        echo "${VISLA_TIPS[$tip_index]}"
        tip_index=$(( (tip_index + 1) % tips_count ))
        attempts=$((attempts + 1))
        sleep 20
    done

    if [ $attempts -ge 90 ]; then
        echo "VISLA_CLI_ERROR_CODE=timeout"
        echo -e "${RED}Error: Timeout waiting for clip rendering${NC}"
        exit 1
    fi

    local share_link=$(jq_get "$export_result" "shareLink")

    echo ""
    echo -e "${GREEN}Video ready!${NC}"
    [ -n "$share_link" ] && echo "View link: $share_link"
}

# Account command (combines info + credit)
cmd_account() {
    preflight_needs_jq
    local info_result=$(visla_request "GET" "/user/info")
    local credit_result=$(visla_request "GET" "/workspace/credit-balance")

    local info_code
    info_code=$(echo "$info_result" | jq -r '.code // empty' 2>/dev/null || true)
    if [ -z "$info_code" ]; then
        echo "VISLA_CLI_ERROR_CODE=api_error"
        echo -e "${RED}Error: Unexpected response from /user/info (non-JSON)${NC}"
        echo "$info_result"
        exit 1
    fi
    if [ "$info_code" != "0" ]; then
        local msg
        msg=$(echo "$info_result" | jq -r '.msg // .message // "Unknown error"' 2>/dev/null || echo "Unknown error")
        echo "VISLA_CLI_ERROR_CODE=$(classify_api_error_code "$msg")"
        echo -e "${RED}Error: ${msg}${NC}"
        exit 1
    fi

    local credit_code
    credit_code=$(echo "$credit_result" | jq -r '.code // empty' 2>/dev/null || true)
    if [ -z "$credit_code" ]; then
        echo "VISLA_CLI_ERROR_CODE=api_error"
        echo -e "${RED}Error: Unexpected response from /workspace/credit-balance (non-JSON)${NC}"
        echo "$credit_result"
        exit 1
    fi
    if [ "$credit_code" != "0" ]; then
        local msg
        msg=$(echo "$credit_result" | jq -r '.msg // .message // "Unknown error"' 2>/dev/null || echo "Unknown error")
        echo "VISLA_CLI_ERROR_CODE=$(classify_api_error_code "$msg")"
        echo -e "${RED}Error: ${msg}${NC}"
        exit 1
    fi

    local email given_name family_name status reg_time login_time credits
    email=$(echo "$info_result" | jq -r '.data.email // ""')
    given_name=$(echo "$info_result" | jq -r '.data.givenName // ""')
    family_name=$(echo "$info_result" | jq -r '.data.familyName // ""')
    status=$(echo "$info_result" | jq -r '.data.userStatus // ""')
    reg_time=$(echo "$info_result" | jq -r '.data.regTime // 0')
    login_time=$(echo "$info_result" | jq -r '.data.loginTime // 0')
    credits=$(echo "$credit_result" | jq -r '.data // ""')

    # Convert timestamps to dates
    local reg_date="N/A"
    local login_date="N/A"
    if [[ "$reg_time" =~ ^[0-9]+$ ]] && [ "$reg_time" -gt 0 ]; then
        reg_date=$(date -r $((reg_time / 1000)) "+%Y-%m-%d" 2>/dev/null || date -d "@$((reg_time / 1000))" "+%Y-%m-%d" 2>/dev/null || echo "N/A")
    fi
    if [[ "$login_time" =~ ^[0-9]+$ ]] && [ "$login_time" -gt 0 ]; then
        login_date=$(date -r $((login_time / 1000)) "+%Y-%m-%d" 2>/dev/null || date -d "@$((login_time / 1000))" "+%Y-%m-%d" 2>/dev/null || echo "N/A")
    fi

    echo "Email: $email"
    echo "Name: $given_name $family_name"
    echo "Status: $status"
    echo "Registered: $reg_date"
    echo "Last Login: $login_date"
    echo "Credits: $credits"
}

# Parse arguments
if [ $# -eq 0 ]; then
    show_help
    exit 0
fi

command="$1"
shift

# Help doesn't need credentials
if [ "$command" = "help" ] || [ "$command" = "-h" ] || [ "$command" = "--help" ]; then
    show_help
    exit 0
fi

# Dependency checks for execution commands.
preflight

# All other commands need credentials
check_credentials

echo "Visla Skill v${VERSION}"

ratio="16:9"
pace="fast"
positional_args=()

# Parse arguments (intentionally keep the user-facing CLI minimal; use internal defaults).
while [[ $# -gt 0 ]]; do
    case "$1" in
        *)
            positional_args+=("$1")
            shift
            ;;
    esac
done

# Restore positional arguments
set -- "${positional_args[@]}"

case "$command" in
    script)
        preflight_needs_jq
        if [ -z "$1" ]; then
            echo -e "${RED}Error: script text or @filename required${NC}"
            exit 1
        fi
        cmd_script "$1" "$ratio" "$pace"
        ;;
    url)
        preflight_needs_jq
        if [ -z "$1" ]; then
            echo -e "${RED}Error: URL required${NC}"
            exit 1
        fi
        cmd_url "$1" "$ratio" "$pace"
        ;;
    doc)
        preflight_needs_jq
        if [ -z "$1" ]; then
            echo -e "${RED}Error: document file path required${NC}"
            exit 1
        fi
        cmd_doc "$1" "$ratio" "$pace"
        ;;
    account)
        cmd_account
        ;;
    help|-h|--help)
        show_help
        ;;
    *)
        echo -e "${RED}Error: Unknown command '$command'${NC}"
        echo ""
        show_help
        exit 1
        ;;
esac

echo "Visla Skill v${VERSION} completed"
