#!/usr/bin/env python3
"""
Visla API CLI Wrapper
Simple wrapper for creating videos from scripts, URLs, or documents.
"""

import warnings

# Keep CLI output clean on macOS system Python where urllib3 may emit SSL backend warnings.
# This needs to run before importing requests/urllib3.
warnings.filterwarnings(
    "ignore",
    message=r"urllib3 v2 only supports OpenSSL .*",
)

import uuid
import hashlib
import hmac
import time
import json
import sys
import argparse
import os
import mimetypes
import re

# ASCII symbols for cross-platform compatibility
# (Python CLI is typically used when Bash fails, so keep output simple)
SYM_OK = "[OK]"
SYM_INFO = "[INFO]"
SYM_VIDEO = "[VIDEO]"
SYM_WARN = "[WARN]"

VISLA_TIPS = [
    "Tip: Visla AI Director creates consistent characters and environments across scenes",
    "Tip: You can convert PDFs and PPTs directly into polished videos",
    "Tip: Visla offers 100+ AI avatars with voice cloning support",
    "Tip: Scene-based editing gives you precision control over individual shots",
    "Tip: Auto-transcription makes your videos accessible with subtitles",
    "Tip: Visla supports real-time collaborative editing with your team",
    "Tip: Full Getty Images library is available for enterprise users",
    "Tip: Multiple brand kits help maintain visual consistency",
    "Tip: Text-based video editing lets you edit by modifying the transcript",
    "Tip: Built-in teleprompter helps with professional recordings",
]

try:
    import requests
except ImportError:
    print("VISLA_CLI_ERROR_CODE=missing_dependency")
    print("Error: Missing Python dependency: requests")
    print("Install: python3 -m pip install requests")
    sys.exit(1)

VERSION = "260218-1410"
USER_AGENT = f"visla-skill/{VERSION}"

def classify_error_code(msg: str) -> str:
    m = (msg or "").lower()
    # Heuristics only; keeps the CLI surface stable while giving the agent a hint.
    # Use specific phrases first to avoid over-classification.
    if any(x in m for x in ["unauthorized", "forbidden", "invalid api key", "invalid api secret", "invalid key", "invalid secret", "invalid sign", "sign error", "signature error", "signature invalid", "invalid signature", "authentication failed", "auth failed"]):
        return "auth_failed"
    if "rate" in m and "limit" in m:
        return "rate_limited"
    if any(x in m for x in ["credit", "quota", "insufficient", "balance"]):
        return "credits_exhausted"
    if any(x in m for x in ["network error", "timeout", "timed out", "connection", "dns"]):
        return "network_error"
    return "api_error"

def _strip_quotes(s: str) -> str:
    s = s.strip()
    if (s.startswith('"') and s.endswith('"')) or (s.startswith("'") and s.endswith("'")):
        return s[1:-1]
    return s


def load_credentials_from_file(path: str):
    """
    Best-effort parser for ~/.config/visla/.credentials.

    Accepts common patterns:
    - export VISLA_API_KEY="..."
    - VISLA_API_KEY=...
    Ignores comments and unrelated lines.
    """
    try:
        with open(path, "r", encoding="utf-8") as f:
            text = f.read()
    except OSError:
        return None, None

    api_key = None
    api_secret = None

    for raw in text.splitlines():
        line = raw.strip()
        if not line or line.startswith("#"):
            continue
        if line.startswith("export "):
            line = line[len("export "):].strip()

        m = re.match(r"^(VISLA_API_KEY|VISLA_API_SECRET)\s*=\s*(.+)$", line)
        if not m:
            continue
        raw_val = m.group(2).strip()
        # Handle inline comments: strip content after # unless inside quotes
        if raw_val and raw_val[0] in ("'", '"'):
            quote = raw_val[0]
            end = raw_val.find(quote, 1)
            if end != -1:
                raw_val = raw_val[:end + 1]
        else:
            raw_val = raw_val.split("#", 1)[0].strip()
        k, v = m.group(1), _strip_quotes(raw_val)
        if k == "VISLA_API_KEY":
            api_key = v
        elif k == "VISLA_API_SECRET":
            api_secret = v

    return api_key, api_secret


class VislaAPI:
    def __init__(self, api_key, api_secret):
        self.base_url = "https://openapi.visla.us/openapi/v1"
        self.api_key = api_key
        self.api_secret = api_secret

    def _safe_json(self, resp):
        try:
            return resp.json()
        except Exception:
            # Keep callers on the same "shape" (code/msg/data) they already expect.
            text = ""
            try:
                text = resp.text
            except Exception:
                text = "<no body>"
            return {
                "code": -1,
                "msg": f"Non-JSON response (HTTP {getattr(resp, 'status_code', 'unknown')}): {text[:500]}",
                "data": {},
            }

    def _sign(self, method, url):
        """Generate HMAC-SHA256 signature"""
        ts = str(int(time.time() * 1000))
        nonce = str(uuid.uuid4())
        sign_str = f"{method.upper()}|{url}|{ts}|{nonce}"
        signature = hmac.new(
            self.api_secret.encode(), sign_str.encode(), hashlib.sha256
        ).hexdigest()
        return {
            "Content-Type": "application/json; charset=utf-8",
            "User-Agent": USER_AGENT,
            "key": self.api_key,
            "ts": ts,
            "nonce": nonce,
            "sign": signature
        }

    def _request(self, method, endpoint, data=None):
        """Make signed API request"""
        url = f"{self.base_url}{endpoint}"
        headers = self._sign(method, url)
        try:
            if method == "GET":
                resp = requests.get(url, params=data, headers=headers, timeout=60)
            else:
                resp = requests.post(url, json=data, headers=headers, timeout=60)
        except requests.RequestException as e:
            return {"code": -1, "msg": f"Network error: {e}", "data": {}}

        return self._safe_json(resp)

    # 1. Create video from script
    def create_from_script(self, script, aspect_ratio="16:9", pace="fast"):
        """Create a video project from a script"""
        payload = {
            "script": script,
            "target_video": {"aspect_ratio": aspect_ratio, "video_pace": pace, "burn_subtitles": False}
        }
        return self._request("POST", "/project/script-to-video", payload)

    # 2. Poll project status until ready
    def wait_project(self, project_uuid, interval=20, max_attempts=180):
        """Poll project status until 'editing' or failed"""
        for i in range(max_attempts):
            result = self._request("GET", f"/project/{project_uuid}/info")
            if result.get("code") != 0:
                msg = result.get("msg", "Unknown error")
                print(f"failed: {msg}")
                return result
            status = result.get("data", {}).get("progressStatus")

            if status == "editing":
                print(f"{SYM_OK} Video generated!")
                share_link = result.get("data", {}).get("shareLink")
                if share_link:
                    print(f"  View link: {share_link}")
                print("  Exporting now, almost done...")
                print()
                return result
            elif status == "failed":
                print("Project failed!")
                return result
            # Show tip before sleeping
            print(VISLA_TIPS[i % len(VISLA_TIPS)])
            time.sleep(interval)
        print("Timeout")
        return {"code": -1, "msg": "Timeout waiting for video generation", "data": {"progressStatus": "timeout"}}

    # 4. Export project to video
    def export_video(self, project_uuid):
        """Export project to downloadable video"""
        return self._request("POST", f"/project/{project_uuid}/export-video", {})

    # 5. Poll clip status until ready
    def wait_clip(self, clip_uuid, interval=20, max_attempts=90):
        """Poll clip status until 'completed'"""
        print("Waiting for clip to render...")
        for i in range(max_attempts):
            result = self._request("GET", f"/clip/{clip_uuid}/info")
            if result.get("code") != 0:
                msg = result.get("msg", "Unknown error")
                print(f"failed: {msg}")
                return result
            status = result.get("data", {}).get("clipStatus")

            if status == "completed":
                print("Clip completed!")
                return result
            elif status == "failed":
                print("Clip failed!")
                return result
            # Show tip before sleeping
            print(VISLA_TIPS[i % len(VISLA_TIPS)])
            time.sleep(interval)
        print("Timeout")
        return {"code": -1, "msg": "Timeout waiting for clip rendering", "data": {"clipStatus": "timeout"}}

    # 6. Create video from URL
    def create_from_url(self, url, aspect_ratio="16:9", pace="fast"):
        """Create a video project from a web URL"""
        payload = {
            "url": url,
            "target_video": {"aspect_ratio": aspect_ratio, "video_pace": pace, "burn_subtitles": False}
        }
        return self._request("POST", "/project/create-video-by-url", payload)

    # 8. Get upload URL for document
    def get_upload_url(self, media_type, suffix):
        """Get pre-signed S3 upload URL for document"""
        params = {"mediaType": media_type, "suffix": suffix}
        return self._request("GET", "/project/get-asset-upload-url", params)

    # 9. Upload file to S3
    def upload_to_s3(self, upload_url, file_path):
        """Upload file to S3 using pre-signed URL"""
        content_type, _ = mimetypes.guess_type(file_path)
        if content_type is None:
            content_type = 'application/octet-stream'

        headers = {'Content-Type': content_type, 'User-Agent': USER_AGENT}
        try:
            # Use streaming upload to avoid loading entire file into memory
            with open(file_path, 'rb') as f:
                response = requests.put(upload_url, data=f, headers=headers, timeout=300)
            return response
        except requests.RequestException as e:
            class DummyResp:
                status_code = 0
                error = str(e)
            return DummyResp()

    # 10. Create video from document
    def create_from_doc(self, doc_url, doc_filename, aspect_ratio="16:9", pace="fast"):
        """Create a video project from uploaded document"""
        payload = {
            "doc_asset_url": doc_url,
            "doc_file_name": doc_filename,
            "target_video": {"aspect_ratio": aspect_ratio, "video_pace": pace, "burn_subtitles": False}
        }
        return self._request("POST", "/project/doc-to-video", payload)

    # Convenience: Full workflow
    def create_and_download(self, script, aspect_ratio="16:9", pace="fast"):
        """Complete workflow: create -> wait -> export -> wait -> download link"""
        # Step 1: Create
        print("Creating video from script...")
        print()
        print(script)
        print()
        result = self.create_from_script(script, aspect_ratio, pace)

        if result.get("code") != 0:
            msg = result.get("msg", "Unknown error")
            print(f"VISLA_CLI_ERROR_CODE={classify_error_code(msg)}")
            print(f"Error: {result.get('msg')}")
            return {"error": msg}

        project_uuid = result.get("data", {}).get("projectUuid")
        share_link = result.get("data", {}).get("shareLink")
        print(f"Project created: {project_uuid}")
        if share_link:
            print(f"View link: {share_link}")
        print()
        print(f"{SYM_INFO} Grab a coffee! Video generation takes a few minutes...")
        print(f"{SYM_VIDEO} Visla AI is creating your video")
        print()

        # Step 2: Wait for project
        project = self.wait_project(project_uuid)
        if project.get("code") != 0:
            msg = project.get("msg", "Unknown error")
            print(f"VISLA_CLI_ERROR_CODE={classify_error_code(msg)}")
            print(f"Error: {msg}")
            return {"project_uuid": project_uuid, "error": msg}
        if project.get("data", {}).get("progressStatus") != "editing":
            print("VISLA_CLI_ERROR_CODE=timeout")
            print("Error: Timeout waiting for video generation")
            return {"project_uuid": project_uuid, "error": "timeout"}

        # Step 3: Export
        print("Exporting video...")
        export_result = self.export_video(project_uuid)
        if export_result.get("code") != 0:
            msg = export_result.get("msg", "Unknown error")
            print(f"VISLA_CLI_ERROR_CODE={classify_error_code(msg)}")
            print("Export failed!")
            return {"project_uuid": project_uuid, "error": msg}

        clip_uuid = export_result.get("data", {}).get("clipUuid")
        share_link = export_result.get("data", {}).get("shareLink")
        print(f"Clip UUID: {clip_uuid}")

        # Step 4: Wait for clip to complete
        clip_result = self.wait_clip(clip_uuid)
        if clip_result.get("code") != 0:
            msg = clip_result.get("msg", "Unknown error")
            print(f"VISLA_CLI_ERROR_CODE={classify_error_code(msg)}")
            print(f"Error: {msg}")
            return {"project_uuid": project_uuid, "clip_uuid": clip_uuid, "error": msg}
        if clip_result.get("data", {}).get("clipStatus") != "completed":
            print("VISLA_CLI_ERROR_CODE=timeout")
            print("Error: Timeout waiting for clip rendering")
            return {"project_uuid": project_uuid, "clip_uuid": clip_uuid, "error": "timeout"}

        print(f"\n{SYM_OK} Video ready!")

        return {
            "project_uuid": project_uuid,
            "clip_uuid": clip_uuid,
            "share_link": share_link
        }

    # Validate URL exists
    def validate_url(self, url):
        """Check if URL is accessible"""
        try:
            response = requests.head(url, timeout=10, allow_redirects=True)
            if response.status_code < 400:
                return True
            # Some sites block HEAD requests, retry with lightweight GET
            if response.status_code in (403, 405):
                response = requests.get(url, headers={"Range": "bytes=0-0"}, timeout=10, allow_redirects=True)
                return response.status_code < 400
            return False
        except Exception:
            return False

    # URL workflow: create from URL -> wait -> export -> download
    def url_and_download(self, url, aspect_ratio="16:9", pace="fast"):
        """Complete workflow for URL to video"""
        print(f"Validating URL: {url}")
        if not self.validate_url(url):
            print("VISLA_CLI_ERROR_CODE=invalid_url")
            print(f"Error: URL is not accessible: {url}")
            return {"error": "invalid_url", "url": url}
        print("URL validated successfully")
        print()
        print("Creating video from URL...")

        result = self.create_from_url(url, aspect_ratio, pace)

        if result.get("code") != 0:
            msg = result.get("msg", "Unknown error")
            print(f"VISLA_CLI_ERROR_CODE={classify_error_code(msg)}")
            print(f"Error: {result.get('msg')}")
            return {"error": msg, "url": url}

        project_uuid = result.get("data", {}).get("projectUuid")
        share_link = result.get("data", {}).get("shareLink")
        print(f"Project created: {project_uuid}")
        if share_link:
            print(f"View link: {share_link}")
        print()
        print(f"{SYM_INFO} Grab a coffee! Video generation takes a few minutes...")
        print(f"{SYM_VIDEO} Visla AI is creating your video")
        print()

        return self._complete_workflow(project_uuid)

    # Document workflow: upload -> create -> wait -> export -> download
    def doc_and_download(self, file_path, aspect_ratio="16:9", pace="fast"):
        """Complete workflow for document to video"""
        if not os.path.exists(file_path):
            print("VISLA_CLI_ERROR_CODE=file_not_found")
            print(f"Error: File not found: {file_path}")
            return {"error": "file_not_found", "file": file_path}

        filename = os.path.basename(file_path)
        suffix = filename.rsplit('.', 1)[-1].lower() if '.' in filename else ''

        # Determine media type
        media_type_map = {'pptx': 'ppt', 'ppt': 'ppt', 'pdf': 'pdf'}
        media_type = media_type_map.get(suffix)
        if not media_type:
            print("VISLA_CLI_ERROR_CODE=unsupported_format")
            print(f"Error: Unsupported file type: {suffix}")
            print("Supported formats: pptx, ppt, pdf")
            return {"error": "unsupported_format", "file": file_path}

        print(f"Uploading document: {filename}")

        # Step 1: Get upload URL
        upload_result = self.get_upload_url(media_type, suffix)
        if upload_result.get("code") != 0:
            msg = upload_result.get("msg", "Unknown error")
            print(f"VISLA_CLI_ERROR_CODE={classify_error_code(msg)}")
            print(f"Error getting upload URL: {upload_result.get('msg')}")
            return {"error": msg, "file": file_path}

        upload_url = upload_result.get("data", {}).get("uploadUrl")
        print("Upload URL obtained")

        # Step 2: Upload to S3
        response = self.upload_to_s3(upload_url, file_path)
        if response.status_code not in [200, 201]:
            if getattr(response, "error", None):
                print("VISLA_CLI_ERROR_CODE=network_error")
                print(f"Error uploading file: {response.error}")
            else:
                print("VISLA_CLI_ERROR_CODE=api_error")
                print(f"Error uploading file: HTTP {response.status_code}")
            return {"error": "upload_failed", "file": file_path}
        print("File uploaded successfully")
        print()

        # Step 3: Create video from document
        print("Creating video from document...")
        result = self.create_from_doc(upload_url, filename, aspect_ratio, pace)

        if result.get("code") != 0:
            msg = result.get("msg", "Unknown error")
            print(f"VISLA_CLI_ERROR_CODE={classify_error_code(msg)}")
            print(f"Error: {result.get('msg')}")
            return {"error": msg, "file": file_path}

        project_uuid = result.get("data", {}).get("projectUuid")
        share_link = result.get("data", {}).get("shareLink")
        print(f"Project created: {project_uuid}")
        if share_link:
            print(f"View link: {share_link}")
        print()
        print(f"{SYM_INFO} Grab a coffee! Video generation takes a few minutes...")
        print(f"{SYM_VIDEO} Visla AI is creating your video")
        print()

        return self._complete_workflow(project_uuid)

    # Shared workflow: wait -> export -> download
    def _complete_workflow(self, project_uuid):
        """Complete the workflow after project creation"""
        # Wait for project
        project = self.wait_project(project_uuid)
        if project.get("code") != 0:
            msg = project.get("msg", "Unknown error")
            print(f"VISLA_CLI_ERROR_CODE={classify_error_code(msg)}")
            print(f"Error: {msg}")
            return {"project_uuid": project_uuid, "error": msg}
        if project.get("data", {}).get("progressStatus") != "editing":
            print("VISLA_CLI_ERROR_CODE=timeout")
            print("Error: Timeout waiting for video generation")
            return {"project_uuid": project_uuid, "error": "timeout"}

        # Export
        print("Exporting video...")
        export_result = self.export_video(project_uuid)
        if export_result.get("code") != 0:
            msg = export_result.get("msg", "Unknown error")
            print(f"VISLA_CLI_ERROR_CODE={classify_error_code(msg)}")
            print("Export failed!")
            return {"project_uuid": project_uuid, "error": msg}

        clip_uuid = export_result.get("data", {}).get("clipUuid")
        share_link = export_result.get("data", {}).get("shareLink")
        print(f"Clip UUID: {clip_uuid}")

        # Wait for clip
        clip_result = self.wait_clip(clip_uuid)
        if clip_result.get("code") != 0:
            msg = clip_result.get("msg", "Unknown error")
            print(f"VISLA_CLI_ERROR_CODE={classify_error_code(msg)}")
            print(f"Error: {msg}")
            return {"project_uuid": project_uuid, "clip_uuid": clip_uuid, "error": msg}
        if clip_result.get("data", {}).get("clipStatus") != "completed":
            print("VISLA_CLI_ERROR_CODE=timeout")
            print("Error: Timeout waiting for clip rendering")
            return {"project_uuid": project_uuid, "clip_uuid": clip_uuid, "error": "timeout"}

        print(f"\n{SYM_OK} Video ready!")

        return {
            "project_uuid": project_uuid,
            "clip_uuid": clip_uuid,
            "share_link": share_link
        }


def main():
    parser = argparse.ArgumentParser(description=f"Visla Skill v{VERSION}")
    parser.add_argument("--key", help="API Key (or set VISLA_API_KEY env var)")
    parser.add_argument("--secret", help="API Secret (or set VISLA_API_SECRET env var)")
    parser.add_argument(
        "--credentials-file",
        help="Path to credentials file (explicit opt-in; requires user consent)",
    )

    subparsers = parser.add_subparsers(dest="command", help="Command")

    # Script command
    script_parser = subparsers.add_parser("script", help="Create video from script")
    script_parser.add_argument("script", help="Script text or @filename")

    # Account command
    subparsers.add_parser("account", help="Show account info and credit balance")

    # URL command
    url_parser = subparsers.add_parser("url", help="Create video from URL")
    url_parser.add_argument("url", help="Web page URL")

    # Doc command
    doc_parser = subparsers.add_parser("doc", help="Create video from document (PPT/PDF)")
    doc_parser.add_argument("file", help="Document file path")

    args = parser.parse_args()

    # Priority: command line args > environment variables
    api_key = args.key or os.environ.get("VISLA_API_KEY")
    api_secret = args.secret or os.environ.get("VISLA_API_SECRET")

    # Optional: read credentials file only when explicitly provided
    if (not api_key or not api_secret) and args.credentials_file:
        cred_path = os.path.expanduser(args.credentials_file)
        if os.path.exists(cred_path):
            file_key, file_secret = load_credentials_from_file(cred_path)
            api_key = api_key or file_key
            api_secret = api_secret or file_secret

    if not api_key or not api_secret:
        print("VISLA_CLI_ERROR_CODE=missing_credentials")
        print("Error: Visla credentials not configured")
        print("")
        print("Option 1: Set environment variables")
        if sys.platform == "win32":
            print('  $env:VISLA_API_KEY = "your_key"')
            print('  $env:VISLA_API_SECRET = "your_secret"')
        else:
            print('  export VISLA_API_KEY="your_key"')
            print('  export VISLA_API_SECRET="your_secret"')
        print("")
        print("Option 2: Pass as arguments")
        print('  python visla_cli.py --key "your_key" --secret "your_secret" <command>')
        print("")
        print("Option 3: Use a credentials file (explicit opt-in)")
        print("  Provide a file path with --credentials-file")
        print("  Example file contents:")
        print('    export VISLA_API_KEY="your_key"')
        print('    export VISLA_API_SECRET="your_secret"')
        print("")
        print("Get your API credentials from:")
        print("  https://www.visla.us/visla-api")
        sys.exit(1)

    print(f"Visla Skill v{VERSION}")
    api = VislaAPI(api_key, api_secret)
    # Intentionally keep the user-facing CLI minimal; use internal defaults.
    default_ratio = "16:9"
    default_pace = "fast"

    if args.command == "script":
        # Read from stdin if "-", from file if starts with @, otherwise use directly
        script = args.script
        if script == "-":
            # Read from stdin (no temp files needed)
            script = sys.stdin.read()
            if not script.strip():
                print("VISLA_CLI_ERROR_CODE=empty_input")
                print("Error: No script content received from stdin")
                sys.exit(1)
        elif script.startswith("@"):
            file_path = script[1:]
            if not os.path.exists(file_path):
                print("VISLA_CLI_ERROR_CODE=file_not_found")
                print(f"Error: File not found: {file_path}")
                sys.exit(1)
            try:
                with open(file_path, "r", encoding="utf-8") as f:
                    script = f.read()
            except OSError as e:
                print("VISLA_CLI_ERROR_CODE=file_read_failed")
                print(f"Error: Failed to read file: {file_path} ({e})")
                sys.exit(1)

        result = api.create_and_download(script, aspect_ratio=default_ratio, pace=default_pace)
        if not result or result.get("error"):
            # Error code already printed by create_and_download
            sys.exit(1)
        print("Video ready!")
        if result.get("share_link"):
            print(f"View link: {result['share_link']}")
        print(f"Visla Skill v{VERSION} completed")

    elif args.command == "url":
        result = api.url_and_download(args.url, aspect_ratio=default_ratio, pace=default_pace)
        if not result or result.get("error"):
            # Error code already printed by url_and_download
            sys.exit(1)
        print("Video ready!")
        if result.get("share_link"):
            print(f"View link: {result['share_link']}")
        print(f"Visla Skill v{VERSION} completed")

    elif args.command == "doc":
        result = api.doc_and_download(args.file, aspect_ratio=default_ratio, pace=default_pace)
        if not result or result.get("error"):
            # Error code already printed by doc_and_download
            sys.exit(1)
        print("Video ready!")
        if result.get("share_link"):
            print(f"View link: {result['share_link']}")
        print(f"Visla Skill v{VERSION} completed")

    elif args.command == "account":
        from datetime import datetime
        info_result = api._request("GET", "/user/info")
        credit_result = api._request("GET", "/workspace/credit-balance")
        if info_result.get("code") != 0:
            msg = info_result.get("msg", "Unknown error")
            print(f"VISLA_CLI_ERROR_CODE={classify_error_code(msg)}")
            print(msg)
            sys.exit(1)
        if credit_result.get("code") != 0:
            msg = credit_result.get("msg", "Unknown error")
            print(f"VISLA_CLI_ERROR_CODE={classify_error_code(msg)}")
            print(msg)
            sys.exit(1)
        data = info_result.get("data", {})
        email = data.get("email", "N/A")
        given_name = data.get("givenName", "")
        family_name = data.get("familyName", "")
        status = data.get("userStatus", "N/A")
        reg_time = data.get("regTime", 0)
        login_time = data.get("loginTime", 0)
        credits = credit_result.get("data", 0)

        reg_date = datetime.fromtimestamp(reg_time / 1000).strftime("%Y-%m-%d") if reg_time else "N/A"
        login_date = datetime.fromtimestamp(login_time / 1000).strftime("%Y-%m-%d") if login_time else "N/A"

        print(f"Email: {email}")
        print(f"Name: {given_name} {family_name}")
        print(f"Status: {status}")
        print(f"Registered: {reg_date}")
        print(f"Last Login: {login_date}")
        print(f"Credits: {credits}")
        print(f"Visla Skill v{VERSION} completed")
        sys.exit(0)

    else:
        parser.print_help()
        sys.exit(2)


if __name__ == "__main__":
    main()
