# Agent Guidelines for Vision Sandbox

This document provides essential information for AI agents and developers working on the `vision-sandbox` repository. It covers environment setup, build commands, testing procedures, and code style guidelines to ensure consistency and quality.

## 1. Project Overview

`vision-sandbox` is a Python-based tool designed to experiment with Agentic Vision capabilities using Gemini's native code execution sandbox. It allows users to send images and prompts to the Gemini model, which can then execute code to analyze or manipulate the inputs.

- **Primary Language:** Python 3.11+ (Locked to 3.11.x for reliability)
- **Dependency Manager:** Standard `pip` / `pyproject.toml` (compatible with `uv`)
- **Main Entry Point:** `scripts/vision_executor.py` (exposed as `vision-sandbox` script)

## 2. Environment Setup & Build

The project uses `pyproject.toml` for configuration.

### Installation

It is recommended to use a virtual environment. If using `uv` (as indicated by `uv.lock`):

```bash
# Sync dependencies from lockfile
uv sync
```

### Environment Variables

The application requires the following environment variable to be set:

- `GEMINI_API_KEY`: Your Google Gemini API key.

```bash
export GEMINI_API_KEY="your_api_key_here"
```

## 3. Build, Lint, and Test Commands

### Build

There is no complex build step. The project is a Python package.

### Linting & Formatting

This project uses `ruff` for formatting and linting.

**Recommended Commands:**

```bash
# Format code
uv run ruff format .

# Check for linting errors and fix them
uv run ruff check --fix .

# Type checking
uv run mypy .
```

### Testing

**Current Status:** No test suite is currently implemented.

**Future Testing Guidelines:**
- Use `pytest` as the testing framework.
- Place tests in a `tests/` directory or alongside source files.
- Naming convention: `test_*.py` or `*_test.py`.

**Running Tests (when implemented):**

```bash
# Run all tests
uv run pytest
```

## 4. Code Style Guidelines

Adhere to the following conventions to maintain codebase consistency.

### General

- **Python Version:** Target Python 3.11 specifically.
- **Paradigm:** Script-based functional style is currently dominant. Classes should be used where state management is complex.

### Formatting

- **Indentation:** 4 spaces (PEP 8).
- **Line Length:** 88 characters (Ruff default).
- **Quotes:** Double quotes `"` are preferred for strings.

### Naming Conventions

- **Functions & Variables:** `snake_case` (e.g., `run_vision_sandbox`).
- **Classes:** `PascalCase`.
- **Constants:** `UPPER_SNAKE_CASE`.

### Imports

Group imports in the following order:
1.  **Standard Library**
2.  **Third-Party Libraries**
3.  **Local Application Imports**

### Type Hinting

- **New code must employ type hints**.
- Use `typing` module or standard types.

### Error Handling

- Use explicit `try...except` blocks for external operations (API calls, file I/O).
- Print errors to `stderr` or use `sys.exit(1)` to indicate failure.

### Documentation

- **Docstrings:** Use triple double-quotes (`"""`).
- **Comments:** Use `#` for inline comments explaining *why*.

## 5. File Structure & Organization

- **`scripts/`**: Main execution logic.
- **`sample/`**: Sample inputs.
- **`main.py`**: Lightweight wrapper.

## 6. Commit Messages

- Use imperative mood (e.g., "Add feature").
- Keep the first line under 72 characters.

---
*Generated by OpenCode Agent.*
