---
name: vimeo
description: |
  Vimeo API integration with managed OAuth. Video hosting and sharing platform.
  Use this skill when users want to upload, manage, or organize videos, create showcases/albums, manage folders, or interact with the Vimeo community.
  For other third party apps, use the api-gateway skill (https://clawhub.ai/byungkyu/api-gateway).
  Requires network access and valid Maton API key.
metadata:
  author: maton
  version: "1.0"
  clawdbot:
    emoji: 🧠
    homepage: "https://maton.ai"
    requires:
      env:
        - MATON_API_KEY
---

# Vimeo

Access the Vimeo API with managed OAuth authentication. Upload and manage videos, create showcases and folders, manage likes and watch later, and interact with the Vimeo community.

## Quick Start

```bash
# Get current user info
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://gateway.maton.ai/vimeo/me')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

## Base URL

```
https://gateway.maton.ai/vimeo/{resource}
```

The gateway proxies requests to `api.vimeo.com` and automatically injects your OAuth token.

## Authentication

All requests require the Maton API key in the Authorization header:

```
Authorization: Bearer $MATON_API_KEY
```

**Environment Variable:** Set your API key as `MATON_API_KEY`:

```bash
export MATON_API_KEY="YOUR_API_KEY"
```

### Getting Your API Key

1. Sign in or create an account at [maton.ai](https://maton.ai)
2. Go to [maton.ai/settings](https://maton.ai/settings)
3. Copy your API key

## Connection Management

Manage your Vimeo OAuth connections at `https://ctrl.maton.ai`.

### List Connections

```bash
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://ctrl.maton.ai/connections?app=vimeo&status=ACTIVE')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

### Create Connection

```bash
python <<'EOF'
import urllib.request, os, json
data = json.dumps({'app': 'vimeo'}).encode()
req = urllib.request.Request('https://ctrl.maton.ai/connections', data=data, method='POST')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
req.add_header('Content-Type', 'application/json')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

### Get Connection

```bash
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://ctrl.maton.ai/connections/{connection_id}')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

**Response:**
```json
{
  "connection": {
    "connection_id": "a6ecb894-3148-4f4c-a54c-e9d917e3f2a9",
    "status": "ACTIVE",
    "creation_time": "2026-02-09T08:56:53.522100Z",
    "last_updated_time": "2026-02-09T08:58:39.407864Z",
    "url": "https://connect.maton.ai/?session_token=...",
    "app": "vimeo",
    "metadata": {}
  }
}
```

Open the returned `url` in a browser to complete OAuth authorization.

### Delete Connection

```bash
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://ctrl.maton.ai/connections/{connection_id}', method='DELETE')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

### Specifying Connection

If you have multiple Vimeo connections, specify which one to use with the `Maton-Connection` header:

```bash
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://gateway.maton.ai/vimeo/me')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
req.add_header('Maton-Connection', 'a6ecb894-3148-4f4c-a54c-e9d917e3f2a9')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

If omitted, the gateway uses the default (oldest) active connection.

## API Reference

### User Operations

#### Get Current User

```bash
GET /vimeo/me
```

**Response:**
```json
{
  "uri": "/users/254399456",
  "name": "Chris",
  "link": "https://vimeo.com/user254399456",
  "account": "free",
  "created_time": "2026-02-09T07:00:20+00:00",
  "pictures": {...},
  "metadata": {
    "connections": {
      "videos": {"uri": "/users/254399456/videos", "total": 2},
      "albums": {"uri": "/users/254399456/albums", "total": 0},
      "folders": {"uri": "/users/254399456/folders", "total": 0},
      "likes": {"uri": "/users/254399456/likes", "total": 0},
      "followers": {"uri": "/users/254399456/followers", "total": 0},
      "following": {"uri": "/users/254399456/following", "total": 0}
    }
  }
}
```

#### Get User by ID

```bash
GET /vimeo/users/{user_id}
```

#### Get User Feed

```bash
GET /vimeo/me/feed
```

### Video Operations

#### List User Videos

```bash
GET /vimeo/me/videos
```

**Response:**
```json
{
  "total": 2,
  "page": 1,
  "per_page": 25,
  "paging": {
    "next": null,
    "previous": null,
    "first": "/me/videos?page=1",
    "last": "/me/videos?page=1"
  },
  "data": [
    {
      "uri": "/videos/1163160198",
      "name": "My Video",
      "description": "Video description",
      "link": "https://vimeo.com/1163160198",
      "duration": 20,
      "width": 1920,
      "height": 1080,
      "created_time": "2026-02-09T07:05:00+00:00"
    }
  ]
}
```

#### Get Video

```bash
GET /vimeo/videos/{video_id}
```

#### Search Videos

```bash
GET /vimeo/videos?query=nature&per_page=10
```

Query parameters:
- `query` - Search query
- `per_page` - Results per page (max 100)
- `page` - Page number
- `sort` - Sort order: `relevant`, `date`, `alphabetical`, `plays`, `likes`, `comments`, `duration`
- `direction` - Sort direction: `asc`, `desc`

#### Update Video

```bash
PATCH /vimeo/videos/{video_id}
Content-Type: application/json

{
  "name": "New Video Title",
  "description": "Updated description"
}
```

#### Delete Video

```bash
DELETE /vimeo/videos/{video_id}
```

Returns 204 No Content on success.

### Folder Operations (Projects)

#### List Folders

```bash
GET /vimeo/me/folders
```

**Response:**
```json
{
  "total": 1,
  "page": 1,
  "per_page": 25,
  "data": [
    {
      "uri": "/users/254399456/projects/28177219",
      "name": "My Folder",
      "created_time": "2026-02-09T08:59:20+00:00",
      "privacy": {"view": "nobody"},
      "manage_link": "https://vimeo.com/user/254399456/folder/28177219"
    }
  ]
}
```

#### Create Folder

```bash
POST /vimeo/me/folders
Content-Type: application/json

{
  "name": "New Folder"
}
```

#### Update Folder

```bash
PATCH /vimeo/me/projects/{project_id}
Content-Type: application/json

{
  "name": "Renamed Folder"
}
```

#### Delete Folder

```bash
DELETE /vimeo/me/projects/{project_id}
```

Returns 204 No Content on success.

#### Get Folder Videos

```bash
GET /vimeo/me/projects/{project_id}/videos
```

#### Add Video to Folder

```bash
PUT /vimeo/me/projects/{project_id}/videos/{video_id}
```

Returns 204 No Content on success.

#### Remove Video from Folder

```bash
DELETE /vimeo/me/projects/{project_id}/videos/{video_id}
```

### Album Operations (Showcases)

#### List Albums

```bash
GET /vimeo/me/albums
```

#### Create Album

```bash
POST /vimeo/me/albums
Content-Type: application/json

{
  "name": "My Showcase",
  "description": "A collection of videos"
}
```

**Response:**
```json
{
  "uri": "/users/254399456/albums/12099981",
  "name": "My Showcase",
  "description": "A collection of videos",
  "created_time": "2026-02-09T09:00:00+00:00"
}
```

#### Update Album

```bash
PATCH /vimeo/me/albums/{album_id}
Content-Type: application/json

{
  "name": "Updated Showcase Name"
}
```

#### Delete Album

```bash
DELETE /vimeo/me/albums/{album_id}
```

Returns 204 No Content on success.

#### Get Album Videos

```bash
GET /vimeo/me/albums/{album_id}/videos
```

#### Add Video to Album

```bash
PUT /vimeo/me/albums/{album_id}/videos/{video_id}
```

Returns 204 No Content on success.

#### Remove Video from Album

```bash
DELETE /vimeo/me/albums/{album_id}/videos/{video_id}
```

### Comments

#### Get Video Comments

```bash
GET /vimeo/videos/{video_id}/comments
```

#### Add Comment

```bash
POST /vimeo/videos/{video_id}/comments
Content-Type: application/json

{
  "text": "Great video!"
}
```

**Response:**
```json
{
  "uri": "/videos/1163160198/comments/21372988",
  "text": "Great video!",
  "created_on": "2026-02-09T09:05:00+00:00"
}
```

#### Delete Comment

```bash
DELETE /vimeo/videos/{video_id}/comments/{comment_id}
```

Returns 204 No Content on success.

### Likes

#### Get Liked Videos

```bash
GET /vimeo/me/likes
```

#### Like a Video

```bash
PUT /vimeo/me/likes/{video_id}
```

Returns 204 No Content on success.

#### Unlike a Video

```bash
DELETE /vimeo/me/likes/{video_id}
```

Returns 204 No Content on success.

### Watch Later

#### Get Watch Later List

```bash
GET /vimeo/me/watchlater
```

#### Add to Watch Later

```bash
PUT /vimeo/me/watchlater/{video_id}
```

Returns 204 No Content on success.

#### Remove from Watch Later

```bash
DELETE /vimeo/me/watchlater/{video_id}
```

Returns 204 No Content on success.

### Followers and Following

#### Get Followers

```bash
GET /vimeo/me/followers
```

#### Get Following

```bash
GET /vimeo/me/following
```

#### Follow a User

```bash
PUT /vimeo/me/following/{user_id}
```

#### Unfollow a User

```bash
DELETE /vimeo/me/following/{user_id}
```

### Channels and Categories

#### List All Channels

```bash
GET /vimeo/channels
```

#### Get Channel

```bash
GET /vimeo/channels/{channel_id}
```

#### List All Categories

```bash
GET /vimeo/categories
```

**Response:**
```json
{
  "total": 10,
  "data": [
    {"uri": "/categories/animation", "name": "Animation"},
    {"uri": "/categories/comedy", "name": "Comedy"},
    {"uri": "/categories/documentary", "name": "Documentary"}
  ]
}
```

#### Get Category Videos

```bash
GET /vimeo/categories/{category}/videos
```

## Pagination

Vimeo uses page-based pagination:

```bash
GET /vimeo/me/videos?page=1&per_page=25
```

**Response:**
```json
{
  "total": 50,
  "page": 1,
  "per_page": 25,
  "paging": {
    "next": "/me/videos?page=2",
    "previous": null,
    "first": "/me/videos?page=1",
    "last": "/me/videos?page=2"
  },
  "data": [...]
}
```

Parameters:
- `page` - Page number (default 1)
- `per_page` - Results per page (default 25, max 100)

## Code Examples

### JavaScript

```javascript
const response = await fetch(
  'https://gateway.maton.ai/vimeo/me/videos',
  {
    headers: {
      'Authorization': `Bearer ${process.env.MATON_API_KEY}`
    }
  }
);
const data = await response.json();
```

### Python

```python
import os
import requests

response = requests.get(
    'https://gateway.maton.ai/vimeo/me/videos',
    headers={'Authorization': f'Bearer {os.environ["MATON_API_KEY"]}'}
)
data = response.json()
```

### Python (Create Folder)

```python
import os
import requests

response = requests.post(
    'https://gateway.maton.ai/vimeo/me/folders',
    headers={
        'Authorization': f'Bearer {os.environ["MATON_API_KEY"]}',
        'Content-Type': 'application/json'
    },
    json={'name': 'New Folder'}
)
folder = response.json()
print(f"Created folder: {folder['uri']}")
```

### Python (Update Video)

```python
import os
import requests

video_id = "1163160198"
response = requests.patch(
    f'https://gateway.maton.ai/vimeo/videos/{video_id}',
    headers={
        'Authorization': f'Bearer {os.environ["MATON_API_KEY"]}',
        'Content-Type': 'application/json'
    },
    json={
        'name': 'Updated Title',
        'description': 'New description'
    }
)
video = response.json()
print(f"Updated video: {video['name']}")
```

## Notes

- Video IDs are numeric (e.g., `1163160198`)
- User IDs are numeric (e.g., `254399456`)
- Folders are called "projects" in the API paths
- Albums are also known as "Showcases" in the Vimeo UI
- DELETE and PUT operations return 204 No Content on success
- Video uploads require the TUS protocol (not covered here)
- Rate limits vary by account type
- IMPORTANT: When piping curl output to `jq` or other commands, environment variables like `$MATON_API_KEY` may not expand correctly in some shell environments

## Error Handling

| Status | Meaning |
|--------|---------|
| 400 | Missing Vimeo connection or bad request |
| 401 | Invalid or missing Maton API key |
| 403 | Insufficient permissions or scope |
| 404 | Resource not found |
| 429 | Rate limited |
| 4xx/5xx | Passthrough error from Vimeo API |

Vimeo errors include detailed messages:
```json
{
  "error": "Your access token does not have the \"create\" scope"
}
```

### Troubleshooting: API Key Issues

1. Check that the `MATON_API_KEY` environment variable is set:

```bash
echo $MATON_API_KEY
```

2. Verify the API key is valid by listing connections:

```bash
python <<'EOF'
import urllib.request, os, json
req = urllib.request.Request('https://ctrl.maton.ai/connections')
req.add_header('Authorization', f'Bearer {os.environ["MATON_API_KEY"]}')
print(json.dumps(json.load(urllib.request.urlopen(req)), indent=2))
EOF
```

### Troubleshooting: Invalid App Name

1. Ensure your URL path starts with `vimeo`. For example:

- Correct: `https://gateway.maton.ai/vimeo/me/videos`
- Incorrect: `https://gateway.maton.ai/me/videos`

## Resources

- [Vimeo API Reference](https://developer.vimeo.com/api/reference)
- [Vimeo Developer Portal](https://developer.vimeo.com)
- [Vimeo API Authentication](https://developer.vimeo.com/api/authentication)
- [Vimeo Upload API](https://developer.vimeo.com/api/upload/videos)
- [Maton Community](https://discord.com/invite/dBfFAcefs2)
- [Maton Support](mailto:support@maton.ai)
