#!/usr/bin/env python3
"""
FireAnt Stock Checker - Automated Vietnamese stock price lookup
Usage: python3 check_stock.py <SYMBOL1> [SYMBOL2] ...
Example: python3 check_stock.py DPM VCB FPT
"""

import sys
import json
import subprocess
import time
from typing import List, Dict, Optional

def run_openclaw_command(command: List[str]) -> Dict:
    """Execute OpenClaw browser command and return result"""
    try:
        result = subprocess.run(['openclaw'] + command, 
                              capture_output=True, text=True, timeout=30)
        if result.returncode != 0:
            return {"error": f"Command failed: {result.stderr}"}
        
        # Parse JSON output
        try:
            return json.loads(result.stdout)
        except json.JSONDecodeError:
            return {"error": "Failed to parse command output"}
    except subprocess.TimeoutExpired:
        return {"error": "Command timeout"}
    except Exception as e:
        return {"error": f"Command error: {str(e)}"}

def search_and_navigate_to_stock(symbol: str) -> Optional[str]:
    """Search Google for stock and navigate to FireAnt page"""
    
    # Open Google and search
    google_result = run_openclaw_command([
        'browser', 'open', '--url', 'https://google.com', 
        '--profile', 'openclaw', '--target', 'host'
    ])
    
    if 'error' in google_result:
        print(f"Failed to open Google: {google_result['error']}")
        return None
    
    target_id = google_result.get('targetId')
    if not target_id:
        print("Failed to get target ID")
        return None
    
    time.sleep(2)  # Wait for page load
    
    # Click search box and enter query
    search_query = f"{symbol} cổ phiếu fireant"
    
    # Get page snapshot to find search elements
    snapshot = run_openclaw_command([
        'browser', 'snapshot', '--target-id', target_id,
        '--profile', 'openclaw', '--target', 'host'
    ])
    
    # Click search box
    run_openclaw_command([
        'browser', 'act', '--target-id', target_id,
        '--profile', 'openclaw', '--target', 'host',
        '--request', json.dumps({"kind": "click", "ref": "combobox"})
    ])
    
    time.sleep(1)
    
    # Type search query  
    run_openclaw_command([
        'browser', 'act', '--target-id', target_id,
        '--profile', 'openclaw', '--target', 'host',
        '--request', json.dumps({"kind": "type", "text": search_query})
    ])
    
    time.sleep(1)
    
    # Press Enter to search
    run_openclaw_command([
        'browser', 'act', '--target-id', target_id,
        '--profile', 'openclaw', '--target', 'host',
        '--request', json.dumps({"kind": "press", "key": "Enter"})
    ])
    
    time.sleep(3)  # Wait for search results
    
    # Get search results and find FireAnt link
    results = run_openclaw_command([
        'browser', 'snapshot', '--target-id', target_id,
        '--profile', 'openclaw', '--target', 'host'
    ])
    
    # Look for FireAnt link in results (simplified - would need proper parsing)
    fireant_url = f"https://fireant.vn/ma-chung-khoan/{symbol}"
    
    # Navigate directly to FireAnt page
    navigate_result = run_openclaw_command([
        'browser', 'navigate', '--target-id', target_id,
        '--target-url', fireant_url,
        '--profile', 'openclaw', '--target', 'host'
    ])
    
    if 'error' in navigate_result:
        print(f"Failed to navigate to FireAnt: {navigate_result['error']}")
        return None
    
    time.sleep(3)  # Wait for page load
    return target_id

def extract_stock_data(target_id: str) -> Dict:
    """Extract stock data from FireAnt page"""
    
    # Get page snapshot
    snapshot = run_openclaw_command([
        'browser', 'snapshot', '--target-id', target_id,
        '--profile', 'openclaw', '--target', 'host'
    ])
    
    if 'error' in snapshot or not snapshot:
        return {"error": "Failed to get page snapshot"}
    
    # Parse the snapshot text to extract stock data
    # This is simplified - in practice you'd parse the structured snapshot
    snapshot_text = str(snapshot)
    
    stock_data = {}
    
    # Extract basic info (simplified parsing)
    if "404" in snapshot_text or "not found" in snapshot_text.lower():
        return {"error": "Stock not found on FireAnt"}
    
    # Look for price patterns in snapshot
    import re
    
    # Try to find current price
    price_match = re.search(r'"([0-9]+\.[0-9]+)"\s*.*?generic.*?[+\-]([0-9.]+)\s*/\s*[+\-]?([0-9.]+)%', snapshot_text)
    if price_match:
        stock_data['current_price'] = price_match.group(1)
        stock_data['change_amount'] = price_match.group(2) 
        stock_data['change_percent'] = price_match.group(3)
    
    # Try to find company name
    name_match = re.search(r'Tổng Công ty ([^"]+)', snapshot_text)
    if name_match:
        stock_data['company_name'] = name_match.group(1).strip()
    
    # Look for volume data
    vol_match = re.search(r'Vol.*?strong.*?([0-9.,]+[KMB]?)', snapshot_text)
    if vol_match:
        stock_data['volume'] = vol_match.group(1)
    
    # Look for market cap
    cap_match = re.search(r'Thị giá vốn.*?([0-9.,]+\s*tỷ)', snapshot_text)
    if cap_match:
        stock_data['market_cap'] = cap_match.group(1)
        
    return stock_data

def format_stock_output(symbol: str, data: Dict) -> str:
    """Format stock data for readable output"""
    
    if 'error' in data:
        return f"❌ {symbol}: {data['error']}"
    
    if not data:
        return f"❌ {symbol}: No data found"
    
    output = [f"📊 **{symbol.upper()}**"]
    
    if 'company_name' in data:
        output.append(f"🏢 {data['company_name']}")
    
    if 'current_price' in data:
        price = data['current_price']
        change = data.get('change_amount', 'N/A')
        percent = data.get('change_percent', 'N/A')
        
        if percent != 'N/A':
            direction = "⬆️" if float(percent) >= 0 else "⬇️"
            output.append(f"💰 Price: **{price}** VND {direction} {change} ({percent}%)")
        else:
            output.append(f"💰 Price: **{price}** VND")
    
    if 'volume' in data:
        output.append(f"📈 Volume: {data['volume']}")
    
    if 'market_cap' in data:
        output.append(f"🏦 Market Cap: {data['market_cap']}")
    
    return "\n".join(output)

def check_stocks(symbols: List[str]) -> str:
    """Main function to check multiple stocks"""
    
    if not symbols:
        return "❌ No stock symbols provided"
    
    # Start browser if not running
    browser_start = run_openclaw_command([
        'browser', 'start', '--profile', 'openclaw', '--target', 'host'
    ])
    
    results = []
    
    for symbol in symbols:
        print(f"🔍 Checking {symbol.upper()}...")
        
        # Search and navigate to stock page
        target_id = search_and_navigate_to_stock(symbol.upper())
        
        if not target_id:
            results.append(f"❌ {symbol.upper()}: Failed to navigate to page")
            continue
        
        # Extract stock data
        data = extract_stock_data(target_id)
        
        # Format output
        formatted = format_stock_output(symbol.upper(), data)
        results.append(formatted)
        
        # Small delay between requests
        time.sleep(1)
    
    return "\n\n".join(results)

def main():
    """Main entry point"""
    
    if len(sys.argv) < 2:
        print("Usage: python3 check_stock.py <SYMBOL1> [SYMBOL2] ...")
        print("Example: python3 check_stock.py DPM VCB FPT")
        sys.exit(1)
    
    symbols = [arg.upper() for arg in sys.argv[1:]]
    
    print(f"🚀 Checking {len(symbols)} stock(s): {', '.join(symbols)}")
    print("=" * 50)
    
    results = check_stocks(symbols)
    print(results)

if __name__ == "__main__":
    main()