---
name: aioz-stream-video-upload
description: Quick upload video to AIOZ Stream API. Create video objects with default or custom encoding configurations, upload the file, complete the upload, then return the video link to the user.
metadata:
  openclaw:
    emoji: "🎬"
    requires:
      bins: 
        - curl
        - jq
        - md5sum
---

# AIOZ Stream Video Upload

Upload videos to AIOZ Stream API quickly with API key authentication. The full upload flow requires 3 API calls: Create → Upload Part → Complete.

## When to use this skill

- User wants to upload or create a video on AIOZ Stream
- User mentions "upload video", "create video", "aioz stream video"
- User wants to get an HLS/DASH streaming link for their video

## Authentication

This skill uses API key authentication. The user must provide:

- `stream-public-key`: their AIOZ Stream public key
- `stream-secret-key`: their AIOZ Stream secret key

Ask the user for these keys if not provided. They will be sent as HTTP headers on ALL API calls.

## Usage Options

When the user wants to upload video, ask them to choose:

### Option 1: Default Upload (Quick)

Creates a video object with minimal config — just a title. Then uploads the file.

Example user prompt:
> "Upload video file /path/to/video.mp4 with title My Video"

### Option 2: Custom Upload (Advanced)

Creates a video object with full encoding configuration including quality presets (240p, 360p, 480p, 720p, 1080p, 1440p, 2160p, 4320p), codecs (h264, h265), bitrates, container types, tags, metadata, etc. Then uploads the file.

Example user prompt:
> "Upload video with custom config: title My Tutorial, qualities 720p and 1080p, h264 codec, tags tutorial,education"

## Full Upload Flow (3 Steps)

### Step 1: Create Video Object

**Default:**

```bash
curl -s -X POST 'https://api-w3stream.attoaioz.cyou/api/videos/create' \
  -H 'stream-public-key: PUBLIC_KEY' \
  -H 'stream-secret-key: SECRET_KEY' \
  -H 'Content-Type: application/json' \
  -d '{
    "title": "VIDEO_TITLE"
  }'
```

**Custom (with encoding config):**

```bash
curl -s -X POST 'https://api-w3stream.attoaioz.cyou/api/videos/create' \
  -H 'stream-public-key: PUBLIC_KEY' \
  -H 'stream-secret-key: SECRET_KEY' \
  -H 'Content-Type: application/json' \
  -d '{
    "title": "VIDEO_TITLE",
    "description": "DESCRIPTION",
    "is_public": true,
    "tags": ["tag1", "tag2"],
    "metadata": [
      {"key": "KEY", "value": "VALUE"}
    ],
    "qualities": [
      {
        "resolution": "1080p",
        "type": "hls",
        "container_type": "mpegts",
        "video_config": {
          "codec": "h264",
          "bitrate": 5000000,
          "index": 0
        },
        "audio_config": {
          "codec": "aac",
          "bitrate": 192000,
          "channels": "2",
          "sample_rate": 48000,
          "language": "en",
          "index": 0
        }
      },
      {
        "resolution": "720p",
        "type": "hls",
        "container_type": "mpegts",
        "video_config": {
          "codec": "h264",
          "bitrate": 3000000,
          "index": 0
        },
        "audio_config": {
          "codec": "aac",
          "bitrate": 128000,
          "channels": "2",
          "sample_rate": 44100,
          "language": "en",
          "index": 0
        }
      }
    ]
  }'
```

Response: Extract `data.id` — this is the `VIDEO_ID` used in the next steps.

### Step 2: Upload File Part

Upload the actual video file binary to the created video object.

First, get the file size and compute the MD5 hash:

```bash
# Get file size (cross-platform compatible)
FILE_SIZE=$(stat -f%z /path/to/video.mp4 2>/dev/null || stat -c%s /path/to/video.mp4)
END_POS=$((FILE_SIZE - 1))

# Compute MD5 hash
HASH=$(md5sum /path/to/video.mp4 | awk '{print $1}')
```

Then upload via multipart form-data with the Content-Range header:

```bash
curl -s -X POST "https://api-w3stream.attoaioz.cyou/api/videos/VIDEO_ID/part" \
  -H 'stream-public-key: PUBLIC_KEY' \
  -H 'stream-secret-key: SECRET_KEY' \
  -H "Content-Range: bytes 0-$END_POS/$FILE_SIZE" \
  -F "file=@/path/to/video.mp4" \
  -F "index=0" \
  -F "hash=$HASH"
```

**Important:** The `Content-Range` header is required for the upload to succeed. Format: `bytes {start}-{end}/{total_size}` where:
- For single-part uploads: `start=0`, `end=file_size-1`, `total_size=file_size`
- For multi-part uploads (files > 50MB): adjust start/end positions for each chunk (chunk size: 50MB - 200MB)

Form-data fields:
- `file`: the video file binary (use `@/path/to/file`)
- `index`: 0 (for single-part upload, increment for multi-part)
- `hash`: MD5 hash of the file part

### Step 3: Complete Upload

After the file part is uploaded, call the complete endpoint to finalize:

```bash
curl -s -X GET "https://api-w3stream.attoaioz.cyou/api/videos/VIDEO_ID/complete" \
  -H 'accept: application/json' \
  -H 'stream-public-key: PUBLIC_KEY' \
  -H 'stream-secret-key: SECRET_KEY'
```

This triggers transcoding. The upload is now considered successful.

## After Upload — Get Video Link

After completing the upload, fetch the video detail to get the streaming URL:

```bash
curl -s 'https://api-w3stream.attoaioz.cyou/api/videos/VIDEO_ID' \
  -H 'stream-public-key: PUBLIC_KEY' \
  -H 'stream-secret-key: SECRET_KEY'
```

Parse the response to find the HLS/DASH URLs from the `assets` or `hls` field and return it to the user.

## Custom Upload Config Reference

### Supported Resolutions:
- `240p` — 426 × 240 (max bitrate: 700,000 bps)
- `360p` — 640 × 360 (max bitrate: 1,200,000 bps)
- `480p` — 854 × 480 (max bitrate: 2,000,000 bps)
- `720p` — 1280 × 720 HD (max bitrate: 4,000,000 bps)
- `1080p` — 1920 × 1080 Full HD (max bitrate: 6,000,000 bps)
- `1440p` — 2560 × 1440 2K/QHD (max bitrate: 12,000,000 bps)
- `2160p` — 3840 × 2160 4K/UHD (max bitrate: 30,000,000 bps)
- `4320p` — 7680 × 4320 8K/UHD-2 (max bitrate: 60,000,000 bps)

### Streaming Formats (`type` field):
- `hls` — HTTP Live Streaming (container: `mpegts` or `mp4`)
- `dash` — Dynamic Adaptive Streaming (container: `fmp4`)

### Container Types:
- For HLS: `mpegts` or `mp4`
- For DASH: `fmp4`

**Apple HLS Compatibility:**
- H.265/HEVC is only supported in HLS with `mp4` container (fMP4/CMAF segments)
- H.265 with `mpegts` is NOT supported on Apple platforms
- H.264 works with both `mpegts` and `mp4` containers

### Video Config:
- `codec`: `h264` (max 4K) or `h265` (max 8K)
- `bitrate`: integer in bits/sec (see resolution table for max values)
- `index`: 0 (default video track)

### Audio Config:
- `codec`: `aac` (only supported codec)
- `bitrate`: 128000 - 256000 bps recommended
- `channels`: `"2"` (stereo)
- `sample_rate`: 8000, 11025, 16000, 22050, 32000, 44100, 48000, 88200, 96000
- `language`: BCP 47 code (e.g., `en`, `vi`)
- `index`: 0 (default audio track)

### Recommended Audio Bitrates:
- Standard: 128,000 – 192,000 bps
- High Quality: 192,000 – 256,000 bps

### Recommended Sample Rates:
- Voice: 22050 or 32000
- Music/Video: 44100 or 48000

## Advanced Configurations

### Video-Only Output
Specify only `video_config` without `audio_config`:

```json
{
  "resolution": "720p",
  "type": "hls",
  "container_type": "mpegts",
  "video_config": {
    "codec": "h264",
    "bitrate": 3000000,
    "index": 0
  }
}
```

### Audio-Only Output
Specify only `audio_config` without `video_config`:

```json
{
  "resolution": "audio",
  "type": "hls",
  "container_type": "mpegts",
  "audio_config": {
    "codec": "aac",
    "bitrate": 192000,
    "channels": "2",
    "sample_rate": 48000,
    "language": "en",
    "index": 0
  }
}
```

## Response Handling

1. Parse the JSON response from the create call → extract `data.id`
2. Compute MD5 hash of the video file
3. Upload the file part with the hash
4. Call complete endpoint
5. Fetch video detail to get streaming URL
6. Return the video link to the user
7. If the video is still transcoding (status: transcoding), inform the user and suggest checking back later

## Error Handling

- **401**: Invalid API keys — ask user to verify their public and secret keys
- **400**: Bad request — check the request body format, ensure resolutions don't exceed source resolution
- **500**: Server error — suggest retrying

## Example Interaction Flow

1. User: "Upload my video to AIOZ Stream"
2. Ask for API keys (public + secret) if not known
3. Ask for the video file path
4. Ask: "Default upload (quick) or custom config?"
   - If default: ask for title only
   - If custom: ask for title, qualities (e.g., 720p, 1080p), codec preference, tags, etc.
5. **Step 1:** Create video object → get `VIDEO_ID`
6. **Step 2:** Compute file hash, upload file part
7. **Step 3:** Call complete endpoint
8. Fetch video detail → return streaming URL to user

## Additional Features

### Calculate Transcode Price
Before uploading, estimate the transcoding cost:

```bash
curl -s 'https://api-w3stream.attoaioz.cyou/api/videos/cost?duration=60&qualities=360p,1080p' \
  -H 'stream-public-key: PUBLIC_KEY' \
  -H 'stream-secret-key: SECRET_KEY'
```

### Upload Thumbnail
After creating a video, upload a custom thumbnail:

```bash
curl -s -X POST "https://api-w3stream.attoaioz.cyou/api/videos/VIDEO_ID/thumbnail" \
  -H 'stream-public-key: PUBLIC_KEY' \
  -H 'stream-secret-key: SECRET_KEY' \
  -F 'file=@/path/to/thumbnail.jpg'
```

Supported formats: `.png`, `.jpg`

### Update Video Object
Modify video metadata after creation:

```bash
curl -s -X PATCH "https://api-w3stream.attoaioz.cyou/api/videos/VIDEO_ID" \
  -H 'stream-public-key: PUBLIC_KEY' \
  -H 'stream-secret-key: SECRET_KEY' \
  -H 'Content-Type: application/json' \
  -d '{
    "title": "Updated Title",
    "description": "Updated description",
    "tags": ["new", "tags"],
    "is_public": true
  }'
```

### List All Videos
Retrieve all videos with filtering:

```bash
curl -s -X POST 'https://api-w3stream.attoaioz.cyou/api/videos' \
  -H 'stream-public-key: PUBLIC_KEY' \
  -H 'stream-secret-key: SECRET_KEY' \
  -H 'Content-Type: application/json' \
  -d '{
    "limit": 10,
    "offset": 0,
    "sort_by": "created_at",
    "order_by": "desc",
    "status": "done"
  }'
```

### Delete Video
Remove a video:

```bash
curl -s -X DELETE "https://api-w3stream.attoaioz.cyou/api/videos/VIDEO_ID" \
  -H 'stream-public-key: PUBLIC_KEY' \
  -H 'stream-secret-key: SECRET_KEY'
```
