#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
Grid Trading Strategy Template
Generated by VibeTrading Code Generator

This template implements a grid trading strategy that places buy and sell orders
at predefined price levels within a specified range.

METADATA:
Description: Grid trading strategy that places multiple buy and sell orders within a price range
Strategy_Type: grid_trading
Supported_Symbols: BTC,ETH,SOL,HYPE,OTHER
Tags: grid,advanced,automated
END METADATA
"""

import os
import sys
import time
import logging
import json
from datetime import datetime
from pathlib import Path


# Add api_wrappers to path
sys.path.insert(0, str(Path(__file__).parent.parent / "api_wrappers"))

try:
    from hyperliquid_api import HyperliquidClient
    API_AVAILABLE = True
except ImportError as e:
    API_AVAILABLE = False
    print("Warning: Could not import HyperliquidClient: {}".format(e))
    print("Please ensure api_wrappers/hyperliquid_api.py exists")

class GridTradingStrategy:
    """Grid Trading Strategy"""
    
    def __init__(self, api_key, account_address, config):
        """
        Initialize grid trading strategy
        
        Args:
            api_key: Hyperliquid API key
            account_address: Hyperliquid account address
            config: Strategy configuration dictionary
        """
        self.api_key = api_key
        self.account_address = account_address
        self.config = config
        
        # Extract parameters from config
        self.symbol = config.get('symbol', 'HYPE')
        self.lower_bound = config.get('lower_bound', 0.1)
        self.upper_bound = config.get('upper_bound', 1.0)
        self.grid_count = config.get('grid_count', 10)
        self.grid_size = config.get('grid_size', 100.0)  # Quantity per grid
        self.order_type = config.get('order_type', 'limit')
        self.check_interval = config.get('check_interval', 60)  # seconds
        
        # Risk management parameters
        self.stop_loss = config.get('stop_loss', 0.05)  # 5% stop loss
        self.take_profit = config.get('take_profit', 0.10)  # 10% take profit
        self.max_position = config.get('max_position', 1000.0)  # Max position size
        
        # Calculate grid prices
        self.grid_prices = self._calculate_grid_prices()
        self.active_orders = {}  # Store active orders {order_id: {price, side, quantity}}
        
        # Setup logging FIRST
        self._setup_logging()
        
        # Initialize API client AFTER logging is setup
        self.client = None
        self._init_api_client()
        
        self.logger.info(f"Initialized {self.symbol} grid trading strategy")
        self.logger.info(f"Price range: ${self.lower_bound:.4f} - ${self.upper_bound:.4f}")
        self.logger.info(f"Grid count: {self.grid_count}")
        self.logger.info(f"Grid size: {self.grid_size}")
    
    def _setup_logging(self):
        """Setup logging configuration"""
        log_dir = Path("logs")
        log_dir.mkdir(exist_ok=True)
        
        timestamp = datetime.now().strftime('%Y%m%d_%H%M%S')
        log_file = log_dir / "grid_trading_{}_{}.log".format(self.symbol, timestamp)
        
        logging.basicConfig(
            level=logging.INFO,
            format='%(asctime)s - %(name)s - %(levelname)s - %(message)s',
            handlers=[
                logging.FileHandler(log_file),
                logging.StreamHandler()
            ]
        )
        self.logger = logging.getLogger(__name__)
    
    def _init_api_client(self):
        """Initialize API client"""
        if API_AVAILABLE:
            try:
                self.client = HyperliquidClient(
                    api_key=self.api_key,
                    account_address=self.account_address,
                    testnet=False  # Set to True for testnet
                )
                
                # Perform health check
                if self.client.health_check():
                    self.logger.info("API client initialized successfully")
                else:
                    self.logger.error("API health check failed")
                    raise Exception("API health check failed")
                    
            except Exception as e:
                self.logger.error(f"Failed to initialize API client: {e}")
                raise
        else:
            self.logger.error("API client not available")
            raise ImportError("HyperliquidClient not available")
    
    def _calculate_grid_prices(self):
        """Calculate grid price levels"""
        if self.grid_count < 2:
            raise ValueError("Grid count must be at least 2")
        
        price_step = (self.upper_bound - self.lower_bound) / (self.grid_count - 1)
        grid_prices = [self.lower_bound + i * price_step for i in range(self.grid_count)]
        
        self.logger.info(f"Calculated grid prices: {grid_prices}")
        return grid_prices
    
    def place_grid_orders(self):
        """Place grid orders at calculated price levels"""
        self.logger.info(f"Placing {self.symbol} grid orders...")
        
        # Cancel existing orders first
        self.cancel_all_orders()
        
        # Place buy orders (even grid levels)
        buy_orders_placed = 0
        for i, price in enumerate(self.grid_prices):
            if i % 2 == 0:  # Even levels are buy orders
                try:
                    result = self.client.place_order(
                        symbol=self.symbol,
                        is_buy=True,
                        size=self.grid_size,
                        order_type=self.order_type,
                        price=price
                    )
                    
                    if "error" not in result:
                        # Extract order ID from response
                        order_id = self._extract_order_id(result)
                        if order_id:
                            self.active_orders[order_id] = {
                                'price': price,
                                'side': 'buy',
                                'quantity': self.grid_size,
                                'timestamp': time.time()
                            }
                            buy_orders_placed += 1
                            self.logger.info("Buy order placed: ${:.4f}, size: {}".format(price, self.grid_size))
                    else:
                        self.logger.error("Buy order failed at ${}: {}".format(price, result.get('error')))
                        
                except Exception as e:
                    self.logger.error("Error placing buy order at ${}: {}".format(price, e))
        
        # Place sell orders (odd grid levels)
        sell_orders_placed = 0
        for i, price in enumerate(self.grid_prices):
            if i % 2 == 1:  # Odd levels are sell orders
                try:
                    result = self.client.place_order(
                        symbol=self.symbol,
                        is_buy=False,
                        size=self.grid_size,
                        order_type=self.order_type,
                        price=price
                    )
                    
                    if "error" not in result:
                        # Extract order ID from response
                        order_id = self._extract_order_id(result)
                        if order_id:
                            self.active_orders[order_id] = {
                                'price': price,
                                'side': 'sell',
                                'quantity': self.grid_size,
                                'timestamp': time.time()
                            }
                            sell_orders_placed += 1
                            self.logger.info("Sell order placed: ${:.4f}, size: {}".format(price, self.grid_size))
                    else:
                        self.logger.error("Sell order failed at ${}: {}".format(price, result.get('error')))
                        
                except Exception as e:
                    self.logger.error("Error placing sell order at ${}: {}".format(price, e))
        
        self.logger.info("Grid orders placed: {} buy, {} sell".format(buy_orders_placed, sell_orders_placed))
    
    def _extract_order_id(self, order_response):
        """Extract order ID from order response"""
        try:
            # Response structure may vary - adjust based on actual API response
            if "response" in order_response:
                response_data = order_response.get("response", {})
                if "data" in response_data:
                    statuses = response_data.get("data", {}).get("statuses", [])
                    if statuses:
                        status = statuses[0]
                        if "resting" in status:
                            return status["resting"].get("oid")
                        elif "filled" in status:
                            return status["filled"].get("oid")
            
            # Alternative: check for direct order ID
            if "oid" in order_response:
                return order_response["oid"]
                
        except Exception as e:
            self.logger.debug("Could not extract order ID: {}".format(e))
        
        return None
    
    def cancel_all_orders(self):
        """Cancel all active orders"""
        self.logger.info("Cancelling all active orders...")
        
        try:
            # Cancel via API
            success = self.client.cancel_all_orders(self.symbol)
            
            if success:
                self.active_orders.clear()
                self.logger.info("All orders cancelled successfully")
            else:
                self.logger.warning("Some orders may not have been cancelled")
                
        except Exception as e:
            self.logger.error("Error cancelling orders: {}".format(e))
    
    def monitor_and_adjust(self):
        """Monitor market and adjust grid as needed"""
        self.logger.info("Starting grid strategy monitoring...")
        
        try:
            while True:
                # Get current price
                current_price = self.client.get_price(self.symbol)
                self.logger.info("Current {} price: ${:.4f}".format(self.symbol, current_price))
                
                # Check if rebalancing is needed
                if self._need_rebalance(current_price):
                    self.logger.info("Price out of grid range, rebalancing...")
                    self.place_grid_orders()
                
                # Check order status
                self._check_order_status()
                
                # Check risk management
                self._check_risk_management(current_price)
                
                # Log strategy status
                self._log_strategy_status(current_price)
                
                # Wait for next check
                time.sleep(self.check_interval)
                
        except KeyboardInterrupt:
            self.logger.info("Strategy stopped by user")
            self._cleanup()
        except Exception as e:
            self.logger.error("Error in monitoring loop: {}".format(e))
            self._cleanup()
            raise
    
    def _need_rebalance(self, current_price):
        """Check if grid needs rebalancing"""
        # Rebalance if price moves outside grid range by more than threshold
        rebalance_threshold = 0.1  # 10%
        
        below_lower = current_price < self.lower_bound * (1 - rebalance_threshold)
        above_upper = current_price > self.upper_bound * (1 + rebalance_threshold)
        
        if below_lower or above_upper:
            self.logger.warning("Price ${:.4f} outside grid range (${}-${})".format(
                current_price, self.lower_bound, self.upper_bound))
            return True
        
        return False
    
    def _check_order_status(self):
        """Check status of active orders"""
        try:
            # Get open orders from API
            open_orders = self.client.get_open_orders(self.symbol)
            
            # Update active orders list
            active_order_ids = set(self.active_orders.keys())
            api_order_ids = {str(order.get('oid')) for order in open_orders}
            
            # Remove orders that are no longer open
            for order_id in list(active_order_ids):
                if order_id not in api_order_ids:
                    order_info = self.active_orders.get(order_id, {})
                    side = order_info.get('side', 'unknown')
                    price = order_info.get('price', 0)
                    self.logger.info("Order {} ({} @ ${:.4f}) filled or cancelled".format(
                        order_id, side, price))
                    del self.active_orders[order_id]
                    
        except Exception as e:
            self.logger.error("Error checking order status: {}".format(e))
    
    def _check_risk_management(self, current_price):
        """Check risk management rules"""
        # Placeholder for risk management logic
        # In production, implement proper risk checks
        pass
    
    def _log_strategy_status(self, current_price):
        """Log current strategy status"""
        status = {
            'timestamp': datetime.now().isoformat(),
            'symbol': self.symbol,
            'current_price': current_price,
            'active_orders': len(self.active_orders),
            'grid_lower': self.lower_bound,
            'grid_upper': self.upper_bound,
            'grid_count': self.grid_count,
            'grid_size': self.grid_size
        }
        
        # Save to status file
        status_dir = Path("status")
        status_dir.mkdir(exist_ok=True)
        
        status_file = status_dir / "grid_status_{}.json".format(self.symbol)
        try:
            with open(status_file, 'w') as f:
                json.dump(status, f, indent=2)
        except Exception as e:
            self.logger.error("Error saving status file: {}".format(e))
    
    def _cleanup(self):
        """Cleanup resources before exit"""
        self.logger.info("Cleaning up strategy resources...")
        self.cancel_all_orders()
        self.logger.info("Cleanup complete")
    
    def run(self):
        """Run the grid trading strategy"""
        self.logger.info("Starting grid trading strategy...")
        
        try:
            # Place initial grid
            self.place_grid_orders()
            
            # Start monitoring
            self.monitor_and_adjust()
            
        except Exception as e:
            self.logger.error("Strategy error: {}".format(e))
            self._cleanup()
            raise

def load_config(config_file="config.json"):
    """Load configuration from file"""
    if os.path.exists(config_file):
        try:
            with open(config_file, 'r') as f:
                return json.load(f)
        except Exception as e:
            print("Error loading config file: {}".format(e))
    
    # Default configuration
    return {
        "symbol": "HYPE",
        "lower_bound": 0.1,
        "upper_bound": 1.0,
        "grid_count": 10,
        "grid_size": 100.0,
        "order_type": "limit",
        "check_interval": 60,
        "stop_loss": 0.05,
        "take_profit": 0.10,
        "max_position": 1000.0
    }

def main():
    """Main function"""
    # Get API credentials from environment
    api_key = os.getenv("HYPERLIQUID_API_KEY")
    account_address = os.getenv("HYPERLIQUID_ACCOUNT_ADDRESS")
    
    if not api_key or not account_address:
        print("Error: HYPERLIQUID_API_KEY and HYPERLIQUID_ACCOUNT_ADDRESS environment variables are required")
        print("\nSetup instructions:")
        print("1. Get your API key and account address from Hyperliquid")
        print("2. Set environment variables:")
        print("   export HYPERLIQUID_API_KEY='your_api_key_here'")
        print("   export HYPERLIQUID_ACCOUNT_ADDRESS='your_account_address_here'")
        print("3. Run the strategy again")
        return
    
    # Load configuration
    config = load_config()
    
    # Create and run strategy
    try:
        strategy = GridTradingStrategy(
            api_key=api_key,
            account_address=account_address,
            config=config
        )
        
        strategy.run()
        
    except Exception as e:
        print("Strategy failed: {}".format(e))
        return 1
    
    return 0

if __name__ == "__main__":
    sys.exit(main())