#!/usr/bin/env node
/**
 * OpenAI-compatible proxy for wire-pod → OpenClaw
 * Supports both streaming and non-streaming responses
 * 
 * Run: node proxy-server.js
 * Configure wire-pod: Custom provider, endpoint http://localhost:11435/v1
 */

const http = require('http');
const fs = require('fs');
const path = require('path');

const PORT = process.env.PROXY_PORT || 11435;
const REQUEST_FILE = path.join(__dirname, 'request.json');
const RESPONSE_FILE = path.join(__dirname, 'response.json');

// Clean up stale files
try { fs.unlinkSync(REQUEST_FILE); } catch {}
try { fs.unlinkSync(RESPONSE_FILE); } catch {}

const server = http.createServer(async (req, res) => {
  res.setHeader('Access-Control-Allow-Origin', '*');
  res.setHeader('Access-Control-Allow-Methods', 'GET, POST, OPTIONS');
  res.setHeader('Access-Control-Allow-Headers', 'Content-Type, Authorization');

  if (req.method === 'OPTIONS') {
    res.writeHead(200);
    res.end();
    return;
  }

  if (req.url === '/health' || req.url === '/') {
    res.writeHead(200, { 'Content-Type': 'application/json' });
    res.end(JSON.stringify({ status: 'ok', service: 'vector-openclaw-proxy' }));
    return;
  }

  if (req.url === '/v1/models') {
    res.writeHead(200, { 'Content-Type': 'application/json' });
    res.end(JSON.stringify({
      data: [{ id: 'openclaw', object: 'model', owned_by: 'openclaw' }]
    }));
    return;
  }

  if (req.url === '/v1/chat/completions' && req.method === 'POST') {
    let body = '';
    req.on('data', chunk => body += chunk);
    req.on('end', async () => {
      try {
        const data = JSON.parse(body);
        const messages = data.messages || [];
        const isStreaming = data.stream === true;
        
        const userMessage = messages.filter(m => m.role === 'user').pop();
        const question = userMessage?.content || 'Hello';

        console.log(`[${new Date().toISOString()}] Question: "${question}" (stream: ${isStreaming})`);

        const request = { timestamp: Date.now(), question, raw: data };
        fs.writeFileSync(REQUEST_FILE, JSON.stringify(request, null, 2));

        // Wait for response
        const startTime = Date.now();
        const timeout = 25000;
        let response = null;

        while (Date.now() - startTime < timeout) {
          if (fs.existsSync(RESPONSE_FILE)) {
            try {
              const responseData = JSON.parse(fs.readFileSync(RESPONSE_FILE, 'utf8'));
              if (responseData.timestamp > request.timestamp - 1000) {
                response = responseData.answer;
                fs.unlinkSync(RESPONSE_FILE);
                break;
              }
            } catch (e) {}
          }
          await new Promise(r => setTimeout(r, 100));
        }

        if (!response) {
          response = "I didn't get a response in time. Please try again.";
          console.log(`[${new Date().toISOString()}] Timeout`);
        }

        console.log(`[${new Date().toISOString()}] Response: "${response}"`);

        if (isStreaming) {
          res.writeHead(200, {
            'Content-Type': 'text/event-stream',
            'Cache-Control': 'no-cache',
            'Connection': 'keep-alive'
          });

          const chunk = {
            id: 'chatcmpl-' + Date.now(),
            object: 'chat.completion.chunk',
            created: Math.floor(Date.now() / 1000),
            model: 'openclaw',
            choices: [{
              index: 0,
              delta: { role: 'assistant', content: response },
              finish_reason: null
            }]
          };
          res.write(`data: ${JSON.stringify(chunk)}\n\n`);

          const finishChunk = {
            id: 'chatcmpl-' + Date.now(),
            object: 'chat.completion.chunk',
            created: Math.floor(Date.now() / 1000),
            model: 'openclaw',
            choices: [{ index: 0, delta: {}, finish_reason: 'stop' }]
          };
          res.write(`data: ${JSON.stringify(finishChunk)}\n\n`);
          res.write('data: [DONE]\n\n');
          res.end();

        } else {
          const openaiResponse = {
            id: 'chatcmpl-' + Date.now(),
            object: 'chat.completion',
            created: Math.floor(Date.now() / 1000),
            model: 'openclaw',
            choices: [{
              index: 0,
              message: { role: 'assistant', content: response },
              finish_reason: 'stop'
            }],
            usage: { prompt_tokens: 0, completion_tokens: 0, total_tokens: 0 }
          };
          res.writeHead(200, { 'Content-Type': 'application/json' });
          res.end(JSON.stringify(openaiResponse));
        }

      } catch (e) {
        console.error('Error:', e);
        res.writeHead(500, { 'Content-Type': 'application/json' });
        res.end(JSON.stringify({ error: e.message }));
      }
    });
    return;
  }

  res.writeHead(404, { 'Content-Type': 'application/json' });
  res.end(JSON.stringify({ error: 'Not found' }));
});

server.listen(PORT, () => {
  console.log(`Vector-OpenClaw proxy running on http://localhost:${PORT}`);
  console.log(`Configure wire-pod: Custom provider, endpoint http://localhost:${PORT}/v1`);
});
