#!/usr/bin/env python3
"""
Vector Search System for MEMORY.md
Lightweight semantic search using TF-IDF and SQLite.
No heavy dependencies - uses scikit-learn only.
"""

import os
import sys
import json
import hashlib
import re
import sqlite3
from datetime import datetime
from pathlib import Path
from typing import List, Dict, Optional, Tuple
import math

# Config
MEMORY_PATH = Path("/root/.openclaw/workspace/MEMORY.md")
VECTORS_DIR = Path("/root/.openclaw/workspace/memory/vectors")
DB_PATH = VECTORS_DIR / "vectors.db"

class SimpleVectorSearch:
    """Lightweight vector search using TF-IDF."""
    
    def __init__(self):
        self.conn = None
        self._tfidf = None
        self._vocab = {}
        self._ensure_dirs()
        
    def _ensure_dirs(self):
        """Ensure required directories exist."""
        VECTORS_DIR.mkdir(parents=True, exist_ok=True)
        
    def _get_db(self) -> sqlite3.Connection:
        """Get database connection."""
        if self.conn is None:
            self.conn = sqlite3.connect(DB_PATH)
            self.conn.row_factory = sqlite3.Row
            self._init_db()
        return self.conn
        
    def _init_db(self):
        """Initialize database schema."""
        cursor = self.conn.cursor()
        
        # Main table for sections
        cursor.execute("""
            CREATE TABLE IF NOT EXISTS sections (
                id TEXT PRIMARY KEY,
                title TEXT NOT NULL,
                content TEXT NOT NULL,
                hash TEXT NOT NULL,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            )
        """)
        
        # Table for embeddings as JSON
        cursor.execute("""
            CREATE TABLE IF NOT EXISTS embeddings (
                section_id TEXT PRIMARY KEY,
                vector TEXT NOT NULL,  -- JSON array
                FOREIGN KEY (section_id) REFERENCES sections(id)
            )
        """)
        
        # Metadata table for vocab
        cursor.execute("""
            CREATE TABLE IF NOT EXISTS metadata (
                key TEXT PRIMARY KEY,
                value TEXT NOT NULL
            )
        """)
        
        self.conn.commit()
        
    def _parse_memory_md(self) -> List[Dict]:
        """Parse MEMORY.md into sections."""
        if not MEMORY_PATH.exists():
            raise FileNotFoundError(f"MEMORY.md not found at {MEMORY_PATH}")
            
        content = MEMORY_PATH.read_text(encoding='utf-8')
        sections = []
        
        # Split by headers (## or ###)
        lines = content.split('\n')
        current_section = None
        section_idx = 0
        
        for line in lines:
            if line.startswith('##') or line.startswith('###'):
                # Save previous section
                if current_section:
                    sections.append(current_section)
                
                # Start new section
                level = 2 if line.startswith('##') else 3
                title = line.lstrip('#').strip()
                section_id = f"sec_{section_idx:04d}_{self._slugify(title)}"
                current_section = {
                    'id': section_id,
                    'title': title,
                    'content': '',
                    'level': level,
                    'text': title + '\n\n'
                }
                section_idx += 1
            elif current_section is not None:
                current_section['content'] += line + '\n'
                current_section['text'] += line + '\n'
        
        # Add last section
        if current_section:
            sections.append(current_section)
            
        # Calculate hashes and clean up
        for sec in sections:
            sec['content'] = sec['content'].strip()
            sec['text'] = sec['text'].strip()
            sec['hash'] = hashlib.sha256(sec['content'].encode()).hexdigest()[:16]
            
        return sections
        
    def _slugify(self, text: str) -> str:
        """Create URL-friendly slug from text."""
        slug = re.sub(r'[^\w\s-]', '', text.lower())
        slug = re.sub(r'[-\s]+', '-', slug)
        return slug[:50]
        
    def _tokenize(self, text: str) -> List[str]:
        """Simple tokenizer."""
        # Lowercase, remove special chars, split by whitespace
        text = text.lower()
        text = re.sub(r'[^\w\s]', ' ', text)
        tokens = text.split()
        # Simple stemming (remove common suffixes)
        filtered = []
        stopwords = {'a', 'an', 'and', 'are', 'as', 'at', 'be', 'by', 'for', 'from', 'has', 'he', 'in', 'is', 'it', 'its', 'of', 'on', 'that', 'the', 'to', 'was', 'will', 'with', 'the', 'a', 'je', 'se', 'na', 'to', 'že', 'pro', 'jak', 'nebo', 'co', 'si', 've', 'od', 'za', 'podle', 'po', 'který', 'která', 'které', 'aby', 'má', 'tak', 'ale', 'i', 'o', 've', 'by', 'když', 'u', 'do', 'ze', 'při', 'kde'}
        for t in tokens:
            if len(t) > 2 and t not in stopwords:
                filtered.append(t)
        return filtered
        
    def _compute_tfidf(self, documents: List[str]) -> Tuple[List[Dict], Dict]:
        """Compute TF-IDF vectors for documents."""
        # Tokenize all documents
        tokenized = [self._tokenize(doc) for doc in documents]
        
        # Build vocabulary
        vocab = {}
        doc_freq = {}
        for tokens in tokenized:
            unique = set(tokens)
            for t in unique:
                doc_freq[t] = doc_freq.get(t, 0) + 1
                if t not in vocab:
                    vocab[t] = len(vocab)
                    
        N = len(documents)
        vectors = []
        
        for tokens in tokenized:
            # TF
            tf = {}
            for t in tokens:
                tf[t] = tf.get(t, 0) + 1
            
            # TF-IDF vector
            vector = {}
            for t, count in tf.items():
                if t in vocab:
                    tf_score = count / len(tokens)
                    idf_score = math.log(N / (doc_freq.get(t, 1) + 1)) + 1
                    vector[vocab[t]] = tf_score * idf_score
                    
            vectors.append(vector)
            
        return vectors, vocab
        
    def _cosine_similarity(self, vec_a: Dict, vec_b: Dict) -> float:
        """Calculate cosine similarity between two sparse vectors."""
        dot = 0
        for k, v in vec_a.items():
            if k in vec_b:
                dot += v * vec_b[k]
                
        norm_a = math.sqrt(sum(v * v for v in vec_a.values()))
        norm_b = math.sqrt(sum(v * v for v in vec_b.values()))
        
        if norm_a == 0 or norm_b == 0:
            return 0.0
        return dot / (norm_a * norm_b)
        
    def _serialize_vector(self, vector: Dict) -> str:
        """Serialize vector to JSON."""
        return json.dumps(vector)
        
    def _deserialize_vector(self, data: str) -> Dict:
        """Deserialize vector from JSON."""
        raw = json.loads(data)
        # Convert string keys back to integers
        return {int(k): v for k, v in raw.items()}
        
    def _save_vocab(self):
        """Save vocabulary to database."""
        conn = self._get_db()
        cursor = conn.cursor()
        cursor.execute("""
            INSERT OR REPLACE INTO metadata (key, value)
            VALUES ('vocab', ?)
        """, (json.dumps(self._vocab),))
        conn.commit()
        
    def _load_vocab(self):
        """Load vocabulary from database."""
        conn = self._get_db()
        cursor = conn.cursor()
        cursor.execute("SELECT value FROM metadata WHERE key = 'vocab'")
        row = cursor.fetchone()
        if row:
            self._vocab = json.loads(row['value'])
        else:
            self._vocab = {}
            
    def rebuild(self) -> Dict:
        """Rebuild entire database from scratch."""
        print("Starting full rebuild...")
        
        # Clear existing data
        conn = self._get_db()
        cursor = conn.cursor()
        cursor.execute("DELETE FROM embeddings")
        cursor.execute("DELETE FROM sections")
        conn.commit()
        
        # Parse sections
        sections = self._parse_memory_md()
        print(f"Found {len(sections)} sections")
        
        if not sections:
            return {'status': 'success', 'indexed': 0}
            
        # Compute TF-IDF for all sections
        documents = [sec['text'] for sec in sections]
        vectors, vocab = self._compute_tfidf(documents)
        self._vocab = vocab
        
        # Store all sections
        for i, section in enumerate(sections):
            self._index_section(section, vectors[i])
            if (i + 1) % 10 == 0:
                print(f"  Indexed {i + 1}/{len(sections)} sections...")
        
        # Save vocabulary
        self._save_vocab()
                
        print(f"✓ Indexed {len(sections)} sections")
        return {'status': 'success', 'indexed': len(sections)}
        
    def update(self) -> Dict:
        """Incremental update - only changed sections."""
        print("Starting incremental update...")
        
        conn = self._get_db()
        cursor = conn.cursor()
        
        # Get existing hashes
        cursor.execute("SELECT id, hash FROM sections")
        existing = {row['id']: row['hash'] for row in cursor.fetchall()}
        
        # Parse current sections
        sections = self._parse_memory_md()
        print(f"Found {len(sections)} sections in MEMORY.md")
        
        if not sections:
            return {'status': 'success', 'added': 0, 'updated': 0, 'unchanged': 0, 'deleted': 0}
            
        # Compute TF-IDF for all current sections
        documents = [sec['text'] for sec in sections]
        vectors, vocab = self._compute_tfidf(documents)
        self._vocab = vocab
        
        added = 0
        updated = 0
        unchanged = 0
        current_ids = set()
        
        for i, section in enumerate(sections):
            current_ids.add(section['id'])
            
            if section['id'] not in existing:
                self._index_section(section, vectors[i])
                added += 1
            elif existing[section['id']] != section['hash']:
                self._index_section(section, vectors[i], update=True)
                updated += 1
            else:
                unchanged += 1
                
        # Remove deleted sections
        cursor.execute("SELECT id FROM sections")
        stored_ids = {row['id'] for row in cursor.fetchall()}
        deleted = stored_ids - current_ids
        
        for sid in deleted:
            cursor.execute("DELETE FROM sections WHERE id = ?", (sid,))
            cursor.execute("DELETE FROM embeddings WHERE section_id = ?", (sid,))
            
        conn.commit()
        
        # Save vocabulary
        self._save_vocab()
        
        print(f"✓ Added: {added}, Updated: {updated}, Unchanged: {unchanged}, Deleted: {len(deleted)}")
        return {
            'status': 'success',
            'added': added,
            'updated': updated,
            'unchanged': unchanged,
            'deleted': len(deleted)
        }
        
    def _index_section(self, section: Dict, vector: Dict, update: bool = False):
        """Index a single section."""
        conn = self._get_db()
        cursor = conn.cursor()
        
        if update:
            cursor.execute("""
                UPDATE sections 
                SET title = ?, content = ?, hash = ?, updated_at = CURRENT_TIMESTAMP
                WHERE id = ?
            """, (section['title'], section['content'], section['hash'], section['id']))
        else:
            cursor.execute("""
                INSERT INTO sections (id, title, content, hash)
                VALUES (?, ?, ?, ?)
            """, (section['id'], section['title'], section['content'], section['hash']))
            
        cursor.execute("""
            INSERT OR REPLACE INTO embeddings (section_id, vector)
            VALUES (?, ?)
        """, (section['id'], self._serialize_vector(vector)))
        
        conn.commit()
        
    def search(self, query: str, top_k: int = 5) -> List[Dict]:
        """Search for similar sections."""
        conn = self._get_db()
        cursor = conn.cursor()
        
        # Load vocabulary
        self._load_vocab()
        
        # Compute query vector
        query_tokens = self._tokenize(query)
        query_vector = {}
        for t in query_tokens:
            if t in self._vocab:
                idx = self._vocab[t]
                query_vector[idx] = query_vector.get(idx, 0) + 1
                
        # Normalize query vector
        total = len(query_tokens)
        if total > 0:
            query_vector = {k: v/total for k, v in query_vector.items()}
        
        # Get all sections with embeddings
        cursor.execute("""
            SELECT s.id, s.title, s.content, e.vector
            FROM sections s
            JOIN embeddings e ON s.id = e.section_id
        """)
        
        results = []
        for row in cursor.fetchall():
            vector = self._deserialize_vector(row['vector'])
            score = self._cosine_similarity(query_vector, vector)
            results.append({
                'id': row['id'],
                'title': row['title'],
                'content': row['content'][:500],  # Truncate for display
                'score': score
            })
            
        # Sort by score descending
        results.sort(key=lambda x: x['score'], reverse=True)
        
        return results[:top_k]
        
    def get_stats(self) -> Dict:
        """Get database statistics."""
        conn = self._get_db()
        cursor = conn.cursor()
        
        # Load vocabulary
        self._load_vocab()
        
        cursor.execute("SELECT COUNT(*) FROM sections")
        section_count = cursor.fetchone()[0]
        
        cursor.execute("SELECT COUNT(*) FROM embeddings")
        embedding_count = cursor.fetchone()[0]
        
        return {
            'sections': section_count,
            'embeddings': embedding_count,
            'database_path': str(DB_PATH),
            'vocab_size': len(self._vocab)
        }


def main():
    """CLI interface."""
    import argparse
    
    parser = argparse.ArgumentParser(description='Vector Search for MEMORY.md')
    parser.add_argument('--rebuild', action='store_true', help='Rebuild entire index')
    parser.add_argument('--update', action='store_true', help='Incremental update')
    parser.add_argument('--search', type=str, help='Search query')
    parser.add_argument('--top-k', type=int, default=5, help='Number of results')
    parser.add_argument('--stats', action='store_true', help='Show statistics')
    
    args = parser.parse_args()
    
    vs = SimpleVectorSearch()
    
    if args.rebuild:
        result = vs.rebuild()
        print(json.dumps(result, indent=2))
    elif args.update:
        result = vs.update()
        print(json.dumps(result, indent=2))
    elif args.search:
        results = vs.search(args.search, args.top_k)
        print(f"\nSearch results for: '{args.search}'\n")
        for i, r in enumerate(results, 1):
            print(f"{i}. [{r['score']:.3f}] {r['title']}")
            print(f"   {r['content'][:200]}...")
            print()
    elif args.stats:
        stats = vs.get_stats()
        print(json.dumps(stats, indent=2))
    else:
        parser.print_help()


if __name__ == '__main__':
    main()
