#!/usr/bin/env python3
import argparse, time, urllib.request, urllib.parse, sys, random, subprocess, os, uuid

def post(base, path, params):
    q = urllib.parse.urlencode(params)
    url = f"{base}{path}?{q}"
    req = urllib.request.Request(url, method="POST")
    with urllib.request.urlopen(req) as r:
        return r.read()

def upload_audio(base, serial, filepath):
    # Endpoint
    url = f"{base}/api-sdk/play_sound?serial={serial}"
    
    # Boundary for multipart
    boundary = uuid.uuid4().hex
    headers = {
        "Content-Type": f"multipart/form-data; boundary={boundary}",
        "User-Agent": "Clawdbot-Vector-Control"
    }
    
    # Read file
    with open(filepath, "rb") as f:
        file_data = f.read()
        
    # Construct body manually
    # --boundary
    # Content-Disposition: form-data; name="sound"; filename="processed.wav"
    # Content-Type: audio/wav
    #
    # <data>
    # --boundary--
    
    body = []
    body.append(f"--{boundary}".encode())
    body.append(b'Content-Disposition: form-data; name="sound"; filename="processed.wav"')
    body.append(b"Content-Type: audio/wav")
    body.append(b"")
    body.append(file_data)
    body.append(f"--{boundary}--".encode())
    body.append(b"")
    
    payload = b"\r\n".join(body)
    
    req = urllib.request.Request(url, data=payload, headers=headers, method="POST")
    with urllib.request.urlopen(req) as r:
        return r.read()

def get(base, path, params):
    q = urllib.parse.urlencode(params)
    url = f"{base}{path}?{q}"
    with urllib.request.urlopen(url) as r:
        return r.read()

def main():
    p = argparse.ArgumentParser(description="Control Vector via Wirepod HTTP API")
    p.add_argument("--base", default="http://localhost:8080", help="Wirepod base URL")
    p.add_argument("--serial", required=True, help="Vector ESN/serial")
    sub = p.add_subparsers(dest="cmd", required=True)

    sub.add_parser("assume")
    sub.add_parser("release")

    say = sub.add_parser("say")
    say.add_argument("--text", required=True)

    move = sub.add_parser("move")
    move.add_argument("--lw", type=int, required=True)
    move.add_argument("--rw", type=int, required=True)
    move.add_argument("--time", type=float, default=0, help="Seconds to move before stop (0 = no auto-stop)")

    head = sub.add_parser("head")
    head.add_argument("--speed", type=int, required=True)
    head.add_argument("--time", type=float, default=0)

    lift = sub.add_parser("lift")
    lift.add_argument("--speed", type=int, required=True)
    lift.add_argument("--time", type=float, default=0)

    snap = sub.add_parser("snapshot")
    snap.add_argument("--out", required=True, help="Output file for MJPG stream")

    play = sub.add_parser("play")
    play.add_argument("--file", required=True, help="Path to audio file (mp3/wav/ogg)")

    patrol = sub.add_parser("patrol")
    patrol.add_argument("--steps", type=int, default=6, help="Number of move steps")
    patrol.add_argument("--speed", type=int, default=140, help="Wheel speed")
    patrol.add_argument("--step-time", type=float, default=1.2, help="Seconds per forward move")
    patrol.add_argument("--turn-time", type=float, default=0.8, help="Seconds per turn")
    patrol.add_argument("--pause", type=float, default=0.4, help="Pause between actions")
    patrol.add_argument("--speak", action="store_true", help="Speak before each move")
    patrol.add_argument("--phrase", default="Patrolling", help="Base phrase for speech")

    explore = sub.add_parser("explore")
    explore.add_argument("--steps", type=int, default=8)
    explore.add_argument("--speed-min", type=int, default=110)
    explore.add_argument("--speed-max", type=int, default=190)
    explore.add_argument("--move-min", type=float, default=0.8)
    explore.add_argument("--move-max", type=float, default=1.8)
    explore.add_argument("--turn-min", type=float, default=0.6)
    explore.add_argument("--turn-max", type=float, default=1.4)
    explore.add_argument("--pause", type=float, default=0.4)
    explore.add_argument("--speak", action="store_true")
    explore.add_argument("--phrase", default="Exploring")

    args = p.parse_args()
    base = args.base.rstrip("/")
    serial = args.serial

    if args.cmd == "assume":
        post(base, "/api-sdk/assume_behavior_control", {"priority":"high","serial":serial})
    elif args.cmd == "release":
        post(base, "/api-sdk/release_behavior_control", {"serial":serial})
    elif args.cmd == "say":
        post(base, "/api-sdk/say_text", {"text":args.text, "serial":serial})
    elif args.cmd == "move":
        post(base, "/api-sdk/move_wheels", {"lw":args.lw, "rw":args.rw, "serial":serial})
        if args.time and args.time > 0:
            time.sleep(args.time)
            post(base, "/api-sdk/move_wheels", {"lw":0, "rw":0, "serial":serial})
    elif args.cmd == "head":
        post(base, "/api-sdk/move_head", {"speed":args.speed, "serial":serial})
        if args.time and args.time > 0:
            time.sleep(args.time)
            post(base, "/api-sdk/move_head", {"speed":0, "serial":serial})
    elif args.cmd == "lift":
        post(base, "/api-sdk/move_lift", {"speed":args.speed, "serial":serial})
        if args.time and args.time > 0:
            time.sleep(args.time)
            post(base, "/api-sdk/move_lift", {"speed":0, "serial":serial})
    elif args.cmd == "snapshot":
        data = get(base, "/cam-stream", {"serial":serial})
        with open(args.out, "wb") as f:
            f.write(data)
    elif args.cmd == "play":
        # Convert audio to required format: WAV, mono, 8kHz, PCM 16-bit
        temp_wav = f"/tmp/vector_audio_{uuid.uuid4().hex}.wav"
        print(f"Processing audio: {args.file} -> {temp_wav}")
        try:
            # ffmpeg -i input -ac 1 -ar 8000 -acodec pcm_s16le output.wav
            subprocess.run([
                "ffmpeg", "-y", "-i", args.file, 
                "-ac", "1", "-ar", "8000", 
                "-f", "wav", "-c:a", "pcm_s16le", 
                "-loglevel", "error",
                temp_wav
            ], check=True)
            
            print("Uploading to Vector...")
            upload_audio(base, serial, temp_wav)
            print("Audio sent successfully.")
        except subprocess.CalledProcessError as e:
            print(f"Error converting audio: {e}")
        except Exception as e:
            print(f"Error uploading audio: {e}")
        finally:
            if os.path.exists(temp_wav):
                os.remove(temp_wav)
    elif args.cmd == "patrol":
        post(base, "/api-sdk/assume_behavior_control", {"priority":"high","serial":serial})
        for i in range(args.steps):
            if args.speak:
                post(base, "/api-sdk/say_text", {"text":f"{args.phrase} {i+1}", "serial":serial})
                time.sleep(1.2)
            post(base, "/api-sdk/move_wheels", {"lw":args.speed, "rw":args.speed, "serial":serial})
            time.sleep(args.step_time)
            post(base, "/api-sdk/move_wheels", {"lw":0, "rw":0, "serial":serial})
            time.sleep(args.pause)
            # alternate left/right turn
            if i % 2 == 0:
                post(base, "/api-sdk/move_wheels", {"lw":60, "rw":args.speed, "serial":serial})
            else:
                post(base, "/api-sdk/move_wheels", {"lw":args.speed, "rw":60, "serial":serial})
            time.sleep(args.turn_time)
            post(base, "/api-sdk/move_wheels", {"lw":0, "rw":0, "serial":serial})
            time.sleep(args.pause)
    elif args.cmd == "explore":
        post(base, "/api-sdk/assume_behavior_control", {"priority":"high","serial":serial})
        for i in range(args.steps):
            if args.speak:
                post(base, "/api-sdk/say_text", {"text":f"{args.phrase} {i+1}", "serial":serial})
                time.sleep(1.2)
            speed = random.randint(args.speed_min, args.speed_max)
            move_t = random.uniform(args.move_min, args.move_max)
            turn_t = random.uniform(args.turn_min, args.turn_max)
            post(base, "/api-sdk/move_wheels", {"lw":speed, "rw":speed, "serial":serial})
            time.sleep(move_t)
            post(base, "/api-sdk/move_wheels", {"lw":0, "rw":0, "serial":serial})
            time.sleep(args.pause)
            # random turn direction
            if random.random() < 0.5:
                post(base, "/api-sdk/move_wheels", {"lw":60, "rw":speed, "serial":serial})
            else:
                post(base, "/api-sdk/move_wheels", {"lw":speed, "rw":60, "serial":serial})
            time.sleep(turn_t)
            post(base, "/api-sdk/move_wheels", {"lw":0, "rw":0, "serial":serial})
            time.sleep(args.pause)
    else:
        p.print_help(); sys.exit(2)

if __name__ == "__main__":
    main()
