#!/usr/bin/env python3
"""
Vapi Call Manager - Native, Lightweight & Robust
Optimized per Vapi recommendations with Safety Watchdogs
"""
import os
import sys
import json
import requests
import threading
import time
from datetime import datetime
from http.server import HTTPServer, BaseHTTPRequestHandler

# ============================================================================
# CONFIGURATION
# ============================================================================
VAPI_API_KEY = os.environ.get("VAPI_API_KEY")
ASSISTANT_ID = os.environ.get("VAPI_ASSISTANT_ID")
PHONE_NUMBER_ID = os.environ.get("VAPI_PHONE_NUMBER_ID")
WEBHOOK_PORT = int(os.environ.get("WEBHOOK_PORT", "4430"))
WEBHOOK_BASE_URL = os.environ.get("WEBHOOK_BASE_URL", "").rstrip("/")

# LLM Configuration (Defaults to OpenAI / GPT-4o Mini)
LLM_PROVIDER = os.environ.get("VAPI_LLM_PROVIDER", "openai")
LLM_MODEL = os.environ.get("VAPI_LLM_MODEL", "gpt-4o-mini")

# Safety Timeouts
INACTIVITY_TIMEOUT = 60  # Abort if no webhook received for 60s
POST_CALL_TIMEOUT = 30   # Max wait for report after 'ended' status
CALL_TIMEOUT = 300       # Global timeout (5 minutes)

# Global State
call_result = None
call_completed = threading.Event()
expected_call_id = None
last_activity_time = time.time()

class WebhookHandler(BaseHTTPRequestHandler):
    def log_message(self, format, *args):
        pass # Silence HTTP logs

    def do_POST(self):
        global call_result, expected_call_id, last_activity_time
        
        # Keep connection alive
        last_activity_time = time.time()
        
        try:
            content_length = int(self.headers.get('Content-Length', 0))
            body = json.loads(self.rfile.read(content_length))
            message = body.get("message", {})
            event_type = message.get("type")
            
            # Robust ID extraction
            call_id = message.get("call", {}).get("id") or body.get("call", {}).get("id")

            # Only process events for OUR call
            if call_id == expected_call_id:
                
                # 1. Watchdog: If call ended physically, set a short timeout for the report
                if event_type == "status-update" and message.get("status") == "ended":
                    # If report doesn't arrive in 30s, force finish
                    threading.Timer(POST_CALL_TIMEOUT, lambda: call_completed.set()).start()

                # 2. End of Call Report: The Goal
                elif event_type == "end-of-call-report":
                    call_result = {
                        "status": "completed",
                        "call_id": call_id,
                        "transcript": message.get("transcript", ""),
                        "summary": message.get("summary", ""),
                        "cost": message.get("cost", 0),
                        "ended_reason": message.get("endedReason", ""),
                        "duration": 0
                    }
                    # Calculate duration
                    started_at = message.get("startedAt")
                    ended_at = message.get("endedAt")
                    if started_at and ended_at:
                        try:
                            start_dt = datetime.fromisoformat(started_at.replace('Z', '+00:00'))
                            end_dt = datetime.fromisoformat(ended_at.replace('Z', '+00:00'))
                            call_result["duration"] = (end_dt - start_dt).total_seconds()
                        except: pass
                    
                    call_completed.set()
                
                # 3. Tool Calls: Intercept tools if Vapi sends them
                elif event_type == "tool-calls":
                    tool_calls = message.get("toolCalls", [])
                    results = []
                    for tc in tool_calls:
                        if tc.get("function", {}).get("name") == "endCall" or tc.get("type") == "endCall":
                             results.append({
                                 "toolCallId": tc["id"],
                                 "result": "Call ended."
                             })
                    
                    self.send_response(200)
                    self.send_header('Content-Type', 'application/json')
                    self.end_headers()
                    self.wfile.write(json.dumps({"results": results}).encode('utf-8'))
                    return

            self.send_response(200)
            self.send_header('Content-Type', 'application/json')
            self.end_headers()
            self.wfile.write(b'{"status":"ok"}')
            
        except Exception as e:
            self.send_response(200)
            self.end_headers()

def monitor_inactivity():
    """Background thread to detect dead connections"""
    while not call_completed.is_set():
        if time.time() - last_activity_time > INACTIVITY_TIMEOUT:
            global call_result
            if not call_result:
                call_result = {"status": "timeout", "error": f"Inactivity timeout ({INACTIVITY_TIMEOUT}s)"}
            call_completed.set()
            break
        time.sleep(2)

def make_call(number, first_message, prompt, end_message="Goodbye"):
    global expected_call_id, call_result, call_completed, last_activity_time
    
    # Validate required configuration
    missing_vars = []
    if not VAPI_API_KEY: missing_vars.append("VAPI_API_KEY")
    if not ASSISTANT_ID: missing_vars.append("VAPI_ASSISTANT_ID")
    if not PHONE_NUMBER_ID: missing_vars.append("VAPI_PHONE_NUMBER_ID")
    if not WEBHOOK_BASE_URL: missing_vars.append("WEBHOOK_BASE_URL")
    
    if missing_vars:
        return {"status": "error", "error": f"Missing environment variables: {', '.join(missing_vars)}"}
    
    call_result = None
    call_completed.clear()
    last_activity_time = time.time()
    
    # Start server
    server = HTTPServer(('0.0.0.0', WEBHOOK_PORT), WebhookHandler)
    threading.Thread(target=server.serve_forever, daemon=True).start()
    
    # Start watchdog
    threading.Thread(target=monitor_inactivity, daemon=True).start()
    
    # Build Overrides
    overrides = {
        "serverUrl": f"{WEBHOOK_BASE_URL}/vapi-webhook",
    }

    if first_message != "DEFAULT":
        overrides["firstMessage"] = first_message
    
    if prompt != "DEFAULT":
        overrides["model"] = {
            "provider": LLM_PROVIDER,
            "model": LLM_MODEL,
            "messages": [{"role": "system", "content": prompt}],
            "tools": [
                {
                    "type": "endCall",
                    "messages": [
                        {"type": "request-start", "content": end_message if end_message != "DEFAULT" else "Ending call..."}
                    ]
                }
            ]
        }
        
    if end_message != "DEFAULT" and end_message:
        overrides["endCallMessage"] = end_message

    try:
        response = requests.post(
            "https://api.vapi.ai/call",
            headers={"Authorization": f"Bearer {VAPI_API_KEY}"},
            json={
                "assistantId": ASSISTANT_ID,
                "phoneNumberId": PHONE_NUMBER_ID,
                "customer": {"number": number},
                "assistantOverrides": overrides
            },
            timeout=30
        )
        
        if not response.ok:
            return {"status": "error", "error": f"API {response.status_code}: {response.text}"}
            
        expected_call_id = response.json().get("id")
        
        # Wait for completion
        if call_completed.wait(timeout=CALL_TIMEOUT):
            return call_result or {"status": "ended_no_report", "error": "Call ended but report was not received."}
        else:
            return {"status": "timeout", "call_id": expected_call_id}
            
    finally:
        server.shutdown()
        server.server_close()

def main():
    if len(sys.argv) < 4:
        print(json.dumps({"error": "Usage: python vapi_calls.py <phone> <msg> <prompt> [end_msg]"}), indent=2)
        sys.exit(1)
        
    phone = sys.argv[1]
    msg = sys.argv[2]
    prompt = sys.argv[3]
    end_msg = sys.argv[4] if len(sys.argv) > 4 else "Goodbye"
    
    result = make_call(phone, msg, prompt, end_msg)
    
    # Save Log
    try:
        log_dir = os.path.expanduser("~/.openclaw/workspace/logs/vapi-calls")
        os.makedirs(log_dir, exist_ok=True)
        fname = f"{log_dir}/{datetime.now().strftime('%Y-%m-%d_%H-%M-%S')}_{result.get('call_id','unknown')}.json"
        with open(fname, 'w') as f: json.dump(result, f, indent=2)
        result["log_file"] = fname
    except: pass

    print(json.dumps(result, indent=2))
    sys.exit(0 if result.get("status") == "completed" else 1)

if __name__ == "__main__":
    main()