#!/usr/bin/env python3
"""
Test script for user-cognitive-profiles skill using WildChat-1M dataset.

This script:
1. Streams conversations from WildChat-1M dataset
2. Groups by hashed_ip (user identifier)
3. Analyzes up to 4 unique users with ~5 conversations each
4. Generates cognitive profiles for each user

Usage:
    python3 test_wildchat.py
    python3 test_wildchat.py --users 4 --conversations 5
    python3 test_wildchat.py --output results.json
"""

import json
import argparse
import sys
from pathlib import Path
from collections import defaultdict
from typing import Dict, List, Any, Optional

# Add the skill scripts to path
sys.path.insert(0, str(Path(__file__).parent / 'scripts'))

from analyze_profile import ConversationAnalyzer, CognitiveProfiler


def load_wildchat_streaming(num_users: int = 4, convs_per_user: int = 5, min_archetypes: int = 2):
    """
    Load conversations from WildChat-1M dataset in streaming mode.
    
    Args:
        num_users: Number of unique users to analyze
        convs_per_user: Max conversations per user
        min_archetypes: Minimum conversations needed for clustering (must be >= n_clusters)
        
    Returns:
        Dict mapping hashed_ip -> list of conversations
    """
    try:
        from datasets import load_dataset
    except ImportError:
        print("❌ Error: 'datasets' library not installed.")
        print("   Install with: pip install datasets")
        sys.exit(1)
    
    print("🌊 Loading WildChat-1M dataset in streaming mode...")
    print("   (This saves RAM by not loading the full 1M conversations)\n")
    
    # Load dataset in streaming mode
    dataset = load_dataset("allenai/WildChat-1M", split="train", streaming=True)
    
    # Collect conversations by user
    user_conversations = defaultdict(list)
    users_complete = set()
    total_processed = 0
    max_iterations = 50000  # Safety limit
    
    # Ensure we collect enough for clustering
    target_convs = max(convs_per_user, min_archetypes)
    
    print(f"📊 Collecting ~{target_convs}+ conversations from {num_users} unique users...")
    print(f"   (Each user needs at least {min_archetypes} conversations for clustering)\n")
    
    for i, conv in enumerate(dataset):
        # Safety limit
        if i > max_iterations:
            print(f"   ⚠️  Reached iteration limit ({max_iterations}), stopping")
            break
        
        # Stop entirely if we have enough complete users
        if len(users_complete) >= num_users:
            break
        
        # Extract user identifier
        hashed_ip = conv.get('hashed_ip', 'unknown')
        
        # Skip if this is a new user and we already have enough users being tracked
        # (We allow existing users to continue collecting, but don't start new ones)
        if len(user_conversations) >= num_users * 2 and hashed_ip not in user_conversations:
            continue
            
        # Skip if this user is already complete
        if hashed_ip in users_complete:
            continue
        
        # Convert WildChat format to our expected format
        formatted_conv = format_wildchat_conversation(conv)
        if not formatted_conv:
            continue
            
        user_conversations[hashed_ip].append(formatted_conv)
        total_processed += 1
        
        # Mark user as complete if they have enough conversations
        if len(user_conversations[hashed_ip]) >= target_convs:
            if hashed_ip not in users_complete:
                users_complete.add(hashed_ip)
                actual_convs = len(user_conversations[hashed_ip])
                print(f"   ✓ User {len(users_complete)}/{num_users}: {hashed_ip[:16]}... ({actual_convs} conversations)")
    
    # Filter to only complete users with enough conversations
    result = {}
    for hashed_ip, convs in user_conversations.items():
        if hashed_ip in users_complete and len(convs) >= min_archetypes:
            result[hashed_ip] = convs[:convs_per_user]  # Limit to requested number
    
    print(f"\n✅ Collected {len(result)} complete users with {sum(len(c) for c in result.values())} total conversations")
    
    if len(result) < num_users:
        print(f"   ⚠️  Warning: Only found {len(result)}/{num_users} users with sufficient data")
    
    return result


def format_wildchat_conversation(conv: Dict[str, Any]) -> Optional[Dict[str, Any]]:
    """
    Convert WildChat format to our expected conversation format.
    
    WildChat format:
    {
        'hashed_ip': '...',
        'conversation': [
            {'role': 'user', 'content': '...'},
            {'role': 'assistant', 'content': '...'},
            ...
        ],
        'conversation_hash': '...',
        'model': '...',
        'language': '...',
        'timestamp': '...'
    }
    
    Our format:
    {
        'id': '...',
        'title': '...',
        'messages': [
            {'role': 'user'|'assistant', 'content': '...'},
            ...
        ]
    }
    """
    conversation_data = conv.get('conversation', [])
    
    if not conversation_data or len(conversation_data) < 2:
        return None
    
    # Extract first user message for title
    first_user_msg = None
    for msg in conversation_data:
        if msg.get('role') == 'user':
            first_user_msg = msg.get('content', '')
            break
    
    # Create title from first message (truncated)
    title = first_user_msg[:50] + '...' if first_user_msg and len(first_user_msg) > 50 else (first_user_msg or 'Untitled')
    
    formatted = {
        'id': conv.get('conversation_hash', 'unknown'),
        'title': title,
        'messages': [
            {'role': msg.get('role'), 'content': msg.get('content', '')}
            for msg in conversation_data
            if msg.get('role') in ['user', 'assistant']
        ],
        'timestamp': conv.get('timestamp'),
        'model': conv.get('model'),
        'language': conv.get('language')
    }
    
    return formatted


def analyze_user_profile(hashed_ip: str, conversations: List[Dict], 
                        num_archetypes: int = 2) -> Optional[Dict[str, Any]]:
    """
    Analyze a single user's conversations and generate cognitive profile.
    
    Args:
        hashed_ip: User identifier
        conversations: List of user's conversations
        num_archetypes: Number of archetypes to identify
        
    Returns:
        Cognitive profile for the user, or None if insufficient data
    """
    print(f"\n🔍 Analyzing user: {hashed_ip[:16]}...")
    print(f"   Conversations: {len(conversations)}")
    
    # Validate we have enough conversations for clustering
    if len(conversations) < num_archetypes:
        print(f"   ⚠️  Skipping: Only {len(conversations)} conversations, need at least {num_archetypes} for clustering")
        return None
    
    # Create profiler
    profiler = CognitiveProfiler(num_archetypes=num_archetypes)
    
    # Generate profile
    try:
        profile = profiler.generate_profile(conversations)
    except ValueError as e:
        print(f"   ⚠️  Analysis failed: {e}")
        return None
    
    # Add user identifier
    profile['user_id'] = hashed_ip[:16] + '...'
    profile['user_hash'] = hashed_ip
    
    # Print summary
    primary = profile['insights']['primary_mode']
    confidence = profile['insights']['primary_confidence']
    print(f"   Primary archetype: {primary} (confidence: {confidence})")
    
    return profile


def compare_users(profiles: List[Dict[str, Any]]):
    """
    Compare cognitive profiles across multiple users.
    
    Args:
        profiles: List of user profiles
    """
    print("\n" + "="*60)
    print("📊 COMPARATIVE ANALYSIS")
    print("="*60)
    
    # Collect archetype distribution
    archetype_counts = defaultdict(int)
    for profile in profiles:
        primary = profile['insights']['primary_mode']
        archetype_counts[primary] += 1
    
    print("\n🏷️  Archetype Distribution:")
    for archetype, count in sorted(archetype_counts.items(), key=lambda x: x[1], reverse=True):
        pct = (count / len(profiles)) * 100
        print(f"   {archetype}: {count}/{len(profiles)} users ({pct:.0f}%)")
    
    # Compare metrics
    print("\n📏 Communication Patterns:")
    for profile in profiles:
        user_id = profile['user_id']
        primary = profile['insights']['primary_mode']
        
        # Get primary archetype metrics
        primary_arch = next((a for a in profile['archetypes'] if a['name'] == primary), None)
        if primary_arch:
            metrics = primary_arch['metrics']
            print(f"\n   User {user_id} ({primary}):")
            print(f"      Avg message length: {metrics['avg_message_length']:.0f} words")
            print(f"      Question ratio: {metrics['question_ratio']:.2f}")
            print(f"      Code blocks: {metrics['code_block_ratio']:.2f}")
    
    # Context switching analysis
    print("\n🔄 Context Switching:")
    for profile in profiles:
        user_id = profile['user_id']
        switching = profile['insights']['context_switching']
        num_shifts = len(profile.get('context_shifts', []))
        print(f"   User {user_id}: {switching} ({num_shifts} shifts detected)")


def main():
    parser = argparse.ArgumentParser(
        description='Test user-cognitive-profiles skill with WildChat-1M dataset',
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  %(prog)s                              # Analyze 4 users, 5 conversations each
  %(prog)s --users 2 --conversations 10 # Analyze 2 users with 10 conversations
  %(prog)s --archetypes 3               # Identify 3 archetypes per user
  %(prog)s --output results.json        # Save results to JSON
        """
    )
    
    parser.add_argument('--users', '-u', type=int, default=4,
                       help='Number of unique users to analyze (default: 4)')
    parser.add_argument('--conversations', '-c', type=int, default=5,
                       help='Conversations per user (default: 5)')
    parser.add_argument('--archetypes', '-a', type=int, default=2,
                       help='Number of archetypes to identify per user (default: 2)')
    parser.add_argument('--output', '-o',
                       help='Output JSON file for results')
    parser.add_argument('--compare', action='store_true',
                       help='Show comparative analysis between users')
    
    args = parser.parse_args()
    
    print("="*60)
    print("🤖🤝🧠 WildChat User Cognitive Profile Tester")
    print("="*60)
    print(f"\nConfiguration:")
    print(f"   Users to analyze: {args.users}")
    print(f"   Conversations per user: {args.conversations}")
    print(f"   Archetypes per user: {args.archetypes}\n")
    
    # Load data (ensure enough conversations for clustering)
    user_conversations = load_wildchat_streaming(
        num_users=args.users, 
        convs_per_user=args.conversations,
        min_archetypes=args.archetypes
    )
    
    if not user_conversations:
        print("❌ No conversations loaded!")
        sys.exit(1)
    
    # Analyze each user
    print("\n" + "="*60)
    print("🧠 GENERATING COGNITIVE PROFILES")
    print("="*60)
    
    profiles = []
    for hashed_ip, conversations in user_conversations.items():
        profile = analyze_user_profile(hashed_ip, conversations, args.archetypes)
        if profile:
            profiles.append(profile)
    
    if not profiles:
        print("\n❌ No profiles could be generated. Try:")
        print("   - Reducing --archetypes (must be <= conversations per user)")
        print("   - Increasing --conversations")
        print("   - Using a smaller --users count")
        sys.exit(1)
    
    print(f"\n✅ Successfully generated {len(profiles)} profiles")
    
    # Comparative analysis
    if args.compare and len(profiles) > 1:
        compare_users(profiles)
    
    # Summary
    print("\n" + "="*60)
    print("📋 SUMMARY")
    print("="*60)
    
    for profile in profiles:
        user_id = profile['user_id']
        primary = profile['insights']['primary_mode']
        confidence = profile['insights']['primary_confidence']
        switching = profile['insights']['context_switching']
        
        print(f"\n👤 User {user_id}:")
        print(f"   Primary: {primary} ({confidence:.0%} confidence)")
        print(f"   Context switching: {switching}")
        print(f"   Preferences:")
        for pref in profile['insights']['communication_preferences'][:3]:
            print(f"      • {pref}")
    
    # Save results if requested
    if args.output:
        results = {
            'metadata': {
                'dataset': 'WildChat-1M',
                'users_analyzed': len(profiles),
                'conversations_per_user': args.conversations,
                'archetypes_per_user': args.archetypes
            },
            'profiles': profiles
        }
        
        with open(args.output, 'w', encoding='utf-8') as f:
            json.dump(results, f, indent=2, ensure_ascii=False)
        
        print(f"\n💾 Results saved to: {args.output}")
    
    print("\n" + "="*60)
    print("✅ Analysis complete!")
    print("="*60)


if __name__ == '__main__':
    main()
