#!/usr/bin/env python3
"""
Compare two cognitive profiles to track evolution over time.

Usage:
    python3 compare_profiles.py profile_jan.json profile_jun.json
"""

import json
import argparse
from pathlib import Path
from typing import Dict, Any


def load_profile(filepath: str) -> Dict[str, Any]:
    """Load a profile JSON file."""
    with open(filepath, 'r') as f:
        return json.load(f)


def compare_archetypes(old_profile: Dict, new_profile: Dict):
    """Compare archetype distributions between profiles."""
    print("\n📊 Archetype Comparison")
    print("-" * 60)
    
    old_archetypes = {a['name']: a for a in old_profile['archetypes']}
    new_archetypes = {a['name']: a for a in new_profile['archetypes']}
    
    all_names = set(old_archetypes.keys()) | set(new_archetypes.keys())
    
    for name in sorted(all_names):
        old = old_archetypes.get(name)
        new = new_archetypes.get(name)
        
        if old and new:
            old_count = old['metrics']['conversation_count']
            new_count = new['metrics']['conversation_count']
            change = new_count - old_count
            pct_change = (change / old_count * 100) if old_count > 0 else 0
            
            print(f"  {name}:")
            print(f"    {old_count} → {new_count} conversations ({change:+d}, {pct_change:+.1f}%)")
        elif old:
            print(f"  {name}: [DECREASED] {old['metrics']['conversation_count']} → 0")
        else:
            print(f"  {name}: [NEW] 0 → {new['metrics']['conversation_count']}")


def compare_primary_mode(old_profile: Dict, new_profile: Dict):
    """Compare primary archetype changes."""
    print("\n🎯 Primary Mode Shift")
    print("-" * 60)
    
    old_primary = old_profile['insights']['primary_mode']
    new_primary = new_profile['insights']['primary_mode']
    
    if old_primary == new_primary:
        print(f"  Stable: {old_primary}")
        old_conf = old_profile['insights']['primary_confidence']
        new_conf = new_profile['insights']['primary_confidence']
        print(f"  Confidence: {old_conf} → {new_conf}")
    else:
        print(f"  SHIFT: {old_primary} → {new_primary}")
        print(f"  This suggests a significant change in communication style!")


def compare_context_switching(old_profile: Dict, new_profile: Dict):
    """Compare context switching patterns."""
    print("\n🔄 Context Switching")
    print("-" * 60)
    
    old_cs = old_profile['insights']['context_switching']
    new_cs = new_profile['insights']['context_switching']
    
    print(f"  {old_cs} → {new_cs}")
    
    if old_cs == new_cs:
        print(f"  Consistent context-switching behavior")
    elif new_cs == 'high':
        print(f"  ⚠️  Increased context switching - more mode variation")
    elif new_cs == 'low':
        print(f"  ✓ Decreased context switching - more consistent style")


def generate_diff_report(old_profile: Dict, new_profile: Dict) -> str:
    """Generate a markdown diff report."""
    report = """# Cognitive Profile Evolution Report

## Summary
"""
    
    old_date = old_profile['metadata']['generated_at']
    new_date = new_profile['metadata']['generated_at']
    
    report += f"""
- **Period:** {old_date} → {new_date}
- **Primary Archetype:** {old_profile['insights']['primary_mode']} → {new_profile['insights']['primary_mode']}
- **Context Switching:** {old_profile['insights']['context_switching']} → {new_profile['insights']['context_switching']}

## Changes in Communication Style
"""
    
    # Compare preferences
    old_prefs = set(old_profile['insights']['communication_preferences'])
    new_prefs = set(new_profile['insights']['communication_preferences'])
    
    added = new_prefs - old_prefs
    removed = old_prefs - new_prefs
    
    if added:
        report += "\n### New Preferences\n"
        for pref in added:
            report += f"- ✓ {pref}\n"
    
    if removed:
        report += "\n### Preferences No Longer Detected\n"
        for pref in removed:
            report += f"- ✗ {pref}\n"
    
    report += "\n## Recommendations\n"
    
    if new_profile['insights']['primary_mode'] != old_profile['insights']['primary_mode']:
        report += """
Your primary communication archetype has shifted. Consider updating your agent configuration
to reflect this new pattern.
"""
    else:
        report += """
Your primary communication style remains consistent. Continue using the current agent
configuration.
"""
    
    return report


def main():
    parser = argparse.ArgumentParser(description='Compare two cognitive profiles')
    parser.add_argument('old_profile', help='Path to older profile JSON')
    parser.add_argument('new_profile', help='Path to newer profile JSON')
    parser.add_argument('--output', '-o', help='Output path for diff report')
    
    args = parser.parse_args()
    
    print("=" * 60)
    print("🤖🤝🧠 Profile Comparison")
    print("=" * 60)
    
    old_profile = load_profile(args.old_profile)
    new_profile = load_profile(args.new_profile)
    
    print(f"\nComparing:")
    print(f"  Old: {old_profile['metadata']['generated_at']}")
    print(f"  New: {new_profile['metadata']['generated_at']}")
    
    compare_primary_mode(old_profile, new_profile)
    compare_archetypes(old_profile, new_profile)
    compare_context_switching(old_profile, new_profile)
    
    # Generate report
    report = generate_diff_report(old_profile, new_profile)
    
    if args.output:
        with open(args.output, 'w') as f:
            f.write(report)
        print(f"\n✓ Report saved to: {args.output}")
    else:
        print("\n" + "=" * 60)
        print(report)


if __name__ == '__main__':
    main()
