#!/usr/bin/env python3
"""
User Cognitive Profile Analyzer

Analyzes ChatGPT conversation exports to identify cognitive archetypes
and communication patterns for optimized AI-human collaboration.

Usage:
    python3 analyze_profile.py --input conversations.json --output profile.json
    python3 analyze_profile.py --input conversations.json --archetypes 5
    python3 analyze_profile.py --input conversations.json --archetypes-config custom.yaml
"""

import json
import argparse
import re
from datetime import datetime
from collections import defaultdict
from pathlib import Path
from typing import List, Dict, Any, Optional
import random

# Optional dependencies with fallbacks
try:
    from sklearn.cluster import KMeans
    from sklearn.preprocessing import StandardScaler
    SKLEARN_AVAILABLE = True
except ImportError:
    SKLEARN_AVAILABLE = False
    print("Warning: scikit-learn not installed. Using fallback clustering.")

try:
    import numpy as np
    NUMPY_AVAILABLE = True
except ImportError:
    NUMPY_AVAILABLE = False
    print("Warning: numpy not installed. Using basic statistics.")


def tokenize(text: str) -> List[str]:
    """Simple tokenization for BM25."""
    return re.findall(r'\b[a-zA-Z]{3,}\b', text.lower())


class BM25:
    """
    BM25 ranking function for information retrieval.
    
    BM25 is better than simple TF-IDF because:
    - Term frequency saturation (repeated words don't keep adding value)
    - Document length normalization
    - Proven effectiveness for information retrieval
    
    Based on: Okapi BM25 (Robertson et al., 1994)
    """
    
    def __init__(self, corpus: List[str], k1: float = 1.5, b: float = 0.75):
        """
        Initialize BM25 with a corpus of documents.
        
        Args:
            corpus: List of document strings
            k1: Term frequency saturation parameter (default: 1.5)
            b: Length normalization parameter (default: 0.75)
        """
        self.k1 = k1
        self.b = b
        self.corpus = corpus
        self.tokenized_corpus = [tokenize(doc) for doc in corpus]
        self.doc_count = len(corpus)
        
        # Calculate document frequencies
        self.df = defaultdict(int)
        for doc in self.tokenized_corpus:
            seen = set(doc)
            for term in seen:
                self.df[term] += 1
        
        # Calculate document lengths and average length
        self.doc_lengths = [len(doc) for doc in self.tokenized_corpus]
        self.avg_doc_length = sum(self.doc_lengths) / self.doc_count if self.doc_count > 0 else 0
        
        # Precompute IDF values
        self.idf = {}
        for term, df in self.df.items():
            # BM25 IDF formula
            self.idf[term] = np.log((self.doc_count - df + 0.5) / (df + 0.5) + 1.0) if NUMPY_AVAILABLE else \
                           __import__('math').log((self.doc_count - df + 0.5) / (df + 0.5) + 1.0)
    
    def get_scores(self, query: str) -> List[float]:
        """
        Get BM25 scores for a query against all documents.
        
        Args:
            query: Query string
            
        Returns:
            List of scores (one per document)
        """
        query_tokens = tokenize(query)
        scores = [0.0] * self.doc_count
        
        for term in query_tokens:
            if term not in self.idf:
                continue
                
            idf = self.idf[term]
            
            for idx, doc in enumerate(self.tokenized_corpus):
                term_freq = doc.count(term)
                if term_freq == 0:
                    continue
                    
                # BM25 term frequency component
                doc_len = self.doc_lengths[idx]
                norm_factor = 1 - self.b + self.b * (doc_len / self.avg_doc_length) if self.avg_doc_length > 0 else 1
                tf_component = (term_freq * (self.k1 + 1)) / (term_freq + self.k1 * norm_factor)
                
                scores[idx] += idf * tf_component
        
        return scores
    
    def get_top_n(self, query: str, n: int = 5) -> List[tuple]:
        """
        Get top N documents for a query.
        
        Args:
            query: Query string
            n: Number of top documents to return
            
        Returns:
            List of (doc_index, score) tuples
        """
        scores = self.get_scores(query)
        indexed_scores = list(enumerate(scores))
        indexed_scores.sort(key=lambda x: x[1], reverse=True)
        return indexed_scores[:n]


class ConversationAnalyzer:
    """Analyzes individual conversations for feature extraction."""
    
    def __init__(self):
        self.word_pattern = re.compile(r'\b\w+\b')
        self.code_pattern = re.compile(r'```[\s\S]*?```|`[^`]+`')
        self.question_pattern = re.compile(r'\?')
        self.url_pattern = re.compile(r'https?://\S+')
        
    def extract_features(self, conversation: Dict[str, Any], corpus: Optional[List[str]] = None) -> Dict[str, Any]:
        """Extract features from a single conversation."""
        messages = conversation.get('messages', [])
        user_messages = [m for m in messages if m.get('role') == 'user']
        assistant_messages = [m for m in messages if m.get('role') == 'assistant']
        
        if not user_messages:
            return None
            
        # Message lengths
        user_lengths = [len(self.word_pattern.findall(m.get('content', ''))) 
                       for m in user_messages]
        assistant_lengths = [len(self.word_pattern.findall(m.get('content', ''))) 
                            for m in assistant_messages]
        
        # Content analysis
        user_content = ' '.join([m.get('content', '') for m in user_messages])
        
        # Features
        features = {
            'conversation_id': conversation.get('id', 'unknown'),
            'title': conversation.get('title', 'Untitled'),
            'num_turns': len(messages),
            'num_user_messages': len(user_messages),
            'avg_user_message_length': sum(user_lengths) / len(user_lengths) if user_lengths else 0,
            'max_user_message_length': max(user_lengths) if user_lengths else 0,
            'total_user_words': sum(user_lengths),
            'avg_assistant_message_length': sum(assistant_lengths) / len(assistant_lengths) if assistant_lengths else 0,
            'question_ratio': sum(1 for m in user_messages if self.question_pattern.search(m.get('content', ''))) / len(user_messages),
            'code_block_count': len(self.code_pattern.findall(user_content)),
            'url_count': len(self.url_pattern.findall(user_content)),
            'first_message': user_messages[0].get('content', '')[:200] if user_messages else '',
            'keywords': self._extract_keywords(user_content, corpus),
            'timestamp': self._parse_timestamp(conversation),
        }
        
        return features
    
    def _extract_keywords(self, text: str, corpus: Optional[List[str]] = None, top_n: int = 10) -> List[str]:
        """
        Extract top keywords from text using BM25 if corpus available,
        otherwise fall back to frequency-based extraction.
        
        Args:
            text: Text to extract keywords from
            corpus: Optional corpus for BM25 ranking (other conversations)
            top_n: Number of keywords to return
        """
        # Filter common stop words
        stop_words = {
            'the', 'a', 'an', 'and', 'or', 'but', 'in', 'on', 'at', 'to', 'for',
            'of', 'with', 'by', 'is', 'are', 'was', 'were', 'be', 'been', 'have',
            'has', 'had', 'do', 'does', 'did', 'will', 'would', 'could', 'should',
            'i', 'you', 'he', 'she', 'it', 'we', 'they', 'this', 'that', 'these',
            'those', 'my', 'your', 'his', 'her', 'its', 'our', 'their', 'can', 'me',
            'than', 'then', 'them', 'from', 'that', 'this', 'with', 'have', 'been',
            'were', 'said', 'each', 'which', 'their', 'time', 'will', 'about',
            'there', 'all', 'also', 'what', 'when', 'where', 'how', 'who', 'why'
        }
        
        # If corpus provided, use BM25 for better keyword extraction
        if corpus and len(corpus) > 1:
            try:
                bm25 = BM25(corpus)
                # Use the text itself as a query to find distinctive terms
                tokens = tokenize(text.lower())
                unique_tokens = [t for t in set(tokens) if t not in stop_words and len(t) > 3]
                
                # Score each unique token
                term_scores = []
                for term in unique_tokens:
                    score = sum(bm25.get_scores(term))
                    term_scores.append((term, score))
                
                term_scores.sort(key=lambda x: x[1], reverse=True)
                return [term for term, _ in term_scores[:top_n]]
            except Exception:
                # Fall back to frequency-based
                pass
        
        # Frequency-based fallback
        words = self.word_pattern.findall(text.lower())
        
        word_freq = defaultdict(int)
        for word in words:
            if len(word) > 3 and word not in stop_words:
                word_freq[word] += 1
        
        return [word for word, _ in sorted(word_freq.items(), key=lambda x: x[1], reverse=True)[:top_n]]
    
    def _parse_timestamp(self, conversation: Dict[str, Any]) -> Optional[str]:
        """Parse conversation timestamp."""
        # Try various timestamp fields
        for field in ['create_time', 'created_at', 'timestamp', 'update_time']:
            if field in conversation:
                return str(conversation[field])
        return None


class CognitiveProfiler:
    """Main class for cognitive profile generation."""
    
    def __init__(self, num_archetypes: int = 3, custom_archetypes: Optional[Dict] = None):
        self.num_archetypes = num_archetypes
        self.custom_archetypes = custom_archetypes or {}
        self.analyzer = ConversationAnalyzer()
        
        # Default archetype definitions
        self.default_archetypes = {
            'efficiency_optimizer': {
                'name': 'Efficiency Optimizer',
                'keywords': ['quick', 'brief', 'simple', 'just', 'fast', 'now'],
                'patterns': ['short_messages', 'imperative_tone'],
                'description': 'Direct, action-oriented, minimal explanation needed'
            },
            'systems_architect': {
                'name': 'Systems Architect',
                'keywords': ['architecture', 'design', 'system', 'trade-off', 'structure', 'framework'],
                'patterns': ['long_messages', 'technical_depth'],
                'description': 'Analytical, comprehensive, strategic thinking'
            },
            'philosophical_explorer': {
                'name': 'Philosophical Explorer',
                'keywords': ['why', 'meaning', 'philosophy', 'question', 'think', 'believe'],
                'patterns': ['open_ended', 'abstract'],
                'description': 'Meaning-seeking, assumption-questioning, deep'
            },
            'creative_synthesizer': {
                'name': 'Creative Synthesizer',
                'keywords': ['like', 'similar', 'analogy', 'connection', 'pattern', 'imagine'],
                'patterns': ['metaphorical', 'cross_domain'],
                'description': 'Pattern-recognizing, analogical, innovative'
            },
            'collaborative_partner': {
                'name': 'Collaborative Partner',
                'keywords': ['help', 'together', 'discuss', 'opinion', 'think', 'suggest'],
                'patterns': ['questions', 'dialogue'],
                'description': 'Interactive, opinion-seeking, co-creative'
            }
        }
    
    def analyze_conversations(self, conversations: List[Dict[str, Any]], 
                             sample_size: Optional[int] = None) -> List[Dict[str, Any]]:
        """Analyze all conversations and extract features."""
        print(f"Analyzing {len(conversations)} conversations...")
        
        # Sample if needed
        if sample_size and len(conversations) > sample_size:
            conversations = random.sample(conversations, sample_size)
            print(f"Sampled to {sample_size} conversations")
        
        # Build corpus for BM25 keyword extraction
        print("Building corpus for BM25 keyword extraction...")
        corpus = []
        for conv in conversations:
            messages = conv.get('messages', [])
            user_content = ' '.join([m.get('content', '') for m in messages if m.get('role') == 'user'])
            corpus.append(user_content)
        
        features_list = []
        for i, conv in enumerate(conversations):
            if i % 500 == 0:
                print(f"  Processed {i}/{len(conversations)}...")
            
            features = self.analyzer.extract_features(conv, corpus)
            if features:
                features_list.append(features)
        
        print(f"Successfully analyzed {len(features_list)} conversations")
        return features_list
    
    def cluster_conversations(self, features: List[Dict[str, Any]]) -> Dict[int, List[Dict]]:
        """Cluster conversations into archetypes."""
        if not features:
            return {}
        
        print(f"\nClustering into {self.num_archetypes} archetypes...")
        
        if SKLEARN_AVAILABLE and NUMPY_AVAILABLE:
            return self._sklearn_cluster(features)
        else:
            return self._fallback_cluster(features)
    
    def _sklearn_cluster(self, features: List[Dict[str, Any]]) -> Dict[int, List[Dict]]:
        """Use scikit-learn for clustering."""
        # Validate we have enough samples
        if len(features) < self.num_archetypes:
            raise ValueError(
                f"Cannot cluster {len(features)} conversations into {self.num_archetypes} archetypes. "
                f"Need at least {self.num_archetypes} conversations (one per cluster)."
            )
        
        # Prepare feature matrix
        feature_vectors = []
        for f in features:
            vector = [
                f['avg_user_message_length'],
                f['max_user_message_length'],
                f['question_ratio'],
                f['code_block_count'],
                f['num_turns'],
            ]
            feature_vectors.append(vector)
        
        X = np.array(feature_vectors)
        
        # Normalize
        scaler = StandardScaler()
        X_scaled = scaler.fit_transform(X)
        
        # Cluster
        kmeans = KMeans(n_clusters=self.num_archetypes, random_state=42, n_init=10)
        labels = kmeans.fit_predict(X_scaled)
        
        # Group by cluster
        clusters = defaultdict(list)
        for i, label in enumerate(labels):
            clusters[int(label)].append(features[i])
        
        return dict(clusters)
    
    def _fallback_cluster(self, features: List[Dict[str, Any]]) -> Dict[int, List[Dict]]:
        """Simple fallback clustering without sklearn."""
        # Sort by message length and split into buckets
        sorted_features = sorted(features, key=lambda x: x['avg_user_message_length'])
        
        cluster_size = len(sorted_features) // self.num_archetypes
        clusters = {}
        
        for i in range(self.num_archetypes):
            start = i * cluster_size
            end = start + cluster_size if i < self.num_archetypes - 1 else len(sorted_features)
            clusters[i] = sorted_features[start:end]
        
        return clusters
    
    def identify_archetypes(self, clusters: Dict[int, List[Dict]]) -> Dict[int, Dict[str, Any]]:
        """Identify archetype characteristics for each cluster using BM25 for keyword relevance."""
        archetypes = {}
        
        # Build corpus from all conversations for BM25
        all_texts = []
        for cluster_id, conversations in clusters.items():
            cluster_texts = []
            for c in conversations:
                # Reconstruct conversation text from keywords
                cluster_texts.append(' '.join(c.get('keywords', [])))
            all_texts.extend(cluster_texts)
        
        # Create BM25 index if we have texts
        bm25 = None
        if all_texts and len(all_texts) > 1:
            try:
                bm25 = BM25(all_texts)
            except Exception:
                bm25 = None
        
        for cluster_id, conversations in clusters.items():
            if not conversations:
                continue
            
            # Aggregate metrics
            avg_length = sum(c['avg_user_message_length'] for c in conversations) / len(conversations)
            avg_question_ratio = sum(c['question_ratio'] for c in conversations) / len(conversations)
            avg_code_ratio = sum(c['code_block_count'] for c in conversations) / len(conversations)
            
            # Collect all keywords
            all_keywords = []
            for c in conversations:
                all_keywords.extend(c['keywords'])
            
            # Use BM25 to score keyword relevance if available
            if bm25:
                # Build query from archetype definitions
                archetype_queries = []
                for aid, adef in {**self.default_archetypes, **self.custom_archetypes}.items():
                    query = ' '.join(adef.get('keywords', []))
                    archetype_queries.append((aid, query))
                
                # Score each archetype against cluster content
                cluster_content = ' '.join(all_keywords)
                best_match = None
                best_score = -1
                
                for aid, query in archetype_queries:
                    try:
                        scores = bm25.get_scores(query)
                        # Average score for this cluster's documents
                        cluster_score = sum(scores) / len(scores) if scores else 0
                        if cluster_score > best_score:
                            best_score = cluster_score
                            best_match = aid
                    except Exception:
                        continue
            else:
                best_match = None
            
            # Find most common keywords (frequency-based fallback)
            keyword_freq = defaultdict(int)
            for kw in all_keywords:
                keyword_freq[kw] += 1
            top_keywords = sorted(keyword_freq.items(), key=lambda x: x[1], reverse=True)[:10]
            
            # Determine archetype name based on characteristics and BM25 match
            archetype_name = self._classify_archetype(avg_length, avg_question_ratio, top_keywords, best_match)
            
            # Calculate confidence based on cluster cohesion and BM25 score
            base_confidence = min(0.95, 0.5 + (len(conversations) / 1000))
            if bm25 and best_score > 0:
                # Boost confidence if we have good BM25 match
                confidence = min(0.98, base_confidence + 0.1)
            else:
                confidence = base_confidence
            
            archetypes[cluster_id] = {
                'id': cluster_id,
                'name': archetype_name['name'],
                'confidence': round(confidence, 2),
                'metrics': {
                    'avg_message_length': round(avg_length, 1),
                    'question_ratio': round(avg_question_ratio, 2),
                    'code_block_ratio': round(avg_code_ratio, 2),
                    'conversation_count': len(conversations)
                },
                'keywords': [kw for kw, _ in top_keywords[:5]],
                'sample_conversations': [c['conversation_id'] for c in conversations[:3]],
                'recommendations': archetype_name['recommendations']
            }
        
        return archetypes
    
    def _classify_archetype(self, avg_length: float, question_ratio: float, 
                           keywords: List[tuple], bm25_match: Optional[str] = None) -> Dict[str, Any]:
        """
        Classify conversation cluster into archetype.
        
        Uses BM25 match if available, otherwise falls back to metric-based classification.
        """
        keyword_set = set([kw for kw, _ in keywords])
        
        # If BM25 found a strong match, use it
        if bm25_match:
            # Check custom archetypes first
            if bm25_match in self.custom_archetypes:
                archetype_def = self.custom_archetypes[bm25_match]
                return {
                    'name': archetype_def.get('name', bm25_match),
                    'recommendations': {
                        'ai_role': archetype_def.get('ai_role', 'Assistant'),
                        'communication_style': archetype_def.get('description', 'Adaptive'),
                        'response_length': 'medium',
                        'structure': 'adaptive'
                    }
                }
            # Check default archetypes
            if bm25_match in self.default_archetypes:
                archetype_def = self.default_archetypes[bm25_match]
                return {
                    'name': archetype_def['name'],
                    'recommendations': {
                        'ai_role': archetype_def.get('ai_role', 'Assistant'),
                        'communication_style': archetype_def['description'],
                        'response_length': 'medium',
                        'structure': 'adaptive'
                    }
                }
        
        # Check custom archetypes by keyword overlap
        for archetype_id, archetype_def in self.custom_archetypes.items():
            archetype_keywords = set(archetype_def.get('keywords', []))
            if keyword_set & archetype_keywords:  # Intersection
                return {
                    'name': archetype_def.get('name', archetype_id),
                    'recommendations': {
                        'ai_role': archetype_def.get('ai_role', 'Assistant'),
                        'communication_style': archetype_def.get('description', 'Adaptive'),
                        'response_length': 'medium',
                        'structure': 'adaptive'
                    }
                }
        
        # Default classification based on metrics
        if avg_length < 50:
            return {
                'name': 'Efficiency Optimizer',
                'recommendations': {
                    'ai_role': 'Efficient Tool',
                    'communication_style': 'Direct, concise, action-oriented',
                    'response_length': 'short',
                    'structure': 'bullet_points'
                }
            }
        elif avg_length > 200 and question_ratio > 0.3:
            return {
                'name': 'Collaborative Explorer',
                'recommendations': {
                    'ai_role': 'Collaborative Partner',
                    'communication_style': 'Interactive, exploratory, co-creative',
                    'response_length': 'medium',
                    'structure': 'dialogue'
                }
            }
        elif avg_length > 200:
            return {
                'name': 'Systems Architect',
                'recommendations': {
                    'ai_role': 'Senior Architect',
                    'communication_style': 'Detailed, systematic, analytical',
                    'response_length': 'long',
                    'structure': 'hierarchical'
                }
            }
        else:
            return {
                'name': 'Balanced Communicator',
                'recommendations': {
                    'ai_role': 'Versatile Assistant',
                    'communication_style': 'Adaptive, context-aware',
                    'response_length': 'adaptive',
                    'structure': 'flexible'
                }
            }
    
    def detect_context_shifts(self, features: List[Dict[str, Any]]) -> List[Dict[str, Any]]:
        """Detect potential context shifts in conversation patterns."""
        # Sort by timestamp if available
        sorted_features = sorted(features, key=lambda x: x.get('timestamp') or '')
        
        shifts = []
        if len(sorted_features) < 10:
            return shifts
        
        # Simple detection: look for significant length changes
        for i in range(1, len(sorted_features)):
            prev = sorted_features[i-1]
            curr = sorted_features[i]
            
            if curr['avg_user_message_length'] > prev['avg_user_message_length'] * 3:
                shifts.append({
                    'trigger': 'message_length_increase',
                    'from_style': 'brief',
                    'to_style': 'detailed',
                    'timestamp': curr.get('timestamp')
                })
            elif prev['avg_user_message_length'] > curr['avg_user_message_length'] * 3:
                shifts.append({
                    'trigger': 'message_length_decrease',
                    'from_style': 'detailed',
                    'to_style': 'brief',
                    'timestamp': curr.get('timestamp')
                })
        
        return shifts[:5]  # Return top 5 shifts
    
    def generate_profile(self, conversations: List[Dict[str, Any]], 
                        sample_size: Optional[int] = None) -> Dict[str, Any]:
        """Generate complete cognitive profile."""
        # Analyze conversations
        features = self.analyze_conversations(conversations, sample_size)
        
        if not features:
            raise ValueError("No valid conversations found in input")
        
        # Cluster
        clusters = self.cluster_conversations(features)
        
        # Identify archetypes
        archetypes = self.identify_archetypes(clusters)
        
        # Detect context shifts
        shifts = self.detect_context_shifts(features)
        
        # Determine primary mode
        primary_archetype = max(archetypes.values(), key=lambda x: x['metrics']['conversation_count'])
        
        # Generate profile
        profile = {
            'metadata': {
                'generated_at': datetime.now().isoformat(),
                'total_conversations_analyzed': len(features),
                'num_archetypes': self.num_archetypes,
                'tool_version': '1.0.0'
            },
            'archetypes': list(archetypes.values()),
            'context_shifts': shifts,
            'insights': {
                'primary_mode': primary_archetype['name'],
                'primary_confidence': primary_archetype['confidence'],
                'context_switching': 'high' if len(shifts) > 3 else 'moderate' if len(shifts) > 0 else 'low',
                'communication_preferences': self._generate_preferences(primary_archetype)
            }
        }
        
        return profile
    
    def _generate_preferences(self, archetype: Dict[str, Any]) -> List[str]:
        """Generate communication preferences based on archetype."""
        prefs = []
        
        metrics = archetype['metrics']
        
        if metrics['avg_message_length'] > 200:
            prefs.append('Detailed responses with depth')
        else:
            prefs.append('Concise, direct answers')
        
        if metrics['question_ratio'] > 0.3:
            prefs.append('Collaborative dialogue')
        else:
            prefs.append('Clear directives')
        
        if metrics['code_block_ratio'] > 0.2:
            prefs.append('Technical/code examples')
        
        prefs.append('Examples + theory + hands-on steps')
        
        return prefs


def load_conversations(filepath: str) -> List[Dict[str, Any]]:
    """Load conversations from JSON file."""
    path = Path(filepath)
    
    if not path.exists():
        raise FileNotFoundError(f"Input file not found: {filepath}")
    
    with open(path, 'r', encoding='utf-8') as f:
        data = json.load(f)
    
    # Handle different export formats
    if isinstance(data, list):
        return data
    elif isinstance(data, dict) and 'conversations' in data:
        return data['conversations']
    else:
        raise ValueError("Unknown conversation format. Expected list or {conversations: [...]}")


def load_custom_archetypes(filepath: str) -> Dict[str, Any]:
    """Load custom archetype definitions from YAML."""
    try:
        import yaml
        with open(filepath, 'r') as f:
            data = yaml.safe_load(f)
        return data.get('archetypes', {})
    except ImportError:
        print("Warning: PyYAML not installed. Using default archetypes only.")
        return {}
    except Exception as e:
        print(f"Warning: Could not load custom archetypes: {e}")
        return {}


def main():
    parser = argparse.ArgumentParser(
        description='Analyze ChatGPT conversations to discover cognitive archetypes',
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples:
  %(prog)s --input conversations.json --output profile.json
  %(prog)s --input conversations.json --archetypes 5
  %(prog)s --input conversations.json --archetypes-config my-archetypes.yaml
  %(prog)s --input conversations.json --sample 1000 --output profile.json
        """
    )
    
    parser.add_argument('--input', '-i', required=True,
                       help='Path to conversations.json from ChatGPT export')
    parser.add_argument('--output', '-o', default='cognitive-profile.json',
                       help='Output path for profile JSON (default: cognitive-profile.json)')
    parser.add_argument('--archetypes', '-n', type=int, default=3,
                       help='Number of archetypes to identify (default: 3)')
    parser.add_argument('--archetypes-config', '-c',
                       help='Path to custom archetypes YAML config')
    parser.add_argument('--sample', '-s', type=int,
                       help='Sample size for faster analysis (analyze N random conversations)')
    parser.add_argument('--format', '-f', choices=['json', 'prompt-snippet'], default='json',
                       help='Output format (default: json)')
    
    args = parser.parse_args()
    
    print("=" * 60)
    print("🤖🤝🧠 User Cognitive Profile Analyzer")
    print("=" * 60)
    
    # Load custom archetypes if provided
    custom_archetypes = {}
    if args.archetypes_config:
        custom_archetypes = load_custom_archetypes(args.archetypes_config)
        print(f"Loaded {len(custom_archetypes)} custom archetypes")
    
    # Load conversations
    print(f"\nLoading conversations from: {args.input}")
    conversations = load_conversations(args.input)
    print(f"Loaded {len(conversations)} conversations")
    
    # Analyze
    profiler = CognitiveProfiler(
        num_archetypes=args.archetypes,
        custom_archetypes=custom_archetypes
    )
    
    profile = profiler.generate_profile(conversations, args.sample)
    
    # Output
    print("\n" + "=" * 60)
    print("ANALYSIS COMPLETE")
    print("=" * 60)
    
    if args.format == 'json':
        with open(args.output, 'w', encoding='utf-8') as f:
            json.dump(profile, f, indent=2, ensure_ascii=False)
        print(f"Profile saved to: {args.output}")
        
        # Print summary
        print("\n📊 Profile Summary:")
        print(f"  Primary Archetype: {profile['insights']['primary_mode']}")
        print(f"  Confidence: {profile['insights']['primary_confidence']}")
        print(f"  Context Switching: {profile['insights']['context_switching']}")
        print(f"\n  Detected Archetypes:")
        for archetype in profile['archetypes']:
            print(f"    • {archetype['name']}: {archetype['metrics']['conversation_count']} conversations")
        
        print(f"\n💡 Communication Preferences:")
        for pref in profile['insights']['communication_preferences']:
            print(f"    • {pref}")
            
    else:  # prompt-snippet
        snippet = f"""## User Cognitive Profile
<!-- Generated by user-cognitive-profiles skill -->

- **Primary Archetype:** {profile['insights']['primary_mode']}
- **Confidence:** {profile['insights']['primary_confidence']}
- **Context Switching:** {profile['insights']['context_switching']}

### Communication Style
"""
        for archetype in profile['archetypes']:
            rec = archetype['recommendations']
            snippet += f"""
**{archetype['name']}** ({archetype['metrics']['conversation_count']} conversations)
- AI Role: {rec['ai_role']}
- Style: {rec['communication_style']}
- Keywords: {', '.join(archetype['keywords'][:5])}
"""
        
        snippet += "\n### Preferences\n"
        for pref in profile['insights']['communication_preferences']:
            snippet += f"- {pref}\n"
        
        with open(args.output, 'w', encoding='utf-8') as f:
            f.write(snippet)
        
        print(f"Prompt snippet saved to: {args.output}")
        print("\n" + snippet)
    
    print("\n" + "=" * 60)
    print("Next steps:")
    print("  1. Review the profile output")
    print("  2. Add relevant insights to your SOUL.md or AGENTS.md")
    print("  3. Configure your OpenClaw agent accordingly")
    print("=" * 60)


if __name__ == '__main__':
    main()
