# Methodology: How the Analysis Works

## Overview

The User Cognitive Profiler uses a multi-stage pipeline to analyze conversation patterns:

1. **Feature Extraction** → Quantify communication characteristics
2. **Clustering** → Group similar conversations
3. **Archetype Classification** → Label clusters with cognitive profiles
4. **Context Shift Detection** → Identify mode transitions
5. **Insight Generation** → Synthesize actionable recommendations

## 1. Feature Extraction

For each conversation, we extract:

### Message Metrics
- `avg_user_message_length` — Average words per user message
- `max_user_message_length` — Longest single message
- `total_user_words` — Cumulative word count
- `num_user_messages` — Number of user turns
- `num_turns` — Total conversation turns

### Content Analysis
- `question_ratio` — Percentage of messages containing "?"
- `code_block_count` — Number of code blocks (```...``` or `...`)
- `url_count` — Number of URLs shared
- `keywords` — Top 10 most frequent content words

### Metadata
- `timestamp` — Conversation creation time
- `title` — Conversation title (if available)
- `conversation_id` — Unique identifier

### Why These Features?

| Feature | Captures |
|---------|----------|
| Message length | Communication depth vs. efficiency preference |
| Question ratio | Collaborative vs. directive style |
| Code blocks | Technical vs. conceptual focus |
| Keywords | Domain interests and vocabulary patterns |

## 2. Clustering Algorithm

### With scikit-learn (Recommended)

We use **K-Means clustering** on normalized feature vectors:

```python
features = [
    avg_user_message_length,
    max_user_message_length,
    question_ratio,
    code_block_count,
    num_turns
]
```

**Process:**
1. StandardScaler normalizes features (mean=0, std=1)
2. K-Means partitions conversations into N clusters
3. Each cluster represents a distinct communication pattern

**Why K-Means?**
- Fast and scalable (handles 10k+ conversations)
- Interpretable centroids
- Well-suited for spherical clusters in feature space

### Fallback (Without sklearn)

If scikit-learn is not available, we use **k-means style bucketing**:
- Sort by primary feature (message length)
- Split into equal-sized buckets
- Less sophisticated but requires no dependencies

## 3. Archetype Classification

Each cluster is classified based on:

### Primary Metrics
- **Average message length** → Communication depth
  - < 50 words: Brief, efficiency-focused
  - 50-200 words: Balanced
  - > 200 words: Detailed, analytical

- **Question ratio** → Interaction style
  - < 0.2: Directive, statement-heavy
  - 0.2-0.4: Balanced
  - > 0.4: Collaborative, inquiry-heavy

### Keyword Extraction with BM25

The tool uses **BM25** (Best Match 25) for superior keyword extraction and archetype classification.

**Why BM25 over simple TF-IDF?**

| Feature | BM25 | Simple Frequency |
|---------|------|------------------|
| Term saturation | ✓ Limits infinite growth | ✗ Linear growth |
| Length normalization | ✓ Fair comparison | ✗ Long docs favored |
| IDF weighting | ✓ Rare terms boosted | ✗ Often missing |
| Proven effectiveness | ✓ Industry standard | ✗ Ad-hoc |

**BM25 Formula:**
```
score = IDF * (tf * (k1 + 1)) / (tf + k1 * (1 - b + b * (doc_len / avg_len)))
```

Where:
- `IDF` = Inverse document frequency (rare terms = higher score)
- `tf` = Term frequency (with saturation)
- `k1` = Saturation parameter (default: 1.5)
- `b` = Length normalization (default: 0.75)

**Implementation:**
- Built-in BM25 class (no external dependencies)
- Uses conversation corpus for context-aware scoring
- Falls back to frequency-based if BM25 unavailable

### Keyword Matching

After BM25 extraction, we match against archetype definitions:

| Archetype | Keywords |
|-----------|----------|
| Efficiency Optimizer | quick, brief, simple, just, fast |
| Systems Architect | architecture, design, system, framework |
| Philosophical Explorer | why, meaning, philosophy, question |
| Creative Synthesizer | like, similar, analogy, connection |

**BM25 improves matching by:**
- Weighting distinctive terms higher
- Handling polysemy (same word, different meanings)
- Accounting for document context

### Custom Archetypes

Users can define custom archetypes with:
- Name and description
- Keyword lists (used as BM25 queries)
- Pattern indicators
- Recommended AI role

## 4. Context Shift Detection

We identify moments when communication style changes significantly:

**Algorithm:**
1. Sort conversations by timestamp
2. Compare consecutive conversations
3. Flag when message length changes by >3x
4. Categorize shift type (brief→detailed or vice versa)

**Interpretation:**
- Many shifts = High context-switching user
- Few shifts = Consistent communication style
- Shift triggers = What causes mode changes

## 5. Confidence Scoring

Archetype confidence is calculated as:

```
confidence = min(0.95, 0.5 + (cluster_size / 1000))
```

**Rationale:**
- Larger clusters = more evidence = higher confidence
- Floor of 0.5 (some uncertainty always exists)
- Ceiling of 0.95 (never 100% certain)

## 6. Recommendation Generation

For each archetype, we generate:

### AI Role
How the agent should position itself:
- "Senior Architect" → Collaborative expert
- "Efficient Tool" → Fast execution
- "Socratic Partner" → Exploratory dialogue

### Communication Style
- Response length (short/medium/long/adaptive)
- Structure (bullet points/hierarchical/dialogue)
- Tone (direct/collaborative/inquisitive)

## Limitations

### What This Tool Captures
- ✅ Macro-level communication patterns
- ✅ Distinct conversation styles
- ✅ Context-switching tendencies

### What It Doesn't Capture
- ❌ Emotional state or sentiment
- ❌ Domain expertise depth
- ❌ Real-time adaptation (post-analysis)
- ❌ Non-text communication patterns

### Known Biases

**ChatGPT Export Limitations:**
- Only captures ChatGPT conversations (not Claude, etc.)
- No metadata about conversation outcome
- Titles may not reflect actual content

**Analysis Biases:**
- English-language optimized
- Code-heavy conversations weighted heavily
- Message length is primary differentiator
- BM25 works best with larger corpora (>100 conversations)

## Validation

### How to Validate Your Profile

1. **Review sample conversations** in each archetype
2. **Check if recommendations feel right** — do they match your preferences?
3. **Test with your agent** — apply insights and observe improvement
4. **Iterate** — adjust archetype definitions if needed

### Red Flags

- All archetypes have similar confidence → Try different cluster count
- Primary archetype doesn't match self-perception → Check custom keywords
- No context shifts detected → May need more conversation history

## Technical Notes

### Performance
- 1,000 conversations: ~5 seconds
- 10,000 conversations: ~30 seconds
- Memory: ~200MB for 10k conversations
- BM25 indexing adds ~1-2 seconds for large corpora

### Dependencies

**Required:**
- Python 3.8+
- Standard library only (json, re, pathlib, etc.)

**Recommended:**
- scikit-learn (better clustering)
- numpy (faster calculations, used by BM25)
- PyYAML (custom archetypes)

**Optional:**
- rank-bm25 (alternative BM25 implementation)
- pandas (for advanced analysis)
- matplotlib (for visualization)

### BM25 Implementation

The script includes a **built-in BM25 class** that works without external dependencies:
- Pure Python implementation
- Uses numpy for vectorization if available
- Falls back to standard math operations otherwise
- Configurable k1 and b parameters

## Future Enhancements

Potential improvements:
- **BM25 tuning**: Adaptive k1/b parameters per corpus
- **Sentiment analysis integration**: Emotional tone detection
- **Topic modeling (LDA)**: For domain detection
- **Time-series analysis**: Evolution tracking
- **Cross-platform support**: Claude, Gemini exports
- **Semantic similarity**: Use embeddings instead of keyword matching

## References

- **BM25**: Robertson, S., & Zaragoza, H. (2009). The Probabilistic Relevance Framework: BM25 and Beyond
- **K-Means Clustering**: [scikit-learn documentation](https://scikit-learn.org/stable/modules/clustering.html#k-means)
- **TF-IDF**: [Wikipedia](https://en.wikipedia.org/wiki/Tf-idf)
- **Communication Styles**: [Harvard Business Review](https://hbr.org/)
