---
name: user-cognitive-profiles
description: "Analyze ChatGPT conversation exports to discover cognitive archetypes and optimize AI-human communication patterns. Enables personalized agent interactions based on detected user profiles."
homepage: https://github.com/openclaw/user-cognitive-profiles
metadata:
  {
    "openclaw":
      {
        "emoji": "🤖🤝🧠",
        "requires": { "bins": ["python3"] },
        "tags": ["communication", "persona", "user-research", "optimization", "nlp"],
      },
  }
---

# User Cognitive Profiles

🤖🤝🧠 Discover how you communicate with AI and optimize your agent interactions.

This skill analyzes your ChatGPT conversation history to identify **cognitive archetypes** — recurring patterns in how you think, communicate, and collaborate. Use these insights to calibrate your OpenClaw agent for more effective, personalized interactions.

---

## Why This Matters

Human-AI communication is not one-size-fits-all. Just as you adapt your communication style between contexts (work meeting vs. casual chat), effective AI assistance requires matching your **cognitive architecture**.

**The Problem:**
- Default AI behavior assumes a generic user
- Your communication style varies dramatically by context (professional vs. personal)
- Misaligned AI responses feel inefficient or frustrating

**The Solution:**
- Analyze your actual conversation patterns
- Identify your dominant cognitive archetypes
- Configure your agent to match your communication style

---

## Quick Start

### 1. Export Your ChatGPT Data

1. Go to **ChatGPT → Settings → Data Controls → Export Data**
2. Click "Export" and confirm
3. Wait for the email (usually arrives within 24 hours)
4. Download the ZIP file from the email link
5. Extract it — you'll find `conversations.json`

### 2. Run the Analysis

```bash
cd /path/to/user-cognitive-profiles
python3 scripts/analyze_profile.py \
  --input ~/Downloads/chatgpt-export/conversations.json \
  --output ~/.openclaw/my-cognitive-profile.json \
  --archetypes 3
```

### 3. Apply to Your Agent

Add to your `SOUL.md` or `AGENTS.md`:

```markdown
## User Cognitive Profile
<!-- Source: generated by user-cognitive-profiles skill -->
- **Primary Archetype:** Efficiency Optimizer
- **Avg Message Length:** 47 words
- **Context Switching:** High (professional vs. personal modes)
- **Key Patterns:** Prefers direct answers, values examples over theory

### Communication Calibration
- Default to concise responses
- Provide examples + theory + hands-on steps
- Watch for professional/personal mode shifts
```

---

## Cognitive Archetypes

The analysis identifies archetypes based on **four dimensions**:

| Dimension | Low | High |
|-----------|-----|------|
| **Message Length** | Brief commands | Extended analysis |
| **Structure** | Organic flow | Systematic breakdown |
| **Depth** | Practical focus | Theoretical exploration |
| **Tone** | Transactional | Collaborative |

### Common Archetypes

#### 🔧 Efficiency Optimizer
- **Messages:** Short, direct, action-oriented
- **Wants:** Quick answers, minimal explanation
- **AI Role:** Tool to get things done
- **Example:** "Set up email. Use pass. Go."

#### 🏗️ Systems Architect
- **Messages:** Long, structured, comprehensive
- **Wants:** Deep analysis, trade-offs, strategic thinking
- **AI Role:** Collaborative partner for complex problems
- **Example:** 300-word technical breakdown with multiple considerations

#### 🧭 Philosophical Explorer
- **Messages:** Varies widely, questions assumptions
- **Wants:** Meaning, patterns, cross-domain connections
- **AI Role:** Socratic partner for insight generation
- **Example:** "How does this relate to [completely different domain]?"

#### 🎨 Creative Synthesizer
- **Messages:** Connects disparate ideas, uses analogies
- **Wants:** Novel combinations, pattern recognition
- **AI Role:** Ideation partner and pattern mirror
- **Example:** "This is like jazz improvisation..."

---

## Customization

### Define Your Own Archetypes

Create `~/.openclaw/my-archetypes.yaml`:

```yaml
archetypes:
  - name: "Research Mode"
    keywords:
      - "research"
      - "analyze"
      - "compare"
      - "trade-off"
    patterns:
      - long_messages
      - multiple_questions
      - citation_requests
    
  - name: "Quick Mode"
    keywords:
      - "quick"
      - "brief"
      - "simple"
      - "just"
    patterns:
      - short_messages
      - imperative_tone
      - minimal_context
```

Run with custom archetypes:

```bash
python3 scripts/analyze_profile.py \
  --input conversations.json \
  --archetypes-config ~/.openclaw/my-archetypes.yaml
```

### Adjust Cluster Count

More archetypes = finer granularity, but harder to act on:

```bash
# Simple: 2-3 archetypes
python3 scripts/analyze_profile.py --archetypes 2

# Detailed: 5-7 archetypes
python3 scripts/analyze_profile.py --archetypes 5

# Complex: 10+ (for power users)
python3 scripts/analyze_profile.py --archetypes 10
```

---

## Understanding the Output

### Profile JSON Structure

```json
{
  "metadata": {
    "total_conversations": 3784,
    "date_range": "2024-01-01 to 2025-01-31",
    "analysis_date": "2026-02-02"
  },
  "archetypes": [
    {
      "id": 0,
      "name": "Systems Architect",
      "confidence": 0.87,
      "metrics": {
        "avg_message_length": 382,
        "avg_response_length": 450,
        "question_ratio": 0.23,
        "code_block_ratio": 0.45
      },
      "keywords": ["architecture", "design", "trade-off", "system"],
      "sample_conversations": ["uuid-1", "uuid-2"],
      "recommendations": {
        "ai_role": "Senior Architect",
        "communication_style": "Detailed, systematic, collaborative",
        "response_length": "long",
        "structure": "hierarchical"
      }
    }
  ],
  "context_shifts": [
    {
      "trigger": "technical_keywords",
      "from_archetype": "Efficiency Optimizer",
      "to_archetype": "Systems Architect"
    }
  ],
  "insights": {
    "primary_mode": "Systems Architect",
    "context_switching": "high",
    "communication_preferences": [
      "Examples before theory",
      "Hands-on application",
      "Cross-domain analogies"
    ]
  }
}
```

### Key Metrics Explained

| Metric | Description | Why It Matters |
|--------|-------------|----------------|
| `avg_message_length` | Average words per user message | Short = efficiency mode, Long = exploration mode |
| `question_ratio` | % of turns that are questions | High = collaborative, Low = directive |
| `code_block_ratio` | % of messages with code | Technical vs. conceptual focus |
| `context_shifts` | Detected mode transitions | Indicates multiple archetypes at play |
| `confidence` | Cluster cohesion score | Higher = more distinct pattern |

---

## Privacy & Security

**All processing is local.** The script:
- ✅ Runs entirely on your machine
- ✅ Never uploads data to external services
- ✅ Stores results in your local OpenClaw workspace
- ✅ You control what gets shared (if anything)

**Recommended workflow:**
1. Export ChatGPT data
2. Run analysis locally
3. Review `my-cognitive-profile.json`
4. Manually add relevant insights to `SOUL.md`
5. (Optional) Delete the export and raw profile

---

## Advanced Usage

### Compare Profiles Over Time

Track how your communication evolves:

```bash
# January analysis
python3 scripts/analyze_profile.py \
  --input conversations_jan.json \
  --output profile_jan.json

# June analysis
python3 scripts/analyze_profile.py \
  --input conversations_jun.json \
  --output profile_jun.json

# Compare
python3 scripts/compare_profiles.py profile_jan.json profile_jun.json
```

### Export for Other Agents

Generate a prompt snippet for Claude, GPT, or other agents:

```bash
python3 scripts/analyze_profile.py \
  --input conversations.json \
  --format prompt-snippet \
  --output agent-prompt.txt
```

Output:
```markdown
## User Communication Profile
- Primary style: Systems Architect (detailed, analytical)
- Secondary style: Efficiency Optimizer (brief, pragmatic)
- Context switching: High (watch for mode shifts)
- Preferences: Examples + theory + hands-on steps
- Treat as: Senior technical partner, not assistant
```

---

## Troubleshooting

### "conversations.json not found"

The export ZIP contains multiple files. Make sure you're pointing to:
```
chatgpt-export/
├── conversations.json  <-- This one
├── user.json
└── ...
```

### "No conversations detected"

Your export might be empty or corrupted. Check:
```bash
head -20 conversations.json
```

Should show: `[{"title": "...", "messages": [...]}, ...]`

### "All archetypes have similar confidence"

Try adjusting the cluster count:
```bash
# Too granular
python3 scripts/analyze_profile.py --archetypes 10

# Try simpler
python3 scripts/analyze_profile.py --archetypes 3
```

### "Analysis takes too long"

For large conversation histories (10k+ messages):
```bash
# Sample for faster analysis
python3 scripts/analyze_profile.py \
  --input conversations.json \
  --sample 1000  # Analyze random 1000 conversations
```

---

## Integration with OpenClaw

### Automatic Profile Loading

Add to your OpenClaw workspace `AGENTS.md`:

```markdown
## On Session Start
1. Read `~/.openclaw/my-cognitive-profile.json` if exists
2. Adapt communication style to primary archetype
3. Watch for context shift indicators
```

### Dynamic Mode Detection

For agents that can switch modes mid-conversation:

```python
# Pseudocode for agent integration
def detect_mode_shift(current_message, profile):
    for shift in profile["context_shifts"]:
        if shift["trigger"] in current_message:
            return shift["to_archetype"]
    return profile["insights"]["primary_mode"]
```

---

## Contributing

Have a new archetype that works well? Submit a PR with:
1. Archetype definition in `examples/`
2. Sample data (anonymized)
3. Validation that it clusters distinctly

---

## References

- `references/methodology.md` — Technical details on clustering algorithm
- `references/archetype-taxonomy.md` — Full archetype definitions
- `examples/` — Sample profiles and configurations

---

*Built for humans who want their AI to truly understand them.* 🤖🤝🧠
