import urllib.parse
import urllib.request
import json
import sys
import argparse

def search(query, category="general", count=10, time_range=None, engines=None):
    base_url = "http://localhost:8080/search"
    params = {
        "q": query,
        "format": "json",
        "categories": category,
        "language": "en-US",
        "safesearch": 1
    }
    
    if time_range:
        params["time_range"] = time_range
    if engines:
        params["engines"] = engines
    
    url = f"{base_url}?{urllib.parse.urlencode(params)}"
    
    try:
        req = urllib.request.Request(url)
        with urllib.request.urlopen(req) as response:
            data = json.loads(response.read().decode())
            
        results = data.get("results", [])[:count]
        
        output = []
        if not results:
            print(f"# No results found for: '{query}'")
            return

        for r in results:
            title = r.get('title', 'No Title')
            url = r.get('url', 'N/A')
            source = r.get('engine', 'unknown')
            content = r.get('content', '').strip()
            
            # Format as Agent-Friendly Markdown
            output.append(f"### {title}")
            output.append(f"**Source:** {source} | **URL:** {url}")
            output.append(f"> {content}")
            output.append("") # Spacer
            
        print(f"# Search Results for: '{query}' ({len(results)} hits)\n")
        if time_range:
            print(f"*Filter: Past {time_range}*\n")
        print("\n".join(output))
        print(f"\n_Generated by Upsurge Private Intelligence Radar v1.1_")
        
    except Exception as e:
        print(f"Error searching SearXNG: {e}")
        sys.exit(1)

if __name__ == "__main__":
    parser = argparse.ArgumentParser()
    parser.add_argument("query", help="Search query")
    parser.add_argument("--category", default="general", help="Search category (general, news, it, science)")
    parser.add_argument("--count", type=int, default=5, help="Number of results")
    parser.add_argument("--time", choices=['day', 'month', 'year'], help="Time range limit")
    parser.add_argument("--engines", help="Comma-separated engine list (e.g. google,bing)")
    args = parser.parse_args()
    
    search(args.query, args.category, args.count, args.time, args.engines)
