---
name: Unity
description: Avoid common Unity mistakes — lifecycle ordering, GetComponent caching, physics timing, and Unity's fake null.
metadata: {"clawdbot":{"emoji":"🎮","os":["linux","darwin","win32"]}}
---

## Lifecycle Order
- `Awake` before `Start` — use Awake for self-init, Start for cross-references
- `OnEnable` called before `Start` — but after `Awake`
- Order between scripts not guaranteed — use Script Execution Order if needed
- `Awake` called even if disabled — `Start` only when enabled

## GetComponent Performance
- `GetComponent` every frame is slow — cache in `Awake` or `Start`
- `GetComponentInChildren` searches recursively — expensive on deep hierarchies
- `TryGetComponent` returns bool — avoids null check, slightly faster
- Use `RequireComponent` attribute — ensures dependency, documents requirement

## Physics Timing
- Physics in `FixedUpdate`, not `Update` — consistent regardless of framerate
- `FixedUpdate` can run 0 or multiple times per frame — don't assume 1:1
- `Rigidbody.MovePosition` in FixedUpdate — `transform.position` bypasses physics
- `Time.deltaTime` in Update, `Time.fixedDeltaTime` in FixedUpdate — or just use deltaTime

## Unity's Fake Null
- Destroyed objects aren't truly null — `== null` returns true, but object exists
- Null-conditional `?.` doesn't work properly — use `== null` or `bool` conversion
- `Destroy` doesn't happen immediately — object gone next frame
- Use `DestroyImmediate` only in editor — causes issues in builds

## Coroutines
- `StartCoroutine` needs MonoBehaviour active — disabled/destroyed stops coroutines
- `yield return null` waits one frame — `yield return new WaitForSeconds(1)` for time
- `StopCoroutine` needs same method or Coroutine reference — string overload unreliable
- Can't return values — use callbacks or set field in coroutine

## Instantiate and Pooling
- `Instantiate` is expensive — pool frequently created/destroyed objects
- `Instantiate(prefab, parent)` sets parent — avoids extra SetParent call
- `SetActive(false)` before returning to pool — not Destroy
- Pool inactive objects under a parent — keeps hierarchy clean

## Serialization
- `[SerializeField]` for private fields in inspector — prefer over public
- `public` fields auto-serialize — but exposes API you may not want
- `[HideInInspector]` hides but still serializes — `[NonSerialized]` to skip entirely
- Serialized fields keep inspector values — code defaults ignored after first serialize

## ScriptableObjects
- Data containers that live as assets — share between scenes/prefabs
- `CreateAssetMenu` attribute for easy creation — right-click → Create
- Don't modify at runtime in builds — changes not saved (except in editor)
- Great for config, item databases — reduces prefab duplication

## Common Mistakes
- `Find` methods every frame — cache references
- String comparisons for tags — use `CompareTag("Enemy")`, not `tag == "Enemy"`
- Physics queries allocate — use `NonAlloc` variants: `RaycastNonAlloc`
- UI anchors wrong — stretches unexpectedly on different resolutions
- `async/await` without context — use UniTask or careful error handling
