#!/usr/bin/env node
/**
 * Membase CLI Router
 *
 * This is the main entry point for all Membase operations.
 * Can be called in multiple ways:
 * 1. Direct: node membase.ts backup
 * 2. Agent: via SKILL.md instructions
 * 3. Extension: via OpenClaw CLI wrapper
 */

async function main() {
  const args = process.argv.slice(2);
  const command = args[0];

  if (!command) {
    console.error('Usage: node membase.ts <command> [options]');
    console.log('\nAvailable commands:');
    console.log('  backup     Backup memories to Membase');
    console.log('  restore    Restore memories from backup');
    console.log('  list       List all backups');
    console.log('  diff       Compare two backups');
    console.log('  status     Show backup status');
    console.log('  cleanup    Clean up old backups');
    process.exit(1);
  }

  try {
    switch (command) {
      case 'backup': {
        const { run } = await import('./commands/backup.js');
        await run(args.slice(1));
        break;
      }

      case 'restore': {
        const { run } = await import('./commands/restore.js');
        await run(args.slice(1));
        break;
      }

      case 'list': {
        const { run } = await import('./commands/list.js');
        await run(args.slice(1));
        break;
      }

      case 'diff': {
        const { run } = await import('./commands/diff.js');
        await run(args.slice(1));
        break;
      }

      case 'status': {
        const { run } = await import('./commands/status.js');
        await run(args.slice(1));
        break;
      }

      case 'cleanup': {
        const { run } = await import('./commands/cleanup.js');
        await run(args.slice(1));
        break;
      }

      default:
        console.error(`[ERROR] Unknown command: ${command}`);
        console.log('\nAvailable commands: backup, restore, list, diff, status, cleanup');
        process.exit(1);
    }
  } catch (error) {
    console.error(`[ERROR] Error: ${error instanceof Error ? error.message : 'Unknown error'}`);
    process.exit(1);
  }
}

main();
